/*******************************************************************************
 * Copyright (c) 2014, 2025 DiffusionData Ltd., All Rights Reserved.
 *
 * Use is subject to licence terms.
 *
 * NOTICE: All information contained herein is, and remains the
 * property of DiffusionData. The intellectual and technical
 * concepts contained herein are proprietary to DiffusionData and
 * may be covered by U.S. and Foreign Patents, patents in process, and
 * are protected by trade secret or copyright law.
 *******************************************************************************/
package com.pushtechnology.diffusion.client.callbacks;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Callback interface that provides responses from the server. Extensions of
 * this interface have callback methods that provide results for successful
 * calls.
 * <p>
 * The {@link Stream} extension supports multiple responses for each call.
 * <p>
 * If the server returns a response for an unknown call context (typically a
 * late response for a call context has timed out), an error will be reported to
 * the {@link com.pushtechnology.diffusion.client.session.Session.ErrorHandler
 * session error handler}.
 *
 * @author DiffusionData Limited
 * @since 5.1
 * @see Stream
 */
public interface Callback {

    /**
     * Notification of a contextual error related to this callback. This is
     * analogous to an exception being raised. Situations in which
     * {@code onError} is called include the session being closed, a
     * communication timeout, or a problem with the provided parameters. No
     * further calls will be made to this callback for the call context.
     *
     * @param errorReason a value representing the error; this can be one of
     *        constants defined in {@link ErrorReason}, or a feature-specific
     *        reason
     */
    void onError(ErrorReason errorReason);

    /**
     * Abstract default callback.
     * <P>
     * This simply logs onError calls at debug level. This should be overridden
     * to perform more specific error handling.
     */
    abstract class Default implements Callback {

        private static final Logger LOG =
            LoggerFactory.getLogger(Default.class);

        @Override
        public void onError(ErrorReason errorReason) {
            LOG.warn("{} error, reason={}", this, errorReason);
        }
    }
}
