/*******************************************************************************
 * Copyright (c) 2014, 2024 DiffusionData Ltd., All Rights Reserved.
 *
 * Use is subject to licence terms.
 *
 * NOTICE: All information contained herein is, and remains the
 * property of DiffusionData. The intellectual and technical
 * concepts contained herein are proprietary to DiffusionData and
 * may be covered by U.S. and Foreign Patents, patents in process, and
 * are protected by trade secret or copyright law.
 *******************************************************************************/
package com.pushtechnology.diffusion.client.callbacks;

/**
 * Reason codes used to report error conditions.
 * <p>
 * An application can check whether an ErrorReason is equal to some expected
 * value using {@link Object#equals(Object)}. Equality is derived from the
 * {@link #getReasonCode() reasonCode}, with instances with equal codes
 * considered equal.
 *
 * <p>
 * Some common ErrorReason values are defined as constants in this class. More
 * specific reasons may be defined by individual
 * {@link com.pushtechnology.diffusion.client.session.Feature features}.
 *
 * @author DiffusionData Limited
 * @since 5.1
 * @see Callback#onError
 */
public final class ErrorReason {

    // Common reasons.

    /**
     * Communication with the server failed.
     */
    public static final ErrorReason COMMUNICATION_FAILURE =
        new ErrorReason(100, "Communication with server failed");

    /**
     * Communication with the server failed because the session is closed.
     */
    public static final ErrorReason SESSION_CLOSED =
        new ErrorReason(101, "Session is closed");

    /**
     * Communication with the server failed because a service request timed out.
     */
    public static final ErrorReason REQUEST_TIME_OUT =
        new ErrorReason(102, "Request time out");

    /**
     * The request was rejected because the caller has insufficient permissions.
     */
    public static final ErrorReason ACCESS_DENIED =
        new ErrorReason(103, "Access denied");

    /**
     * The request was rejected because the requested service is unsupported for this caller.
     * @since 5.9
     */
    public static final ErrorReason UNSUPPORTED =
        new ErrorReason(104, "Unsupported service");

    /**
     * An application callback threw an exception. Check logs for more information.
     * @since 5.9
     */
    public static final ErrorReason CALLBACK_EXCEPTION =
        new ErrorReason(105, "An application callback threw an exception. Check logs for more information.");

    /**
     * An operation failed because invalid data was received.
     * @since 6.0
     */
    public static final ErrorReason INVALID_DATA =
        new ErrorReason(106, "Invalid data.");

    /**
     * The session does not exist on the server.
     * @since 6.0
     */
    public static final ErrorReason NO_SUCH_SESSION =
        new ErrorReason(107, "No session was found");

    /**
     * A datatype operation has failed due to incompatibility.
     * @since 6.0
     */
    public static final ErrorReason INCOMPATIBLE_DATATYPE =
        new ErrorReason(108, "Datatype is incompatible");

    /**
     * A message was not handled by the server.
     * @since 6.0
     */
    public static final ErrorReason UNHANDLED_MESSAGE =
        new ErrorReason(109, "A message was not handled");

    /**
     * A cluster operation failed because partition ownership changed during
     * processing.
     * <p>
     * This is a transient error that occurs while the cluster is recovering
     * from failure. The session can retry the operation.
     *
     * @since 6.0
     * @see #CLUSTER_ROUTING
     */
    public static final ErrorReason CLUSTER_REPARTITION =
        new ErrorReason(110, "The cluster was repartitioning and the request could not be routed.");

    /**
     * A topic update could not be performed because the topic is managed by a
     * component (for example, fan-out) which prohibits external updates.
     *
     * @since 6.0
     */
    public static final ErrorReason INCOMPATIBLE_STATE =
        new ErrorReason(
            111,
            "Topic operation not performed because it is managed by a component that prohibits external updates");

    /**
     * A cluster operation failed to be routed to a server within the cluster
     * due to a communication failure, or the server that owns a partition is
     * not currently known.
     *
     * <p>
     * This is a transient error that occurs while the cluster is recovering
     * from failure. The session can retry the operation.
     *
     * @since 6.5
     * @see #CLUSTER_REPARTITION
     */
    public static final ErrorReason CLUSTER_ROUTING =
        new ErrorReason(112, "The cluster operation could not be routed.");

    /**
     * A conflicting registration exists on the same branch of the topic tree.
     */
    public static final ErrorReason TOPIC_TREE_REGISTRATION_CONFLICT =
        new ErrorReason(
            200,
            "A conflicting registration exists on the same branch of the topic tree");

    /**
     * A conflicting registration exists.
     */
    public static final ErrorReason HANDLER_CONFLICT  =
        new ErrorReason(201, "Conflict with existing handler");

    /**
     * An invalid path was supplied.
     */
    public static final ErrorReason INVALID_PATH =
        new ErrorReason(202, "Invalid path");

    private final int reasonCode;
    private final String description;

    /**
     * Constructor.
     *
     * <p>
     * Application code should prefer pre-defined constants such as
     * {@link ErrorReason#COMMUNICATION_FAILURE}, rather than calling this.
     *
     * @param reasonCode a unique code
     * @param description a description of the reason, may be context-specific
     */
    public ErrorReason(int reasonCode, String description) {
        this.reasonCode = reasonCode;
        this.description = description;
    }

    /**
     * A code that uniquely identifies this reason.
     *
     * <p>
     * Applications should rarely need to call this method and prefer
     * {@link #equals(Object)} to compare instances for equality.
     *
     * <p>
     * Reasons defined by DiffusionData have reason codes in the range 0 to
     * 65535 inclusive.
     *
     * @return the code
     */
    public int getReasonCode() {
        return reasonCode;
    }

    /**
     * Description of the error reason, may be context-specific.
     *
     * @return the description
     */
    public String getDescription() {
        return description;
    }

    @Override
    public int hashCode() {
        return getReasonCode();
    }

    @Override
    public boolean equals(Object o) {
        if (o == this) {
            return true;
        }

        if (o == null || o.getClass() != getClass()) {
            return false;
        }

        final ErrorReason other = (ErrorReason) o;

        return other.getReasonCode() == getReasonCode();
    }

    @Override
    public String toString() {
        return reasonCode + ": " + description;
    }
}
