/*******************************************************************************
 * Copyright (c) 2014, 2025 DiffusionData Ltd., All Rights Reserved.
 *
 * Use is subject to licence terms.
 *
 * NOTICE: All information contained herein is, and remains the
 * property of DiffusionData. The intellectual and technical
 * concepts contained herein are proprietary to DiffusionData and
 * may be covered by U.S. and Foreign Patents, patents in process, and
 * are protected by trade secret or copyright law.
 *******************************************************************************/
package com.pushtechnology.diffusion.client.features;

import java.util.Collections;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;

import com.pushtechnology.diffusion.client.session.Feature;
import com.pushtechnology.diffusion.client.session.Session;
import com.pushtechnology.diffusion.client.session.SessionClosedException;
import com.pushtechnology.diffusion.client.types.Credentials;
import com.pushtechnology.diffusion.client.types.GlobalPermission;
import com.pushtechnology.diffusion.client.types.PathPermission;

/**
 * This feature provides a client session with the ability to re-authenticate
 * the session, as well as to query permissions assigned to it.
 *
 * <H3>Access control</H3> There are no permissions requirements associated with
 * this feature. The client provides credentials that authenticate the
 * principal.
 *
 * <H3>Accessing the feature</H3> This feature may be obtained from a
 * {@link Session session} as follows:
 *
 * <pre>
 * <code>
 * Security security = session.feature(Security.class);
 * </code>
 * </pre>
 *
 * @author DiffusionData Limited
 * @since 5.0
 */
public interface Security extends Feature {

    /**
     * Re-authenticate the session.
     * <p>
     * This may be used to change the principal for the session, or to
     * re-authenticate the session before it expires.
     * <p>
     * A session may determine when it is due to expire by querying the value of
     * the {@link Session#EXPIRY_TIME EXPIRY_TIME} session property using
     * {@link Session#getSessionProperties}. If this property is not present
     * the session will not expire and there is no need to re-authenticate
     * unless the principal in use is to be changed.
     *
     * @param principal the principal name. This may be the same principal as
     *        supplied when the session was originally opened, or it may be an
     *        entirely different principal.
     *
     * @param credentials the credentials authenticating the principal
     *
     * @param properties a map of the proposed user session properties. The
     *        supplied properties will be validated during authentication and
     *        may be discarded or changed. If no user properties are required,
     *        an {@link Collections#emptyMap empty map} should be supplied.
     *
     * @return a CompletableFuture that completes when a response is received
     *         from the server.
     *
     *         <p>
     *         If authentication succeeded, the CompletableFuture will complete
     *         successfully with a {@link Boolean#TRUE TRUE} result.
     *
     *         <p>
     *         If authentication failed because the {@code principal} was
     *         unknown or the {@code credentials} were invalid, the session will
     *         not have been re-authenticated and the CompletableFuture will
     *         complete successfully with a {@link Boolean#FALSE FALSE} result.
     *
     *         <p>
     *         Otherwise, the CompletableFuture will complete exceptionally with
     *         a {@link CompletionException}. Common reasons for failure, listed
     *         by the exception reported as the
     *         {@link CompletionException#getCause() cause}, include:
     *
     *         <ul>
     *         <li>{@link SessionClosedException} &ndash; if the session is
     *         closed.
     *         </ul>
     *
     * @since 6.12
     */
    CompletableFuture<Boolean> reauthenticate(
        String principal,
        Credentials credentials,
        Map<String, String> properties);

    /**
     * Change the security principal associated with the session.
     *
     * @param principal the new principal name
     *
     * @param credentials the credentials authenticating the new principal
     *
     * @return a CompletableFuture that completes when a response is received
     *         from the server.
     *
     *         <p>
     *         If authentication succeeded and the session principal has been
     *         changed, the CompletableFuture will complete successfully with a
     *         {@link Boolean#TRUE TRUE} result.
     *
     *         <p>
     *         If authentication failed because the {@code principal} was
     *         unknown or the {@code credentials} were invalid, the session's
     *         principal has not changed and the CompletableFuture will complete
     *         successfully with a {@link Boolean#FALSE FALSE} result.
     *
     *         <p>
     *         Otherwise, the CompletableFuture will complete exceptionally with
     *         a {@link CompletionException}. Common reasons for failure, listed
     *         by the exception reported as the
     *         {@link CompletionException#getCause() cause}, include:
     *
     *         <ul>
     *         <li>{@link SessionClosedException} &ndash; if the session is
     *         closed.
     *         </ul>
     *
     * @since 6.0
     *
     * @deprecated since 6.12
     *             <p>
     *             Use the new {@link #reauthenticate} in preference, which
     *             allows user proposed session properties to be supplied. This
     *             method will be removed in a future release.
     */
    @Deprecated
    default CompletableFuture<Boolean> changePrincipal(
        String principal,
        Credentials credentials) {
        return reauthenticate(principal, credentials, Collections.emptyMap());
    }

    /**
     * Query the global permissions assigned to the calling session.
     *
     * @return a CompletableFuture which completes when the response is received
     *         from the server.
     *
     *         <p>
     *         If the request was successful, the CompletableFuture will
     *         complete successfully with the session's global permissions.
     *
     * @since 6.3
     */
    CompletableFuture<Set<GlobalPermission>> getGlobalPermissions();

    /**
     * Query the path permissions assigned to the calling session on a given
     * path.
     *
     * @param path the path to query for permissions
     * @return a CompletableFuture which completes when the response is received
     *         from the server.
     *
     *         <p>
     *         If the request was successful, the CompletableFuture will
     *         complete successfully with the session's permissions for
     *         {@code path}.
     *
     * @since 6.5
     */
    CompletableFuture<Set<PathPermission>> getPermissionsForPath(String path);
}
