/*******************************************************************************
 * Copyright (c) 2014, 2025 DiffusionData Ltd., All Rights Reserved.
 *
 * Use is subject to licence terms.
 *
 * NOTICE: All information contained herein is, and remains the
 * property of DiffusionData. The intellectual and technical
 * concepts contained herein are proprietary to DiffusionData and
 * may be covered by U.S. and Foreign Patents, patents in process, and
 * are protected by trade secret or copyright law.
 *******************************************************************************/
package com.pushtechnology.diffusion.client.features.control.clients;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;

import com.pushtechnology.diffusion.client.callbacks.Registration;
import com.pushtechnology.diffusion.client.callbacks.Stream;
import com.pushtechnology.diffusion.client.features.NoSuchSessionException;
import com.pushtechnology.diffusion.client.security.authentication.Authenticator;
import com.pushtechnology.diffusion.client.session.Feature;
import com.pushtechnology.diffusion.client.session.PermissionsException;
import com.pushtechnology.diffusion.client.session.Session;
import com.pushtechnology.diffusion.client.session.SessionClosedException;
import com.pushtechnology.diffusion.client.session.SessionException;
import com.pushtechnology.diffusion.client.session.SessionId;
import com.pushtechnology.diffusion.client.types.GlobalPermission;

/**
 * This feature allows a client session to authenticate the credentials of other
 * sessions, and optionally revoke such authentications.
 * <P>
 * Sessions are authenticated by a chain of authenticators. A client session can
 * participate in the authentication process by creating an authenticator and
 * registering it with the server.
 * <P>
 * Each authenticator is registered under a particular authenticator name. For
 * registration to succeed, the server's security configuration must include a
 * matching {@code control-authentication-handler} entry for the name, otherwise
 * registration will fail and the authenticator will be closed immediately.
 * <P>
 * Each client session can register a single authenticator for a given
 * authenticator name.
 * <P>
 * For each authentication event, the server will use its configuration to
 * determine the authenticator priority order. The server may call
 * authenticators in serial or parallel. The server may stop the authentication
 * process as soon as it has an allow or deny response from an authenticator and
 * all higher priority authenticators have abstained.
 * <P>
 * For a configured control authenticator, the server will select a single
 * authenticator from those registered for the authenticator name. If no
 * authenticators are currently registered, the server will consult the next
 * configured authenticator in priority order.
 * <H3>Access control</H3>
 * <P>
 * In order to register an authenticator a session needs both
 * {@link GlobalPermission#REGISTER_HANDLER REGISTER_HANDLER} and
 * {@link GlobalPermission#AUTHENTICATE AUTHENTICATE} permissions.
 * <P>
 * In order to revoke a session's authentication a session needs both
 * {@link GlobalPermission#MODIFY_SESSION MODIFY_SESSION} and
 * {@link GlobalPermission#AUTHENTICATE AUTHENTICATE} permissions.
 *
 * <H3>Accessing the feature</H3> This feature may be obtained from a
 * {@link Session session} as follows:
 *
 * <pre>
 * <code>
 * AuthenticationControl authenticationControl = session.feature(AuthenticationControl.class);
 * </code>
 * </pre>
 *
 * @author DiffusionData Limited
 * @since 5.0
 */
public interface AuthenticationControl extends Feature {

    /**
     * Register an authenticator for client authentication events.
     *
     * @param name the authenticator name which must match an entry in the
     *        server's security configuration
     *
     * @param authenticator specifies the authenticator
     *
     * @return a CompletableFuture that completes when the authenticator has
     *         been registered, returning a {@link Registration} which can be
     *         used to unregister the authenticator.
     *         <p>
     *         Otherwise, the CompletableFuture will complete exceptionally with
     *         a {@link CompletionException}. Common reasons for failure, listed
     *         by the exception reported as the
     *         {@link CompletionException#getCause() cause}, include:
     *
     *         <ul>
     *         <li>{@link SessionClosedException} &ndash; if the session is
     *         closed;
     *
     *         <li>{@link PermissionsException} &ndash; if the session does not
     *         have {@code REGISTER_HANDLER} and {@code AUTHENTICATE}
     *         permission;
     *
     *         <li>{@link SessionException} &ndash; will occur if the server
     *         configuration does not contain a
     *         {@code control-authentication-handler} element with the given
     *         name.
     *         </ul>
     *
     * @since 6.2
     */
    CompletableFuture<Registration> setAuthenticationHandler(
        String name,
        ControlAuthenticator authenticator);

    /**
     * Revokes a session's authentication.
     * <p>
     * This will immediately close the specified client session.
     *
     * @param sessionId identifies the client session to revoke
     *
     * @return a CompletableFuture that completes when a response is received
     *         from the server.
     *
     *         <p>
     *         If the identified session was revoked, the CompletableFuture will
     *         complete successfully. The result type is any rather than Void to
     *         provide forward compatibility with future iterations of this API
     *         that may provide a non-null result with a more specific result
     *         type.
     *
     *         <p>
     *         Otherwise, the CompletableFuture will complete exceptionally with
     *         a {@link CompletionException}. Common reasons for failure, listed
     *         by the exception reported as the
     *         {@link CompletionException#getCause() cause}, include:
     *
     *         <ul>
     *         <li>{@link NoSuchSessionException} &ndash; if the identified
     *         session did not exist or was closed before the response was
     *         delivered;
     *
     *         <li>{@link PermissionsException} &ndash; if the calling session
     *         does not have {@code AUTHENTICATE} and {@code MODIFY_SESSION}
     *         permissions;
     *
     *         <li>{@link SessionClosedException} &ndash; if the calling session
     *         is closed.
     *         </ul>
     *
     * @since 6.12
     */
    CompletableFuture<?> revokeAuthentication(SessionId sessionId);

    /**
     * A control authenticator.
     *
     * @since 6.2
     */
    interface ControlAuthenticator extends Authenticator, Stream {
    }

}
