/*******************************************************************************
 * Copyright (c) 2014, 2025 DiffusionData Ltd., All Rights Reserved.
 *
 * Use is subject to licence terms.
 *
 * NOTICE: All information contained herein is, and remains the
 * property of DiffusionData. The intellectual and technical
 * concepts contained herein are proprietary to DiffusionData and
 * may be covered by U.S. and Foreign Patents, patents in process, and
 * are protected by trade secret or copyright law.
 *******************************************************************************/
package com.pushtechnology.diffusion.client.features.control.clients;

import java.time.Instant;
import java.util.Collection;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.pushtechnology.diffusion.client.Diffusion;
import com.pushtechnology.diffusion.client.callbacks.Registration;
import com.pushtechnology.diffusion.client.callbacks.Stream;
import com.pushtechnology.diffusion.client.features.ErrorReportsException;
import com.pushtechnology.diffusion.client.features.HandlerConflictException;
import com.pushtechnology.diffusion.client.features.InvalidFilterException;
import com.pushtechnology.diffusion.client.features.NoSuchSessionException;
import com.pushtechnology.diffusion.client.features.Security;
import com.pushtechnology.diffusion.client.session.Feature;
import com.pushtechnology.diffusion.client.session.PermissionsException;
import com.pushtechnology.diffusion.client.session.Session;
import com.pushtechnology.diffusion.client.session.SessionClosedException;
import com.pushtechnology.diffusion.client.session.SessionId;
import com.pushtechnology.diffusion.client.types.GlobalPermission;

/**
 * This feature provides the ability for a client session to control other
 * client sessions.
 * <P>
 * It allows for notifications of client session events as well as the ability
 * to manage client sessions (forcibly closing them etc).
 * <p>
 * It also provides the ability to monitor session locks.
 *
 * <H3>Access control</H3> A session must have
 * {@link GlobalPermission#VIEW_SESSION VIEW_SESSION} permission to be able to
 * listen for notifications using {@link #addSessionEventListener} or
 * {@link #setSessionPropertiesListener}, or {@link #getSessionProperties get
 * properties} of sessions or listen for {@link #setQueueEventHandler queue
 * events}.
 * <p>
 * In addition, {@link GlobalPermission#REGISTER_HANDLER REGISTER_HANDLER}
 * permission is required to set a session event listener, a session property
 * listener, or queue event handler.
 * <P>
 * In order to perform operations that change a specific session's state (such
 * as conflating, closing sessions, or changing roles),
 * {@link GlobalPermission#MODIFY_SESSION MODIFY_SESSION} permission is
 * required.
 *
 * <H3>Accessing the feature</H3> This feature may be obtained from a
 * {@link Session session} as follows:
 *
 * <pre>
 * <code>
 * ClientControl clientControl = session.feature(ClientControl.class);
 * </code>
 * </pre>
 *
 * @author DiffusionData Limited
 * @since 5.0
 */
public interface ClientControl extends Feature {

    /**
     * Register a listener that will be notified of
     * {@link ClientControl.SessionEventStream.Event client session events}.
     * <p>
     * These events may occur when a client session is opened, becomes
     * disconnected, reconnects, fails over, or is closed. Events may also occur
     * if any of the session properties of the client session change as a result
     * of a call to {@link #setSessionProperties}.
     * <p>
     * When a listener is first added, by default, it will be called with
     * {@link ClientControl.SessionEventStream.Event.Type#STATE STATE} events
     * for all currently open sessions. The amount of data transferred from the
     * server is proportional to the number of connected clients and is
     * potentially large, especially if session properties are requested. To
     * mitigate this the caller can choose to only receive events for sessions
     * started after a specified time (e.g. from now), in which case events will
     * only be received for sessions that start after that time.
     * <p>
     * A control session can register any number of listeners with different
     * parameters, but the cost in network traffic should be carefully
     * considered.
     * <p>
     * When a listener is no longer required, it may be closed using the
     * {@link Registration} provided by the CompletableFuture result.
     * <p>
     * The {@code parameters} parameter is used to define the level of event
     * detail required.
     *
     * @param listener the listener to be called with session event
     *        notifications
     *
     * @param parameters a {@link SessionEventParameters} object defining the
     *        level of event detail required.
     *        <p>
     *        {@link SessionEventParameters#DEFAULT} may be provided to notify
     *        events for all sessions, including those that already exist, but
     *        returning no session properties.
     *
     * @return a CompletableFuture that completes when the listener has been
     *         registered, returning a {@link Registration} which can be used to
     *         unregister the listener.
     *         <p>
     *         Otherwise, the CompletableFuture will complete exceptionally with
     *         a {@link CompletionException}. Common reasons for failure, listed
     *         by the exception reported as the
     *         {@link CompletionException#getCause() cause}, include:
     *
     *         <ul>
     *         <li>{@link SessionClosedException} &ndash; if the session is
     *         closed;
     *
     *         <li>{@link ErrorReportsException} &ndash; if the session filter
     *         specified in the {@code parameters} was invalid;
     *
     *         <li>{@link PermissionsException} &ndash; if the session does not
     *         have {@code REGISTER_HANDLER} and {@code VIEW_SESSION}
     *         permissions.
     *         </ul>
     *
     * @since 6.11
     */
    CompletableFuture<Registration> addSessionEventListener(
        SessionEventStream listener,
        SessionEventParameters parameters);

    /**
     * Register a listener that will be notified when client sessions are
     * opened, disconnected, reconnected, closed or when selected session
     * property values are updated.
     * <p>
     * This only notifies sessions connecting to the same server as the current
     * session and therefore if the Diffusion system is operating as a cluster
     * then the only way to receive notifications for sessions at all cluster
     * members would be to connect separate sessions (and listeners) to each
     * cluster member. When connecting to a cluster it is recommended that the
     * {@link #addSessionEventListener} method is used instead.
     * <p>
     * When a listener is first set, it will be called with the required
     * properties of all currently open client sessions. The amount of data
     * transferred from the server is proportional to the number of connected
     * clients and is potentially large. The amount of data can be reduced using
     * the {@code requiredProperties} parameter.
     * <p>
     * Each control session can register a single listener. When the listener is
     * no longer required, it may be closed using the {@link Registration}
     * provided by the CompletableFuture result. To change the listener, the
     * previous listener must first be closed.
     * <p>
     * The {@code requiredProperties} parameter is used to select the property
     * values required.
     * <p>
     * The requested property set controls the level of detail provided and
     * whether the listener is called for updates to sessions. If no properties
     * are requested, the listener is not called when session properties are
     * updated.
     *
     * @param listener the listener to be called with session notifications
     *
     * @param requiredProperties a list of required property keys. See
     *        {@link Session} for a full list of available fixed property keys.
     *        To request no properties supply an empty list. To request all
     *        fixed properties include {@link Session#ALL_FIXED_PROPERTIES} as a
     *        key. In this case any other fixed property keys would be ignored.
     *        To request all user properties include
     *        {@link Session#ALL_USER_PROPERTIES} as a key. In this case any
     *        other user properties are ignored.
     *
     * @return a CompletableFuture that completes when the listener has been
     *         registered, returning a {@link Registration} which can be used to
     *         unregister the listener.
     *         <p>
     *         Otherwise, the CompletableFuture will complete exceptionally with
     *         a {@link CompletionException}. Common reasons for failure, listed
     *         by the exception reported as the
     *         {@link CompletionException#getCause() cause}, include:
     *
     *         <ul>
     *         <li>{@link SessionClosedException} &ndash; if the session is
     *         closed;
     *
     *         <li>{@link HandlerConflictException} &ndash; if the session has
     *         already registered a session properties listener;
     *
     *         <li>{@link PermissionsException} &ndash; if the session does not
     *         have {@code REGISTER_HANDLER} and {@code VIEW_SESSION}
     *         permissions.
     *         </ul>
     *
     * @since 6.0
     *
     * @deprecated since 6.11
     *             <p>
     *             Use the new {@link #addSessionEventListener}, which provides
     *             greater functionality and reports on sessions at all cluster
     *             members. This method will be removed in a future release.
     */
    @Deprecated
    CompletableFuture<Registration> setSessionPropertiesListener(
        SessionPropertiesStream listener,
        String... requiredProperties);

    /**
     * Query the server for property values of a specified client session.
     *
     * @param sessionId identifies the client session
     *
     * @param requiredProperties a list of required property keys. See
     *        {@link Session} for a full list of available fixed property keys.
     *        To request no properties supply an empty list. To request all
     *        fixed properties include {@link Session#ALL_FIXED_PROPERTIES} as a
     *        key. In this case any other fixed property keys would be ignored.
     *        To request all user properties include
     *        {@link Session#ALL_USER_PROPERTIES} as a key. In this case any
     *        other user properties are ignored.
     *
     * @return a CompletableFuture that completes when a response is received
     *         from the server, returning a map of the requested session
     *         property values.
     *
     *         <p>
     *         If the session properties were retrieved, the CompletableFuture
     *         will complete successfully. The result type is a map of
     *         properties that were required. If no properties were selected,
     *         the map will be empty.
     *         <p>
     *         Otherwise, the CompletableFuture will complete exceptionally with
     *         a {@link CompletionException}. Common reasons for failure, listed
     *         by the exception reported as the
     *         {@link CompletionException#getCause() cause}, include:
     *
     *         <ul>
     *         <li>{@link NoSuchSessionException} &ndash; if the identified
     *         session was closed before the response was delivered;
     *
     *         <li>{@link PermissionsException} &ndash; if the calling session
     *         does not have the {@code VIEW_SESSION} permission;
     *
     *         <li>{@link SessionClosedException} &ndash; if the calling session
     *         is closed.
     *         </ul>
     *
     * @since 6.5
     */
    CompletableFuture<Map<String, String>> getSessionProperties(
        SessionId sessionId,
        Collection<String> requiredProperties);

    /**
     * Send a request to the server to change the user-defined session
     * properties for a session.
     * <p>
     * It is also permissible to change the values of the following fixed
     * session properties :-
     * <ul>
     * <li>{@link Session#COUNTRY $Country} - will be normalised to upper case
     * <li>{@link Session#LANGUAGE $Language} - will be normalised to lower case
     * <li>{@link Session#LATITUDE $Latitude} - Invalid value will be set to
     * "NaN"
     * <li>{@link Session#LONGITUDE $Longitude} - Invalid value will be set to
     * "NaN"
     * </ul>
     * If values are provided for any other fixed session properties they will
     * be ignored.
     *
     * @param sessionId identifies the client session
     *
     * @param properties the properties to change. Each entry in the map is a
     *        property name and the new value. If the value is {@code null}, any
     *        existing property with that name will be removed (unless it is a
     *        fixed property). Otherwise if the property name does not match any
     *        existing property, that entry will be added as a new property
     *        (although new properties starting with '$' will be ignored).
     *
     * @return a CompletableFuture that completes when a response is received
     *         from the server, returning a map of session properties that
     *         changed.
     *
     *         <p>
     *         If the session properties were updated, the CompletableFuture
     *         will complete successfully. The result type is a map of
     *         properties that changed with their previous values. If no
     *         properties were changed, the map will be empty. If any new
     *         properties were added, the values in the map will be {@code null}
     *         to indicate that they do not have an old value.
     *         <p>
     *         Otherwise, the CompletableFuture will complete exceptionally with
     *         a {@link CompletionException}. Common reasons for failure, listed
     *         by the exception reported as the
     *         {@link CompletionException#getCause() cause}, include:
     *
     *         <ul>
     *         <li>{@link NoSuchSessionException} &ndash; if the identified
     *         session was closed before the response was delivered;
     *
     *         <li>{@link PermissionsException} &ndash; if the calling session
     *         does not have {@code MODIFY_SESSION} permission;
     *
     *         <li>{@link SessionClosedException} &ndash; if the calling session
     *         is closed.
     *         </ul>
     *
     * @since 6.0
     */
    CompletableFuture<Map<String, String>> setSessionProperties(
        SessionId sessionId, Map<String, String> properties);

    /**
     * Send a request to the server to set all sessions that satisfy a session
     * filter with the new user-defined session properties.
     * <p>
     * It is also permissible to change the values of the following fixed
     * session properties:
     * <ul>
     * <li>{@link Session#COUNTRY $Country} - will be normalised to upper case
     * <li>{@link Session#LANGUAGE $Language} - will be normalised to lower case
     * <li>{@link Session#LATITUDE $Latitude} - Invalid value will be set to
     * "NaN"
     * <li>{@link Session#LONGITUDE $Longitude} - Invalid value will be set to
     * "NaN"
     * </ul>
     * If values are provided for any other fixed session properties they will
     * be ignored.
     *
     * @param filter the session filter
     *
     * @param properties the properties to change. Each key should be a valid
     *        session property key (see {@link Session}). Further, keys starting
     *        with '$' should be one of the allowed fixed session property keys
     *        listed above. If the value is null, any existing user-defined
     *        session property with the key will be removed. Fixed session
     *        properties cannot be removed. If the value is non-null, it will be
     *        used to set the value of the the session property with the given
     *        key, adding it if necessary.
     *
     * @return a CompletableFuture that completes when a response is received
     *         from the server.
     *
     *         <p>
     *         If the sessions properties were updated for the sessions that
     *         satisfied the filter, the CompletableFuture will complete
     *         successfully with a {@link SessionFilterOperationResult}.
     *
     *         <p>
     *         Otherwise, the CompletableFuture will complete exceptionally with
     *         a {@link CompletionException}. Common reasons for failure, listed
     *         by the exception reported as the
     *         {@link CompletionException#getCause() cause}, include:
     *
     *         <ul>
     *         <li>{@link PermissionsException} &ndash; if the calling session
     *         does not have {@code MODIFY_SESSION} permission;
     *
     *         <li>{@link SessionClosedException} &ndash; if the calling session
     *         is closed.
     *         </ul>
     *
     * @since 6.0
     */
    CompletableFuture<SessionFilterOperationResult> setSessionProperties(
        String filter, Map<String, String> properties);

    /**
     * Register a handler for client queue threshold events.
     * <P>
     * Each control session can register a single handler. When the handler is
     * no longer required, it may be closed using the {@link Registration}
     * provided by the CompletableFuture result. To set a different handler the
     * current handler must first be closed. For each event, the server will
     * select a single handler.
     * <P>
     * The control session may choose to act upon queue events for a session by
     * activating conflation for the session.
     *
     * @param handler the queue handler to set
     *
     * @return a CompletableFuture that completes when the handler has been
     *         registered, returning a {@link Registration} which can be used to
     *         unregister the handler.
     *         <p>
     *         Otherwise, the CompletableFuture will complete exceptionally with
     *         a {@link CompletionException}. Common reasons for failure, listed
     *         by the exception reported as the
     *         {@link CompletionException#getCause() cause}, include:
     *
     *         <ul>
     *         <li>{@link SessionClosedException} &ndash; if the session is
     *         closed;
     *
     *         <li>{@link HandlerConflictException} &ndash; if the session has
     *         already registered a queue event handler;
     *
     *         <li>{@link PermissionsException} &ndash; if the session does not
     *         have {@code REGISTER_HANDLER} permission and {@code VIEW_SESSION}
     *         permission.
     *         </ul>
     *
     * @since 6.0
     */
    CompletableFuture<Registration> setQueueEventHandler(
        QueueEventStream handler);

    /**
     * Close a client session.
     *
     * @param sessionId identifies the client session to close
     *
     * @return a CompletableFuture that completes when a response is received
     *         from the server.
     *
     *         <p>
     *         If the identified session was closed, the CompletableFuture will
     *         complete successfully. The result type is any rather than Void to
     *         provide forward compatibility with future iterations of this API
     *         that may provide a non-null result with a more specific result
     *         type.
     *
     *         <p>
     *         Otherwise, the CompletableFuture will complete exceptionally with
     *         a {@link CompletionException}. Common reasons for failure, listed
     *         by the exception reported as the
     *         {@link CompletionException#getCause() cause}, include:
     *
     *         <ul>
     *         <li>{@link NoSuchSessionException} &ndash; if the identified
     *         session was closed before the response was delivered;
     *
     *         <li>{@link PermissionsException} &ndash; if the calling session
     *         does not have {@code MODIFY_SESSION} permission;
     *
     *         <li>{@link SessionClosedException} &ndash; if the calling session
     *         is closed.
     *         </ul>
     *
     * @since 6.0
     */
    CompletableFuture<?> close(SessionId sessionId);

    /**
     * Close all client sessions matching a given session filter.
     *
     * @param filter matches the set of client sessions to close. For details on
     *        specifying session filters see {@link Session}.
     *
     * @return a {@link CompletableFuture} that completes when the matching
     *         sessions have been closed.
     *         <p>
     *         If the CompletableFuture completes normally then it returns a
     *         {@link SessionFilterOperationResult} which provides the number of
     *         sessions that matched the filter and have been closed.
     *         <p>
     *         Otherwise, the CompletableFuture will complete exceptionally with
     *         a {@link CompletionException}. Common reasons for failure, listed
     *         by the exception reported as the
     *         {@link CompletionException#getCause() cause}, include:
     *
     *         <ul>
     *         <li>{@link InvalidFilterException} &ndash; if the {@code filter}
     *         was invalid;
     *
     *         <li>{@link PermissionsException} &ndash; if the calling session
     *         does not have {@code MODIFY_SESSION} permission;
     *
     *         <li>{@link SessionClosedException} &ndash; if the calling session
     *         is closed.
     *         </ul>
     *
     * @since 6.5
     */
    CompletableFuture<SessionFilterOperationResult> close(String filter);

    /**
     * Control client queue conflation.
     * <p>
     * Each session begins with conflation enabled or disabled based on the
     * queue configuration of the connector it is using. This method allows
     * conflation to be enabled or disabled for specific sessions at runtime.
     * <p>
     * Conflation is the process of merging or discarding topic updates queued
     * for a session to reduce the server memory footprint and network data.
     * Conflation needs to be enabled for a session and a policy configured for
     * the topic to have an effect. Policies are configured on a per-topic basis
     * using the
     * {@link com.pushtechnology.diffusion.client.topics.details.TopicSpecification#CONFLATION
     * topic property CONFLATION}.
     *
     * @param sessionId identifies the client session
     *
     * @param conflate {@code true} to enable conflation, {@code false} to
     *        disable conflation
     *
     * @return a CompletableFuture that completes when a response is received
     *         from the server.
     *
     *         <p>
     *         If the conflation policy was updated for the identified session,
     *         the CompletableFuture will complete successfully. The result type
     *         is any rather than Void to provide forward compatibility with
     *         future iterations of this API that may provide a non-null result
     *         with a more specific result type.
     *
     *         <p>
     *         Otherwise, the CompletableFuture will complete exceptionally with
     *         a {@link CompletionException}. Common reasons for failure, listed
     *         by the exception reported as the
     *         {@link CompletionException#getCause() cause}, include:
     *
     *         <ul>
     *         <li>{@link NoSuchSessionException} &ndash; if the identified
     *         session was closed before the response was delivered;
     *
     *         <li>{@link PermissionsException} &ndash; if the calling session
     *         does not have {@code MODIFY_SESSION} permission;
     *
     *         <li>{@link SessionClosedException} &ndash; if the calling session
     *         is closed.
     *         </ul>
     *
     * @since 6.0
     */
    CompletableFuture<?> setConflated(
        SessionId sessionId,
        boolean conflate);

    /**
     * Control client queue conflation.
     * <p>
     * Each session begins with conflation enabled or disabled based on the
     * queue configuration of the connector it is using. This method allows
     * conflation to be enabled or disabled for a set of sessions matching a
     * filter at runtime. For more detail on specifying session filters see
     * {@link Session}.
     * <p>
     * Conflation is the process of merging or discarding topic updates queued
     * for a session to reduce the server memory footprint and network data.
     * Conflation needs to be enabled for a session and a policy configured for
     * the topic to have an effect. Policies are configured on a per-topic basis
     * using the
     * {@link com.pushtechnology.diffusion.client.topics.details.TopicSpecification#CONFLATION
     * topic property CONFLATION}.
     *
     * @param filter identifies the client sessions
     *
     * @param conflate {@code true} to enable conflation, {@code false} to
     *        disable conflation
     *
     * @return a CompletableFuture that completes when a response is received
     *         from the server. It contains a
     *         {@link SessionFilterOperationResult} with the number of sessions
     *         that matched the filter.
     *
     *         <p>
     *         If the conflation policy was updated for the identified sessions,
     *         the CompletableFuture will complete successfully. The result type
     *         is a {@link SessionFilterOperationResult} rather than Integer to
     *         provide forward compatibility with future iterations of this API
     *         that may provide additional information.
     *
     *         <p>
     *         Otherwise, the CompletableFuture will complete exceptionally with
     *         a {@link CompletionException}. Common reasons for failure, listed
     *         by the exception reported as the
     *         {@link CompletionException#getCause() cause}, include:
     *
     *         <ul>
     *         <li>{@link InvalidFilterException} &ndash; if the {@code filter}
     *         parameter could not be parsed;
     *
     *         <li>{@link PermissionsException} &ndash; if the calling session
     *         does not have {@code MODIFY_SESSION} permission;
     *
     *         <li>{@link SessionClosedException} &ndash; if the calling session
     *         is closed.
     *         </ul>
     *
     * @since 6.5
     */
    CompletableFuture<SessionFilterOperationResult> setConflated(
        String filter,
        boolean conflate);

    /**
     * Changes the assigned roles of another session.
     * <p>
     * Initially a session has a set of roles assigned during authentication.
     * The set of assigned roles can be obtained from the session's $Roles
     * {@link Session session} property.
     * <p>
     * When a session's assigned roles change, its $Roles property changes
     * accordingly. Changing the assigned roles can change the READ_TOPIC
     * permissions granted to the session. The session's subscriptions will be
     * updated accordingly.
     * <p>
     * The same role must not occur in both {@code rolesToRemove} and
     * {@code rolesToAdd} sets. Either set can be an empty set but not both.
     *
     * @param sessionId identifies the client session for which the change will
     *        be applied
     * @param rolesToRemove a set of roles to be removed from the session. If
     *        one or more roles from the list are not currently assigned, they
     *        are ignored.
     * @param rolesToAdd a set of roles to be added to the session. If one or
     *        more roles from the list are already assigned, they are ignored.
     * @return a {@link CompletableFuture} that completes when session roles
     *         have been changed.
     *         <p>
     *         If the CompletableFuture completes normally then it indicates
     *         that the specified role changes have been applied to the session.
     *         <p>
     *         Otherwise, the CompletableFuture will complete exceptionally with
     *         a {@link CompletionException}. Common reasons for failure, listed
     *         by the exception reported as the
     *         {@link CompletionException#getCause() cause}, include:
     *
     *         <ul>
     *         <li>{@link PermissionsException} &ndash; if the calling session
     *         does not have {@code MODIFY_SESSION} permission;
     *
     *         <li>{@link NoSuchSessionException} &ndash; if there is no session
     *         with the given {@code sessionId};
     *
     *         <li>{@link SessionClosedException} &ndash; if the calling session
     *         is closed.
     *         </ul>
     * @throws IllegalArgumentException if the same role occurs in both
     *         {@code rolesToRemove} and {@code rolesToAdd} sets or if both sets
     *         are empty
     * @since 6.2
     */
    CompletableFuture<?> changeRoles(
        SessionId sessionId,
        Set<String> rolesToRemove,
        Set<String> rolesToAdd) throws IllegalArgumentException;

    /**
     * Allows a session to change the assigned roles of all sessions that
     * satisfy a given session filter.
     * <p>
     * Initially a session has a set of roles assigned during authentication. A
     * current roles set can be obtained from the $Roles {@link Session}
     * property.
     * <p>
     * When a set of session roles changes, its $Roles property changes
     * accordingly. As a role can constrain 'topic' permissions, session
     * subscriptions are re-evaluated based on the new roles set.
     * <p>
     * The same role must not occur in both {@code rolesToRemove} and
     * {@code rolesToAdd} sets. Either set can be an empty set but not both.
     *
     * @param filter identifies the set of client sessions for which the change
     *        will be applied. For details on how to specify session filters see
     *        {@link Session}.
     * @param rolesToRemove a set of roles to be removed from the session. If
     *        one or more roles from the list are not currently assigned, they
     *        are ignored.
     * @param rolesToAdd a set of roles to be added to the session. If one or
     *        more roles from the list are already assigned, they are ignored.
     * @return a {@link CompletableFuture} that completes when session roles
     *         have been changed.
     *         <p>
     *         If the CompletableFuture completes normally then it returns an
     *         integer value which represents a number of sessions that have
     *         matched the filter and for which the specified role changes have
     *         been applied.
     *         <p>
     *         Otherwise, the CompletableFuture will complete exceptionally with
     *         a {@link CompletionException}. Common reasons for failure, listed
     *         by the exception reported as the
     *         {@link CompletionException#getCause() cause}, include:
     *
     *         <ul>
     *         <li>{@link PermissionsException} &ndash; if the calling session
     *         does not have {@code MODIFY_SESSION} permission;
     *
     *         <li>{@link SessionClosedException} &ndash; if the calling session
     *         is closed.
     *
     *         <li>{@link InvalidFilterException} &ndash; if the {@code filter}
     *         parameter could not be parsed;
     *         </ul>
     * @throws IllegalArgumentException if the same role occurs in both
     *         {@code rolesToRemove} and {@code rolesToAdd} sets or if both sets
     *         are empty
     * @since 6.2
     */
    CompletableFuture<Integer> changeRoles(
        String filter,
        Set<String> rolesToRemove,
        Set<String> rolesToAdd) throws IllegalArgumentException;

    /**
     * Returns details of the session (if any) that holds a named session lock.
     *
     * @param lockName the lock name
     *
     * @return a {@link CompletableFuture} that completes with details of the
     *         named lock.
     *         <p>
     *         If the CompletableFuture completes normally then it returns the
     *         details of the session that currently holds the lock, or null if
     *         no session holds the named lock.
     *         <p>
     *         Otherwise, the CompletableFuture will complete exceptionally with
     *         a {@link CompletionException}. Common reasons for failure, listed
     *         by the exception reported as the
     *         {@link CompletionException#getCause() cause}, include:
     *
     *         <ul>
     *         <li>{@link PermissionsException} &ndash; if the calling session
     *         does not have {@code VIEW_SERVER} permissions;
     *
     *         <li>{@link SessionClosedException} &ndash; if the calling session
     *         is closed.
     *         </ul>
     * @since 6.12
     */
    CompletableFuture<SessionLockDetails> getSessionLock(String lockName);

    /**
     * Returns details of all session locks currently held.
     *
     * @return a {@link CompletableFuture} that completes with details of all
     *         session locks that are currently held.
     *         <p>
     *         If the CompletableFuture completes normally then it returns a map
     *         of session locks keyed on the lock name where the value
     *         identifies the session currently holding the lock. If there are
     *         no session locks currently held by anybody this will return an
     *         empty map.
     *         <p>
     *         Otherwise, the CompletableFuture will complete exceptionally with
     *         a {@link CompletionException}. Common reasons for failure, listed
     *         by the exception reported as the
     *         {@link CompletionException#getCause() cause}, include:
     *
     *         <ul>
     *         <li>{@link PermissionsException} &ndash; if the calling session
     *         does not have {@code VIEW_SERVER} permissions;
     *
     *         <li>{@link SessionClosedException} &ndash; if the calling session
     *         is closed.
     *         </ul>
     * @since 6.12
     */
    CompletableFuture<Map<String, SessionLockDetails>> getSessionLocks();

    /**
     * Handler for client queue events.
     * <P>
     * An implementation can be registered using
     * {@link ClientControl#setQueueEventHandler(QueueEventStream)}.
     *
     * @since 6.0
     */
    interface QueueEventStream extends Stream {

        /**
         * The configured upper threshold for a client's queue has been reached.
         *
         * @param sessionId the client session identifier
         *
         * @param policy the message queue policy
         */
        void onUpperThresholdCrossed(
            SessionId sessionId,
            MessageQueuePolicy policy);

        /**
         * The configured lower threshold for a client's queue has been reached.
         *
         * @param sessionId the client session identifier
         *
         * @param policy the message queue policy
         */
        void onLowerThresholdCrossed(
            SessionId sessionId,
            MessageQueuePolicy policy);

        /**
         * This provides a default implementation of {@link QueueEventStream}.
         * <P>
         * This simply logs events at debug level and should only be used for
         * diagnostic purposes. This may be extended to implement the
         * notifications you wish to act upon.
         */
        class Default extends Stream.Default
            implements QueueEventStream {

            private static final Logger LOG =
                LoggerFactory
                    .getLogger(QueueEventStream.Default.class);

            @Override
            public void onUpperThresholdCrossed(
                final SessionId sessionId,
                final MessageQueuePolicy policy) {

                LOG.debug(
                    "{} - Session {} with policy {} : " +
                        "Upper queue threshold crossed",
                    this,
                    sessionId,
                    policy);
            }

            @Override
            public void onLowerThresholdCrossed(
                final SessionId sessionId,
                final MessageQueuePolicy policy) {

                LOG.debug(
                    "{} - Session {} with policy {} : " +
                        "Lower queue threshold crossed",
                    this,
                    sessionId,
                    policy);
            }
        }
    }

    /**
     * The handler for session properties listener notifications.
     * <P>
     * This is used along with
     * {@link ClientControl#setSessionPropertiesListener} to obtain
     * notifications for client sessions.
     * <P>
     * Callbacks with a {@code properties} parameter provide a map of the
     * property values requested when registering the listener.
     *
     * @since 6.0
     *
     * @deprecated since 6.11
     *             <p>
     *             Use the new {@link ClientControl#addSessionEventListener},
     *             which provides greater functionality and reports sessions at
     *             all cluster members. This interface will be removed in a
     *             future release.
     */
    @Deprecated
    interface SessionPropertiesStream extends Stream {

        /**
         * Event type.
         */
        enum EventType {

            /**
             * One or more relevant session properties have been updated.
             */
            UPDATED,

            /**
             * Session has reconnected.
             */
            RECONNECTED,

            /**
             * Use in a clustered configuration to indicate that the session
             * failed over from another server.
             */
            FAILED_OVER,

            /**
             * Session has disconnected.
             */
            DISCONNECTED
        }

        /**
         * Notification that a new client session has been opened.
         * <P>
         * When the listener is registered, this will be called for all existing
         * sessions. It will then be called for every new session that opens
         * whilst the listener is registered.
         * <P>
         * This will be called for every client session regardless of requested
         * session properties.
         *
         * @param sessionId the session identifier
         *
         * @param properties the map of requested session property values. This
         *        can be empty if no properties were requested. If a requested
         *        property did not exist, it will not be present in the map.
         */
        void onSessionOpen(
            SessionId sessionId,
            Map<String, String> properties);

        /**
         * Notification of a session event that can result in a change of
         * properties.
         *
         * @param sessionId the client session id
         *
         * @param eventType indicates the type of event
         *
         * @param properties the map of current requested property values. This
         *        can be empty if no properties were requested. If a requested
         *        property did not exist, it will not be present in the map.
         *
         * @param previousValues a map of the previous values of keys that have
         *        changed. This will only contain keys that have changed and not
         *        the whole required property set. This can be empty if an event
         *        is being reported that did not result in the change of any of
         *        the required properties. When a new property is added, the
         *        value in this map will be null. When a property is removed, it
         *        will have a value in this map but not in {@code properties}.
         */
        void onSessionEvent(
            SessionId sessionId,
            EventType eventType,
            Map<String, String> properties,
            Map<String, String> previousValues);

        /**
         * Notification that a client session has closed.
         * <P>
         * This will be called for every client that closes whilst the listener
         * is registered regardless of requested session properties.
         *
         * @param sessionId the session identifier of the client that has closed
         *
         * @param properties the map of requested property values at the point
         *        when the session was closed. This can be empty if no
         *        properties were requested. If a requested property did not
         *        exist, it will not be present in the map.
         *
         * @param closeReason why the session was closed
         */
        void onSessionClose(
            SessionId sessionId,
            Map<String, String> properties,
            CloseReason closeReason);

        /**
         * This provides a default implementation of
         * {@link SessionPropertiesStream} which will simply log session
         * properties callbacks at debug level. This should only be used for
         * diagnostic purposes.
         */
        class Default extends Stream.Default
            implements SessionPropertiesStream {

            private static final Logger LOG =
                LoggerFactory
                    .getLogger(SessionPropertiesStream.Default.class);

            @Override
            public void onSessionOpen(
                SessionId sessionId,
                Map<String, String> properties) {
                LOG.debug(
                    "{} - Client session {} opened : {}",
                    this,
                    sessionId,
                    properties);
            }

            @Override
            public void onSessionEvent(
                SessionId sessionId,
                EventType eventType,
                Map<String, String> properties,
                Map<String, String> previousValues) {
                LOG.debug(
                    "{} - Client session {} event : {}, Properties={}, Previous Values={}",
                    this,
                    sessionId,
                    eventType,
                    properties,
                    previousValues);
            }

            @Override
            public void onSessionClose(
                SessionId sessionId,
                Map<String, String> properties,
                CloseReason closeReason) {
                LOG.debug(
                    "{} - Client session {} closed - {} : {}",
                    this,
                    sessionId,
                    closeReason,
                    properties);
            }
        }
    }

    /**
     * The stream to receive for session event listener notifications.
     * <P>
     * This is used along with {@link ClientControl#addSessionEventListener
     * addSessionEventListener} to obtain notifications for client sessions.
     *
     * @since 6.11
     */
    interface SessionEventStream extends Stream {

        /**
         * The interface for an event delivered by the stream.
         */
        interface Event {

            /**
             * The event type.
             */
            enum Type {

                /**
                 * The event indicates a change of state of the identified
                 * session, including the opening of a new session.
                 */
                STATE,

                /**
                 * The event indicates a change to the session properties of the
                 * identified session.
                 */
                PROPERTIES
            }

            /**
             * The session state.
             */
            enum State {

                /**
                 * The session is connected and active.
                 */
                ACTIVE,

                /**
                 * The session is disconnected.
                 */
                DISCONNECTED,

                /**
                 * The session has reconnected to the same server after a
                 * disconnection.
                 */
                RECONNECTED,

                /**
                 * The session has failed over to a different server in the
                 * cluster after a disconnection.
                 */
                FAILED_OVER,

                /**
                 * The session has closed.
                 */
                CLOSED
            }

            /**
             * Provides the session identifier of the client session that the
             * event relates to.
             *
             * @return the client session id
             */
            SessionId sessionId();

            /**
             * Provides the event type.
             *
             * @return the event type
             */
            Type type();

            /**
             * Provides the current state of the session.
             *
             * @return the session state
             */
            State state();

            /**
             * Convenience method to determine whether the event notifies the
             * opening of a new session.
             * <p>
             * This is equivalent to:
             * <p>
             * <code>
             * event.type() == Type.STATE &amp;&amp; event.state() == State.ACTIVE
             * </code>
             *
             * @return true if the type is {@code STATE} and the state is {@code
             *         ACTIVE}, otherwise false
             */
            boolean isOpenEvent();

            /**
             * Provides the session's properties.
             * <p>
             * Only those properties requested when registering the listener
             * will be present in the map.
             * <p>
             * If no properties were requested or {@link #state} is
             * {@link State#CLOSED CLOSED} the map will be empty.
             *
             * @return the map of session properties
             */
            Map<String, String> properties();

            /**
             * Provides a map of the previous values of session properties that
             * have changed.
             * <p>
             * This map will be empty if no session properties were requested or
             * the event did not result in a change of any properties.
             * <p>
             * This map will only be populated if {@link #type} is
             * {@link Type#PROPERTIES PROPERTIES} or {@link #state} is
             * {@link State#RECONNECTED RECONNECTED}, {@link State#FAILED_OVER
             * FAILED_OVER}, or {@link State#DISCONNECTED DISCONNECTED}.
             * <p>
             * This will only contain keys that have changed and not the whole
             * required property set.
             * <p>
             * When a new property is added the value in this map will be
             * {@code null}.
             * <p>
             * When a property is removed there will be a value in this map but
             * not in {@link #properties}.
             *
             * @return the map of changed session properties
             */
            Map<String, String> changedProperties();

            /**
             * Provides the reason a session was closed.
             * <p>
             * If {@link #state} is {@link State#CLOSED CLOSED} this returns the
             * reason the session was closed.
             *
             * @return the reason for session closure or null if {@link #state}
             *         is not {@link State#CLOSED CLOSED}.
             */
            CloseReason closeReason();

        }

        /**
         * Notification of a session event.
         *
         * @param event the session event
         */
        void onSessionEvent(Event event);

        /**
         * This provides a default implementation of {@link SessionEventStream}
         * which will simply log session event callbacks at debug level. This
         * should only be used for diagnostic purposes.
         */
        class Default extends Stream.Default implements SessionEventStream {

            private static final Logger LOG =
                LoggerFactory
                    .getLogger(SessionEventStream.Default.class);

            @Override
            public void onSessionEvent(Event event) {
                LOG.debug(event.toString());
            }
        }
    }

    /**
     * Provides parameters which specify the level of detail required by a
     * {@link SessionEventStream} registered using
     * {@link ClientControl#addSessionEventListener setSessionEventListener}.
     * <p>
     * Parameters may be built using the
     * {@link ClientControl.SessionEventParameters.Builder builder}.
     *
     * @since 6.11
     */
    interface SessionEventParameters {

        /**
         * Provides a default instance of
         * {@link ClientControl.SessionEventParameters SessionEventParameters}
         * that will request notifications for all sessions, including those
         * currently active, but providing no session properties.
         */
        SessionEventParameters DEFAULT =
            Diffusion.newSessionEventParametersBuilder().build();

        /**
         * A builder for {@link ClientControl.SessionEventParameters
         * SessionEventParameters}.
         *
         * An instance of such a builder is created using
         * {@link Diffusion#newSessionEventParametersBuilder}.
         */
        interface Builder {

            /**
             * Sets a session filter which will determine which sessions events
             * will be notified for.
             * <p>
             * See {@link Session} for a full description of how to specify
             * session filters.
             * <p>
             * If no filter is specified then notifications will be provided for
             * all sessions that satisfy any other specified requirements.
             * <p>
             * Specifying {@code null} will remove any current filter from the
             * builder.
             *
             * @param filter the session filter
             *
             * @return this builder
             */
            Builder filter(String filter);

            /**
             * Specifies the session property keys of all session properties to
             * be returned with events.
             * <p>
             * See {@link Session} for a full list of available fixed property
             * keys.
             * <p>
             * To request all fixed properties include
             * {@link Session#ALL_FIXED_PROPERTIES} as a key. In this case any
             * other fixed property keys would be ignored.
             * <p>
             * To request all user properties include
             * {@link Session#ALL_USER_PROPERTIES} as a key. In this case any
             * other user properties are ignored.
             * <p>
             * If this is not specified (or no property keys are provided) then
             * no session property values will be returned with the events and
             * events of type
             * {@link ClientControl.SessionEventStream.Event.Type#PROPERTIES
             * PROPERTIES} will not be notified.
             *
             * @param properties a list of required property keys
             *
             * @return this builder
             */
            Builder properties(String... properties);

            /**
             * Used to indicate that events are only to be reported for sessions
             * that start after a specified time.
             * <p>
             * If this is not set, the default is to notify events for all
             * current sessions, even if they started before registration of the
             * listener. This could potentially result in a large number of
             * events for all current sessions.
             * <p>
             * If the user is only interested in new sessions
             * {@code after(Instant.now())} could be used.
             *
             * @param time only sessions starting after the specified time will
             *        be notified. Specifying null will remove any previously
             *        specified time from the builder
             *
             * @return this builder
             */
            Builder after(Instant time);

            /**
             * Builds a new {@link ClientControl.SessionEventParameters
             * SessionEventParameters} instance with the current settings of the
             * builder.
             *
             * @return a new {@link ClientControl.SessionEventParameters
             *         SessionEventParameters} instance
             */
            SessionEventParameters build();

        }

        /**
         * Returns the session filter.
         *
         * @return the session filter or null if one has not been set
         */
        String filter();

        /**
         * Returns the requested properties.
         *
         * @return the set of requested property keys
         */
        Set<String> properties();

        /**
         * Indicates a session start time after which session events should be
         * notified.
         *
         * @return the instant representing the session start time after which
         *         sessions should be notified. This will return null if not set
         */
        Instant after();

    }

    /**
     * The server's view of why a client session was closed.
     */
    enum CloseReason {

        /**
         * The connection to the client was lost - possibly dropped by the
         * client. Recoverable.
         * <P>
         * A client may be closed for many reasons that are presented as
         * CONNECTION_LOST.
         * <P>
         * During connection the connection can be lost. The server might have
         * received a connection or reconnection request from a client already
         * connected. The server might have received a reconnection request
         * without a session Id. The connection may not have been authorised
         * because the credentials are wrong. The maximum number of clients
         * might already be connected.
         * <P>
         * Once connected the connection can be lost for different reasons. If
         * the client closes its connection while the server is writing a
         * message to the client. With the chunked encoding based connection the
         * HTTP response is completed by the server. If the client does not open
         * a new request within a timeout the client will be closed. If a poll
         * request times out and the server finds that the connection has
         * already been closed by the client.
         */
        CONNECTION_LOST,

        /**
         * An unexpected IO Exception occurred. Recoverable.
         * <P>
         * While trying to perform an I/O operation an exception was generated.
         * This often means that a read was attempted from a closed TCP
         * connection.
         * <P>
         * When handling SSL connections if there is a problem encrypting or
         * decrypting a message the client will be closed for this reason.
         */
        IO_EXCEPTION,

        /**
         * The client had become unresponsive.
         * <P>
         * The client has either failed to respond to a ping message in a timely
         * manner or the client has failed to open an HTTP poll for messages.
         * The client does not appear to be receiving messages.
         */
        CLIENT_UNRESPONSIVE,

        /**
         * The maximum outbound queue size was reached for the client. Not
         * recoverable.
         * <P>
         * Messages sent to the client are placed in a queue. This queue has a
         * maximum allowed size. If the queue limit is reached the client is
         * closed and the queue discarded. The queue is intended to protect
         * against slow consumers, reaching the queue limit is taken to mean
         * that the client cannot keep up with the number of messages sent to
         * it.
         */
        MESSAGE_QUEUE_LIMIT_REACHED,

        /**
         * The client requested close. Not recoverable.
         */
        CLOSED_BY_CLIENT,

        /**
         * The client sent a message that exceeded the maximum message size that
         * can be processed by the server.
         * <P>
         * The server has a maximum message size that it can process on any
         * single connector. If a client sends a message larger than this the
         * server is unable to process it. When this happens the message is
         * discarded and the client is closed.
         */
        MESSAGE_TOO_LARGE,

        /**
         * An internal error occurred.
         * <P>
         * An unexpected internal error has occurred. The server logs will
         * contain more detail.
         */
        INTERNAL_ERROR,

        /**
         * An inbound message with an invalid format was received.
         * <P>
         * A message received by the server is not a valid Diffusion message.
         * The server is unable to process this and closes the client that sent
         * it.
         */
        INVALID_INBOUND_MESSAGE,

        /**
         * The client connection was aborted by the server, possibly because the
         * connection was disallowed.
         * <P>
         * This is may be because the connection was disallowed. Abort messages
         * are also sent to clients that have unrecognised client IDs. This may
         * be because the server closed the client previously but the client is
         * unaware of this and tried to continue interacting with the server.
         */
        ABORTED,

        /**
         * Loss of messages from the client has been detected. For example,
         * whilst waiting for the arrival of missing messages in a sequence of
         * messages a timeout has occurred.
         * <P>
         * HTTP based transports use multiple TCP connections. This can cause
         * the messages to be received out of order. To reorder the messages
         * those sent to the server may contain a sequence number indicating the
         * correct order.
         * <P>
         * If a message is received out of order there is a short time for the
         * earlier messages to be received. If the messages are not received in
         * this time the client is closed.
         * <P>
         * Missing, invalid or duplicate sequence numbers will also close the
         * client for this reason.
         * <P>
         * This cannot be recovered from as the client and the server are in
         * inconsistent states.
         */
        LOST_MESSAGES,

        /**
         * Closed by a client session.
         * <P>
         * A control session initiated the client close.
         */
        CLOSED_BY_CONTROLLER,

        /**
         * The session has failed over to a different Diffusion server.
         * <P>
         * The session is still open but is now connected to a different
         * Diffusion server. This server has evicted its view of the session
         * from its set of local sessions.
         *
         * @since 5.8
         */
        FAILED_OVER,

        /**
         * The session had an {@link Session#EXPIRY_TIME} specified which
         * expired before the session {@link Security#reauthenticate
         * re-authenticated}.
         *
         * @since 6.12
         */
        EXPIRED,

        /**
         * The session's authentication was revoked by a privileged user.
         *
         * @since 6.12
         */
        REVOKED,

        /**
         * The session has been closed to make way for a new session.
         * <p>
         * This can only occur for an MQTT session.
         *
         * @since 6.12
         */
        SESSION_TAKEN_OVER

    }

    /**
     * Result of ClientControl operations that identify sessions using a client
     * filter and provide a result through a {@link CompletableFuture}.
     *
     * @since 6.5
     */
    interface SessionFilterOperationResult {
        /**
         * @return the number of sessions that were selected by the provided
         *         filter.
         */
        int selected();
    }

    /**
     * Provides details of a session lock.
     *
     * @since 6.12
     */
    interface SessionLockDetails {

        /**
         * Returns the name of the server that the session holding the lock is
         * connected to.
         *
         * @return the server name
         */
        String serverName();

        /**
         * Returns the session identifier of the session that holds the lock.
         *
         * @return the session holding the lock
         */
        SessionId sessionId();

        /**
         * Returns the lock sequence number.
         *
         * @return lock sequence number
         */
        long sequence();
    }
}
