/*******************************************************************************
 * Copyright (c) 2014, 2025 DiffusionData Ltd., All Rights Reserved.
 *
 * Use is subject to licence terms.
 *
 * NOTICE: All information contained herein is, and remains the
 * property of DiffusionData. The intellectual and technical
 * concepts contained herein are proprietary to DiffusionData and
 * may be covered by U.S. and Foreign Patents, patents in process, and
 * are protected by trade secret or copyright law.
 *******************************************************************************/
package com.pushtechnology.diffusion.client.features.control.clients;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;

import com.pushtechnology.diffusion.client.features.ClusterRoutingException;
import com.pushtechnology.diffusion.client.features.ScriptException;
import com.pushtechnology.diffusion.client.session.Feature;
import com.pushtechnology.diffusion.client.session.PermissionsException;
import com.pushtechnology.diffusion.client.session.SessionClosedException;

/**
 * The super interface for features that update security stores.
 *
 * @author DiffusionData Limited
 * @since 5.3
 */
public interface SecurityStoreFeature extends Feature {

    /**
     * Send a command script to the server to update the security store.
     * <P>
     * The script may be generated using a {@code ScriptBuilder} obtained from
     * the feature.
     * <p>
     * Each line of <code>script</code> is a command to update the store.
     * <p>
     * The server attempts to execute each command in order against a copy of
     * the store. If any command fails, none of the changes will be applied. If
     * all commands succeed, the changes will be applied.
     * <p>
     * If the server is configured for topic replication then the changes will
     * be replicated to all members of the cluster.
     *
     * @param commandScript the script
     *
     * @return a CompletableFuture that completes when a response is received
     *         from the server.
     *
     *         <p>
     *         If the request was successful, the CompletableFuture will
     *         complete successfully. The result type is any rather than Void to
     *         provide forward compatibility with future iterations of this API
     *         that may provide a non-null result with a more specific result
     *         type.
     *
     *         <p>
     *         Otherwise, the CompletableFuture will complete exceptionally with
     *         a {@link CompletionException}. Common reasons for failure, listed
     *         by the exception reported as the
     *         {@link CompletionException#getCause() cause}, include:
     *
     *         <ul>
     *
     *         <li>{@link ScriptException} &ndash; if {@code commandScript} is
     *         invalid;
     *
     *         <li>{@link PermissionsException} &ndash; if the session does
     *         not have {@code MODIFY_SECURITY} permission;
     *
     *         <li>{@link ClusterRoutingException} &ndash; if the operation failed
     *         due to a transient cluster error;
     *
     *         <li>{@link SessionClosedException} &ndash; if the session is
     *         closed.
     *         </ul>
     * @since 6.0
     */
    CompletableFuture<?> updateStore(String commandScript);

}
