/*******************************************************************************
 * Copyright (c) 2014, 2025 DiffusionData Ltd., All Rights Reserved.
 *
 * Use is subject to licence terms.
 *
 * NOTICE: All information contained herein is, and remains the
 * property of DiffusionData. The intellectual and technical
 * concepts contained herein are proprietary to DiffusionData and
 * may be covered by U.S. and Foreign Patents, patents in process, and
 * are protected by trade secret or copyright law.
 *******************************************************************************/
package com.pushtechnology.diffusion.client.features.control.topics;

import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;

import com.pushtechnology.diffusion.client.features.ClusterRoutingException;
import com.pushtechnology.diffusion.client.features.InvalidFilterException;
import com.pushtechnology.diffusion.client.features.NoSuchSessionException;
import com.pushtechnology.diffusion.client.features.Topics;
import com.pushtechnology.diffusion.client.features.control.clients.ClientControl.SessionEventStream;
import com.pushtechnology.diffusion.client.session.Feature;
import com.pushtechnology.diffusion.client.session.PermissionsException;
import com.pushtechnology.diffusion.client.session.Session;
import com.pushtechnology.diffusion.client.session.SessionClosedException;
import com.pushtechnology.diffusion.client.session.SessionId;
import com.pushtechnology.diffusion.client.topics.TopicSelection;
import com.pushtechnology.diffusion.client.topics.TopicSelector;
import com.pushtechnology.diffusion.client.topics.TopicSelectors;
import com.pushtechnology.diffusion.client.types.GlobalPermission;
import com.pushtechnology.diffusion.client.types.PathPermission;

/**
 * This feature allows a session to subscribe or unsubscribe other sessions to
 * topics, on a single server or across a cluster.
 * <P>
 * Requests to subscribe sessions to topics can be submitted at any time even if
 * the topics do not exist at the server. {@link TopicSelector Topic selectors}
 * are used on subscription to match against topics at the server. The session
 * will become subscribed to any topics that exist at the server that match the
 * selector (unless they are already subscribed, or the session has insufficient
 * permission). The subscription request is also retained at the server so that
 * if any newly created topics match the selector, the session will then become
 * subscribed to it (unless a subsequent unsubscription cancels it).
 * <P>
 * Specific sessions (anywhere in a cluster) may be subscribed/unsubscribed if
 * the {@link SessionId} is known.
 * <P>
 * Subscriptions may also be requested using 'session filters' (see
 * {@link Session} for a full description of session filters), where all
 * sessions (anywhere in a cluster) that satisfy a particular filter expression
 * will be subscribed/unsubscribed. The filter is only evaluated once against
 * the current sessions that exist at the time - it is not retained and applied
 * to any sessions that are created later. In order to be notified of new
 * sessions as they are created {@link SessionEventStream session event
 * listeners} can be used and those sessions subscribed as required based upon
 * their session properties.
 * <p>
 * This feature also provides the ability to query the topic selections of
 * another session.
 *
 * <H3>Access control</H3><br>
 * To directly subscribe other sessions to topics, a session must have
 * {@link GlobalPermission#MODIFY_SESSION MODIFY_SESSION} permission, and
 * {@link PathPermission#SELECT_TOPIC SELECT_TOPIC} permission for the
 * topic selector used for subscription. The subscribed sessions
 * will only be subscribed to matching topics for which they have
 * {@link PathPermission#READ_TOPIC READ_TOPIC} permission.
 * <p>
 * To directly unsubscribe other sessions, a session must have
 * {@link GlobalPermission#MODIFY_SESSION MODIFY_SESSION} permission, and
 * {@link PathPermission#SELECT_TOPIC SELECT_TOPIC} permission for
 * the topic selector used for unsubscription.
 * <p>
 * Getting the topic selections for a specified session requires
 * {@link GlobalPermission#VIEW_SESSION VIEW_SESSION} permission.
 *
 * <H3>Accessing the feature</H3> This feature may be obtained from a
 * {@link Session session} as follows:
 *
 * <pre>
 * <code>
 * SubscriptionControl subsControl = session.feature(SubscriptionControl.class);
 * </code>
 * </pre>
 *
 * @author DiffusionData Limited
 * @since 5.0
 */
public interface SubscriptionControl extends Feature {

    /**
     * Subscribe another session to topics.
     * <p>
     * This is equivalent to calling
     * {@link #subscribe(SessionId, String, String)} specifying the
     * {@link Topics#DEFAULT_SELECTION_SCOPE default selection scope}.
     *
     * @see Topics Topic Selection Scopes
     *
     * @since 6.0
     */
    default CompletableFuture<?> subscribe(SessionId sessionId, String topics) {
        return subscribe(sessionId, topics, Topics.DEFAULT_SELECTION_SCOPE);
    }

    /**
     * Subscribe another session to topics.
     * <p>
     * This is equivalent to calling
     * {@link #subscribe(SessionId, TopicSelector, String)} with a selector
     * parsed using {@link TopicSelectors#parse(String)}.
     *
     * @param sessionId identifies the session to subscribe
     *
     * @param topics the topics to subscribe to specified as a
     *        {@link TopicSelector} expression
     *
     * @param scope specifies the scope of the selection. See {@link Topics
     *        Topic Selection Scopes}
     *
     * @throws IllegalArgumentException if {@code topics} is not a valid
     *         selector expression
     *
     * @return a CompletableFuture that completes when the server has processed
     *         the subscription request.
     *
     *         <p>
     *         If the subscription was accepted, the CompletableFuture will
     *         complete successfully. The result type is any rather than Void to
     *         provide forward compatibility with future iterations of this API
     *         that may provide a non-null result with a more specific result
     *         type.
     *
     *         <p>
     *         Otherwise, the CompletableFuture will complete exceptionally with
     *         a {@link CompletionException}. Common reasons for failure, listed
     *         by the exception reported as the
     *         {@link CompletionException#getCause() cause}, include:
     *
     *         <ul>
     *         <li>{@link PermissionsException} &ndash; if the calling session
     *         does not have {@code MODIFY_SESSION} permission and
     *         {@code SELECT_TOPIC} permission for the selector expression;
     *
     *         <li>{@link NoSuchSessionException} &ndash; if there is no session
     *         with the given {@code sessionId} on the current server or on any
     *         other servers in a clustered environment;
     *
     *         <li>{@link SessionClosedException} &ndash; if the session is
     *         closed.
     *
     *         <li>{@link ClusterRoutingException} &ndash; if the operation
     *         failed due to a transient cluster error;
     *         </ul>
     *
     * @since 6.12
     */
    CompletableFuture<?> subscribe(
        SessionId sessionId,
        String topics,
        String scope)
        throws IllegalArgumentException;

    /**
     * Subscribe another session to topics.
     * <p>
     * This is equivalent to calling
     * {@link #subscribe(SessionId, TopicSelector, String)} specifying the
     * {@link Topics#DEFAULT_SELECTION_SCOPE default selection scope}.
     *
     * @see Topics Topic Selection Scopes
     *
     * @since 6.0
     */
    default CompletableFuture<?> subscribe(
        SessionId sessionId,
        TopicSelector topics) {

        return subscribe(sessionId, topics, Topics.DEFAULT_SELECTION_SCOPE);
    }

    /**
     * Subscribe another session to topics.
     *
     * <p>
     * New subscriptions will be established for existing topics that match the
     * provided topic selector and for which the subscribed session has
     * {@code READ_TOPIC} permission. The topic selector will be added to the
     * topic selections of the subscribed session, and re-evaluated when new
     * topics are added or the session's security roles change.
     *
     * <p>
     * A session that does not have {@code SELECT_TOPIC} permission for a topic
     * cannot subscribe directly, but can be subscribed indirectly using this
     * method.
     *
     * @param sessionId identifies the session to subscribe
     *
     * @param topics identifies the topics to subscribe to
     *
     * @param scope specifies the scope of the selection. See {@link Topics
     *        Topic Selection Scopes}
     *
     * @return a CompletableFuture that completes when the server has processed
     *         the subscription request.
     *
     *         <p>
     *         If the subscription was accepted, the CompletableFuture will
     *         complete successfully. The result type is any rather than Void to
     *         provide forward compatibility with future iterations of this API
     *         that may provide a non-null result with a more specific result
     *         type.
     *
     *         <p>
     *         Otherwise, the CompletableFuture will complete exceptionally with
     *         a {@link CompletionException}. Common reasons for failure, listed
     *         by the exception reported as the
     *         {@link CompletionException#getCause() cause}, include:
     *
     *         <ul>
     *         <li>{@link PermissionsException} &ndash; if the calling session
     *         does not have {@code MODIFY_SESSION} permission and
     *         {@code SELECT_TOPIC} permission for the selector expression;
     *
     *         <li>{@link NoSuchSessionException} &ndash; if there is no session
     *         with the given {@code sessionId} on the current server or on any
     *         other servers in a clustered environment;
     *
     *         <li>{@link SessionClosedException} &ndash; if the session is
     *         closed.
     *
     *         <li>{@link ClusterRoutingException} &ndash; if the operation
     *         failed due to a transient cluster error;
     *         </ul>
     * @since 6.12
     */
    CompletableFuture<?> subscribe(
        SessionId sessionId,
        TopicSelector topics,
        String scope);

    /**
     * Unsubscribe another session from topics.
     * <p>
     * This is equivalent to calling
     * {@link #unsubscribe(SessionId, String, String)} specifying the
     * {@link Topics#DEFAULT_SELECTION_SCOPE default selection scope}.
     *
     * @see Topics Topic Selection Scopes
     *
     * @since 6.0
     */
    default CompletableFuture<?> unsubscribe(
        SessionId sessionId,
        String topics) {
        return unsubscribe(sessionId, topics, Topics.DEFAULT_SELECTION_SCOPE);
    }

    /**
     * Unsubscribe another session from topics.
     *
     * <P>
     * This is equivalent to calling
     * {@link #unsubscribe(SessionId, TopicSelector, String)} with a selector
     * parsed using {@link TopicSelectors#parse(String)}.
     *
     * @param sessionId identifies the session to unsubscribe
     *
     * @param topics the topics to unsubscribe specified as a
     *        {@link TopicSelector} expression
     *
     * @param scope specifies the scope of the selection. See {@link Topics
     *        Topic Selection Scopes}
     *
     * @throws IllegalArgumentException if {@code topics} is not a valid
     *         selector expression
     *
     * @return a CompletableFuture that completes when the server has processed
     *         the unsubscription request.
     *
     *         <p>
     *         If the unsubscription was accepted, the CompletableFuture will
     *         complete successfully. The result type is any rather than Void to
     *         provide forward compatibility with future iterations of this API
     *         that may provide a non-null result with a more specific result
     *         type.
     *
     *         <p>
     *         Otherwise, the CompletableFuture will complete exceptionally with
     *         a {@link CompletionException}. Common reasons for failure, listed
     *         by the exception reported as the
     *         {@link CompletionException#getCause() cause}, include:
     *
     *         <ul>
     *         <li>{@link PermissionsException} &ndash; if the calling session
     *         does not have {@code MODIFY_SESSION} permission and
     *         {@code SELECT_TOPIC} permission for the selector expression;
     *
     *         <li>{@link NoSuchSessionException} &ndash; if there is no session
     *         with the given {@code sessionId} on the current server or on any
     *         other servers in a clustered environment;
     *
     *         <li>{@link SessionClosedException} &ndash; if the session is
     *         closed.
     *
     *         <li>{@link ClusterRoutingException} &ndash; if the operation
     *         failed due to a transient cluster error;
     *         </ul>
     *
     * @since 6.12
     */
    CompletableFuture<?> unsubscribe(
        SessionId sessionId,
        String topics,
        String scope)
        throws IllegalArgumentException;

    /**
     * Unsubscribe another session from topics.
     * <p>
     * This is equivalent to calling
     * {@link #unsubscribe(SessionId, TopicSelector, String)} specifying the
     * {@link Topics#DEFAULT_SELECTION_SCOPE default selection scope}.
     *
     * @see Topics Topic Selection Scopes
     *
     * @since 6.0
     */
    default CompletableFuture<?> unsubscribe(
        SessionId sessionId,
        TopicSelector topics) {
        return unsubscribe(sessionId, topics, Topics.DEFAULT_SELECTION_SCOPE);
    }

    /**
     * Unsubscribe another session from topics.
     *
     * @param sessionId identifies the session to unsubscribe
     *
     * @param topics the topics to unsubscribe from
     *
     * @param scope specifies the scope of the selection. See {@link Topics
     *        Topic Selection Scopes}
     *
     * @return a CompletableFuture that completes when the server has processed
     *         the unsubscription request.
     *
     *         <p>
     *         If the unsubscription was accepted, the CompletableFuture will
     *         complete successfully. The result type is any rather than Void to
     *         provide forward compatibility with future iterations of this API
     *         that may provide a non-null result with a more specific result
     *         type.
     *
     *         <p>
     *         Otherwise, the CompletableFuture will complete exceptionally with
     *         a {@link CompletionException}. Common reasons for failure, listed
     *         by the exception reported as the
     *         {@link CompletionException#getCause() cause}, include:
     *
     *         <ul>
     *         <li>{@link PermissionsException} &ndash; if the calling session
     *         does not have {@code MODIFY_SESSION} permission and
     *         {@code SELECT_TOPIC} permission for the selector expression;
     *
     *         <li>{@link NoSuchSessionException} &ndash; if there is no session
     *         with the given {@code sessionId} on the current server or on any
     *         other servers in a clustered environment;
     *
     *         <li>{@link SessionClosedException} &ndash; if the session is
     *         closed.
     *
     *         <li>{@link ClusterRoutingException} &ndash; if the operation
     *         failed due to a transient cluster error;
     *         </ul>
     * @since 6.12
     */
    CompletableFuture<?> unsubscribe(
        SessionId sessionId,
        TopicSelector topics,
        String scope);

    /**
     * Subscribe sessions that satisfy a given session filter to topics.
     * <p>
     * This is equivalent to calling
     * {@link #subscribeByFilter(String, String, String)} specifying the
     * {@link Topics#DEFAULT_SELECTION_SCOPE default selection scope}.
     *
     * @see Topics Topic Selection Scopes
     *
     * @since 6.0
     */
    default CompletableFuture<SubscriptionByFilterResult> subscribeByFilter(
        String filter,
        String topics) {
        return subscribeByFilter(
            filter,
            topics,
            Topics.DEFAULT_SELECTION_SCOPE);
    }

    /**
     * Subscribe sessions that satisfy a given session filter to topics. In a
     * clustered environment the filter will be applied to all servers in the
     * cluster.
     *
     * <P>
     * This is equivalent to calling
     * {@link #subscribeByFilter(String, TopicSelector, String)} with a selector
     * parsed using {@link TopicSelectors#parse(String)}.
     *
     * @param filter the session filter expression
     *
     * @param topics the topics to subscribe to specified as a
     *        {@link TopicSelector} expression
     *
     * @param scope specifies the scope of the selection. See {@link Topics
     *        Topic Selection Scopes}
     *
     * @throws IllegalArgumentException if {@code topics} is not a valid
     *         selector expression
     *
     * @return a CompletableFuture that completes when the server has processed
     *         the subscription request.
     *
     *         <p>
     *         If the subscription was accepted, the CompletableFuture will
     *         complete successfully with a {@link SubscriptionByFilterResult}.
     *
     *         <p>
     *         Otherwise, the CompletableFuture will complete exceptionally with
     *         a {@link CompletionException}. Common reasons for failure, listed
     *         by the exception reported as the
     *         {@link CompletionException#getCause() cause}, include:
     *
     *         <ul>
     *         <li>{@link InvalidFilterException} &ndash; if the filter is
     *         invalid;
     *
     *         <li>{@link PermissionsException} &ndash; if the calling session
     *         does not have {@code MODIFY_SESSION} permission as well as
     *         {@code SELECT_TOPIC} permission for the selector expression;
     *
     *         <li>{@link SessionClosedException} &ndash; if the session is
     *         closed.
     *
     *         <li>{@link ClusterRoutingException} &ndash; if the operation
     *         failed due to a transient cluster error;
     *         </ul>
     *
     * @since 6.12
     */
    CompletableFuture<SubscriptionByFilterResult> subscribeByFilter(
        String filter,
        String topics,
        String scope)
        throws IllegalArgumentException;

    /**
     * Subscribe sessions that satisfy a given session filter to topics.
     * <p>
     * This is equivalent to calling
     * {@link #subscribeByFilter(String, TopicSelector, String)} specifying the
     * {@link Topics#DEFAULT_SELECTION_SCOPE default selection scope}.
     *
     * @see Topics Topic Selection Scopes
     *
     * @since 6.0
     */
    default CompletableFuture<SubscriptionByFilterResult> subscribeByFilter(
        String filter,
        TopicSelector topics) {
        return subscribeByFilter(
            filter,
            topics,
            Topics.DEFAULT_SELECTION_SCOPE);
    }

    /**
     * Subscribe sessions that satisfy a given session filter to topics. In a
     * clustered environment the filter will be applied to all servers in the
     * cluster.
     *
     * <p>
     * For each session that matches the filter, new subscriptions will be
     * established for existing topics that match the provided topic selector
     * and for which the sessions has {@code READ_TOPIC} permission. The topic
     * selector will be added to the topic selections of the subscribed session,
     * and re-evaluated when new topics are added or the session's security
     * roles change.
     *
     * <p>
     * A session that does not have {@code SELECT_TOPIC} permission for a topic
     * cannot subscribe directly, but can be subscribed indirectly using this
     * method.
     *
     * @param filter the session filter expression
     *
     * @param topics the topics to subscribe to specified as a
     *        {@link TopicSelector} expression
     *
     * @param scope specifies the scope of the selection. See {@link Topics
     *        Topic Selection Scopes}
     *
     * @return a CompletableFuture that completes when the server has processed
     *         the subscription request.
     *
     *         <p>
     *         If the subscription was accepted, the CompletableFuture will
     *         complete successfully with a {@link SubscriptionByFilterResult}.
     *
     *         <p>
     *         Otherwise, the CompletableFuture will complete exceptionally with
     *         a {@link CompletionException}. Common reasons for failure, listed
     *         by the exception reported as the
     *         {@link CompletionException#getCause() cause}, include:
     *
     *         <ul>
     *         <li>{@link InvalidFilterException} &ndash; if the filter is
     *         invalid;
     *
     *         <li>{@link PermissionsException} &ndash; if the calling session
     *         does not have {@code MODIFY_SESSION} permission as well as
     *         {@code SELECT_TOPIC} permission for the selector expression;
     *
     *         <li>{@link SessionClosedException} &ndash; if the session is
     *         closed.
     *
     *         <li>{@link ClusterRoutingException} &ndash; if the operation
     *         failed due to a transient cluster error;
     *         </ul>
     *
     * @since 6.12
     */
    CompletableFuture<SubscriptionByFilterResult> subscribeByFilter(
        String filter,
        TopicSelector topics,
        String scope);

    /**
     * Unsubscribe sessions that satisfy a given session filter from topics.
     * <p>
     * This is equivalent to calling
     * {@link #unsubscribeByFilter(String, String, String)} specifying the
     * {@link Topics#DEFAULT_SELECTION_SCOPE default selection scope}.
     *
     * @see Topics Topic Selection Scopes
     *
     * @since 6.0
     */
    default CompletableFuture<SubscriptionByFilterResult> unsubscribeByFilter(
        String filter,
        String topics) {
        return unsubscribeByFilter(
            filter,
            topics,
            Topics.DEFAULT_SELECTION_SCOPE);
    }

    /**
     * Unsubscribe sessions that satisfy a given session filter from topics. In
     * a clustered environment the filter will be applied to all servers in the
     * cluster.
     * <P>
     * This is equivalent to calling
     * {@link #unsubscribeByFilter(String, TopicSelector, String)} with a
     * selector parsed using {@link TopicSelectors#parse(String)}.
     *
     * @param filter the session filter expression
     *
     * @param topics the topics to unsubscribe from specified as a
     *        {@link TopicSelector} expression
     *
     * @param scope specifies the scope of the selection. See {@link Topics
     *        Topic Selection Scopes}
     *
     * @throws IllegalArgumentException if {@code topics} is not a valid
     *         selector expression
     *
     * @return a CompletableFuture that completes when the server has processed
     *         the unsubscription request.
     *
     *         <p>
     *         If the unsubscription was accepted, the CompletableFuture will
     *         complete successfully with a {@link SubscriptionByFilterResult}.
     *
     *         <p>
     *         Otherwise, the CompletableFuture will complete exceptionally with
     *         a {@link CompletionException}. Common reasons for failure, listed
     *         by the exception reported as the
     *         {@link CompletionException#getCause() cause}, include:
     *
     *         <ul>
     *         <li>{@link InvalidFilterException} &ndash; if the filter is
     *         invalid;
     *
     *         <li>{@link PermissionsException} &ndash; if the calling session
     *         does not have {@code MODIFY_SESSION} permission as well as
     *         {@code SELECT_TOPIC} permission for the selector expression;
     *
     *         <li>{@link SessionClosedException} &ndash; if the session is
     *         closed.
     *
     *         <li>{@link ClusterRoutingException} &ndash; if the operation
     *         failed due to a transient cluster error;
     *         </ul>
     *
     * @since 6.12
     */
    CompletableFuture<SubscriptionByFilterResult> unsubscribeByFilter(
        String filter,
        String topics,
        String scope)
        throws IllegalArgumentException;

    /**
     * Unsubscribe sessions that satisfy a given session filter from topics.
     * <p>
     * This is equivalent to calling
     * {@link #unsubscribeByFilter(String, TopicSelector, String)} specifying
     * the {@link Topics#DEFAULT_SELECTION_SCOPE default selection scope}.
     *
     * @see Topics Topic Selection Scopes
     *
     * @since 6.0
     */
    default CompletableFuture<SubscriptionByFilterResult> unsubscribeByFilter(
        String filter,
        TopicSelector topics) {
        return unsubscribeByFilter(
            filter,
            topics,
            Topics.DEFAULT_SELECTION_SCOPE);
    }

    /**
     * Unsubscribe sessions that satisfy a given session filter from topics. In
     * a clustered environment the filter will be applied to all servers in the
     * cluster.
     *
     * @param filter the session filter expression
     *
     * @param topics the topics to unsubscribe from specified as a
     *        {@link TopicSelector}
     *
     * @param scope specifies the scope of the selection. See {@link Topics
     *        Topic Selection Scopes}
     *
     * @return a CompletableFuture that completes when the server has processed
     *         the unsubscription request.
     *
     *         <p>
     *         If the unsubscription was accepted, the CompletableFuture will
     *         complete successfully with a {@link SubscriptionByFilterResult}.
     *
     *         <p>
     *         Otherwise, the CompletableFuture will complete exceptionally with
     *         a {@link CompletionException}. Common reasons for failure, listed
     *         by the exception reported as the
     *         {@link CompletionException#getCause() cause}, include:
     *
     *         <ul>
     *         <li>{@link InvalidFilterException} &ndash; if the filter is
     *         invalid;
     *
     *         <li>{@link PermissionsException} &ndash; if the calling session
     *         does not have {@code MODIFY_SESSION} permission as well as
     *         {@code SELECT_TOPIC} permission for the selector expression;
     *
     *         <li>{@link SessionClosedException} &ndash; if the session is
     *         closed.
     *
     *         <li>{@link ClusterRoutingException} &ndash; if the operation
     *         failed due to a transient cluster error;
     *         </ul>
     *
     * @since 6.12
     */
    CompletableFuture<SubscriptionByFilterResult> unsubscribeByFilter(
        String filter,
        TopicSelector topics,
        String scope);

    /**
     * Unsubscribe another session from topics for all topic selection scopes.
     * <p>
     * This is equivalent to calling
     * {@link #unsubscribeAllScopes(SessionId, TopicSelector)} with a selector
     * parsed using {@link TopicSelectors#parse(String)}.
     *
     * @param sessionId identifies the session to unsubscribe
     *
     * @param topics the topics to unsubscribe from
     *
     * @throws IllegalArgumentException if topics is an invalid topic selector
     *         expression
     *
     * @return a CompletableFuture that completes when the server has processed
     *         the unsubscription request.
     *
     *         <p>
     *         If the unsubscription was accepted, the CompletableFuture will
     *         complete successfully. The result type is any rather than Void to
     *         provide forward compatibility with future iterations of this API
     *         that may provide a non-null result with a more specific result
     *         type.
     *
     *         <p>
     *         Otherwise, the CompletableFuture will complete exceptionally with
     *         a {@link CompletionException}. Common reasons for failure, listed
     *         by the exception reported as the
     *         {@link CompletionException#getCause() cause}, include:
     *
     *         <ul>
     *         <li>{@link PermissionsException} &ndash; if the calling session
     *         does not have {@code MODIFY_SESSION} permission and
     *         {@code SELECT_TOPIC} permission for the selector expression;
     *
     *         <li>{@link NoSuchSessionException} &ndash; if there is no session
     *         with the given {@code sessionId} on the current server or on any
     *         other servers in a clustered environment;
     *
     *         <li>{@link SessionClosedException} &ndash; if the session is
     *         closed.
     *
     *         <li>{@link ClusterRoutingException} &ndash; if the operation
     *         failed due to a transient cluster error;
     *         </ul>
     *
     * @since 6.12
     */
    CompletableFuture<?> unsubscribeAllScopes(
        SessionId sessionId,
        String topics)
        throws IllegalArgumentException;

    /**
     * Unsubscribe topics from all topic selection scopes.
     * <P>
     * This can be used at any time whilst connected to reduce the set of topics
     * to which the session is subscribed or negate earlier subscription
     * requests and will apply to all scopes in use.
     *
     * @param sessionId identifies the session to unsubscribe
     *
     * @param topics the topics to unsubscribe from
     *
     * @return a CompletableFuture that completes when the server has processed
     *         the unsubscription request.
     *
     *         <p>
     *         If the unsubscription was accepted, the CompletableFuture will
     *         complete successfully. The result type is any rather than Void to
     *         provide forward compatibility with future iterations of this API
     *         that may provide a non-null result with a more specific result
     *         type.
     *
     *         <p>
     *         Otherwise, the CompletableFuture will complete exceptionally with
     *         a {@link CompletionException}. Common reasons for failure, listed
     *         by the exception reported as the
     *         {@link CompletionException#getCause() cause}, include:
     *
     *         <ul>
     *         <li>{@link PermissionsException} &ndash; if the calling session
     *         does not have {@code MODIFY_SESSION} permission and
     *         {@code SELECT_TOPIC} permission for the selector expression;
     *
     *         <li>{@link NoSuchSessionException} &ndash; if there is no session
     *         with the given {@code sessionId} on the current server or on any
     *         other servers in a clustered environment;
     *
     *         <li>{@link SessionClosedException} &ndash; if the session is
     *         closed.
     *
     *         <li>{@link ClusterRoutingException} &ndash; if the operation
     *         failed due to a transient cluster error;
     *         </ul>
     *
     * @since 6.12
     */
    CompletableFuture<?> unsubscribeAllScopes(
        SessionId sessionId,
        TopicSelector topics);

    /**
     * Unsubscribe sessions that satisfy a given session filter from topics for
     * all scopes. In a clustered environment the filter will be applied to all
     * servers in the cluster.
     * <p>
     * This is equivalent to calling
     * {@link #unsubscribeAllScopesByFilter(String, TopicSelector)} with a
     * selector parsed using {@link TopicSelectors#parse(String)}.
     *
     * @param filter the session filter expression
     *
     * @param topics the topics to unsubscribe from specified as a
     *        {@link TopicSelector} expression
     *
     * @throws IllegalArgumentException if {@code topics} is not a valid
     *         selector expression
     *
     * @return a CompletableFuture that completes when the server has processed
     *         the unsubscription request.
     *
     *         <p>
     *         If the unsubscription was accepted, the CompletableFuture will
     *         complete successfully with a {@link SubscriptionByFilterResult}.
     *
     *         <p>
     *         Otherwise, the CompletableFuture will complete exceptionally with
     *         a {@link CompletionException}. Common reasons for failure, listed
     *         by the exception reported as the
     *         {@link CompletionException#getCause() cause}, include:
     *
     *         <ul>
     *         <li>{@link InvalidFilterException} &ndash; if the filter is
     *         invalid;
     *
     *         <li>{@link PermissionsException} &ndash; if the calling session
     *         does not have {@code MODIFY_SESSION} permission as well as
     *         {@code SELECT_TOPIC} permission for the selector expression;
     *
     *         <li>{@link SessionClosedException} &ndash; if the session is
     *         closed.
     *
     *         <li>{@link ClusterRoutingException} &ndash; if the operation
     *         failed due to a transient cluster error;
     *         </ul>
     *
     * @since 6.12
     */
    CompletableFuture<SubscriptionByFilterResult> unsubscribeAllScopesByFilter(
        String filter,
        String topics)
        throws IllegalArgumentException;

    /**
     * Unsubscribe sessions that satisfy a given session filter from topics for
     * all scopes. In a clustered environment the filter will be applied to all
     * servers in the cluster.
     *
     * @param filter the session filter expression
     *
     * @param topics the topics to unsubscribe from specified as a
     *        {@link TopicSelector}
     *
     * @return a CompletableFuture that completes when the server has processed
     *         the unsubscription request.
     *
     *         <p>
     *         If the unsubscription was accepted, the CompletableFuture will
     *         complete successfully with a {@link SubscriptionByFilterResult}.
     *
     *         <p>
     *         Otherwise, the CompletableFuture will complete exceptionally with
     *         a {@link CompletionException}. Common reasons for failure, listed
     *         by the exception reported as the
     *         {@link CompletionException#getCause() cause}, include:
     *
     *         <ul>
     *         <li>{@link InvalidFilterException} &ndash; if the filter is
     *         invalid;
     *
     *         <li>{@link PermissionsException} &ndash; if the calling session
     *         does not have {@code MODIFY_SESSION} permission as well as
     *         {@code SELECT_TOPIC} permission for the selector expression;
     *
     *         <li>{@link SessionClosedException} &ndash; if the session is
     *         closed.
     *
     *         <li>{@link ClusterRoutingException} &ndash; if the operation
     *         failed due to a transient cluster error;
     *         </ul>
     *
     * @since 6.12
     */
    CompletableFuture<SubscriptionByFilterResult> unsubscribeAllScopesByFilter(
        String filter,
        TopicSelector topics);

    /**
     * Returns a map of the current topic selection state for a specified
     * session, keyed on topic selection scope.
     * <p>
     * Each scope will have an ordered list of selections and/or deselections.
     * <p>
     * The server conflates selections, so if there has been a selection that is
     * later rendered redundant by a deselection it will not be present. A scope
     * that has been used but fully deselected will therefore not be present in
     * the map and therefore no entry will have an empty list.
     * <p>
     * If the session has no current selections the map will be empty.
     *
     * @param sessionId identifies the session to return topic selections for
     *
     * @return a CompletableFuture that completes when a response is received
     *         from the server with the results of the operation.
     *         <p>
     *         If the task completes successfully, the CompletableFuture result
     *         will be a map of an ordered {@link List} of
     *         {@link TopicSelection}s keyed on scope. If the default scope is
     *         in use the key will be {@link Topics#DEFAULT_SELECTION_SCOPE}.
     *         <p>
     *         Otherwise, the CompletableFuture will complete exceptionally with
     *         a {@link CompletionException}. Common reasons for failure, listed
     *         by the exception reported as the
     *         {@link CompletionException#getCause() cause}, include:
     *         <ul>
     *         <li>{@link PermissionsException} &ndash; if the calling session
     *         does not have {@code VIEW_SESSION} permission;
     *
     *         <li>{@link NoSuchSessionException} &ndash; if there is no session
     *         with the given {@code sessionId} on the current server or on any
     *         other servers in a clustered environment;;
     *
     *         <li>{@link SessionClosedException} &ndash; if the current session
     *         is closed.
     *
     *         <li>{@link ClusterRoutingException} &ndash; if the operation
     *         failed due to a transient cluster error;
     *         </ul>
     *
     * @since 6.12
     */
    CompletableFuture<Map<String, List<TopicSelection>>> getTopicSelections(
        SessionId sessionId);

    /**
     * Result used by CompletableFuture variants of
     * {@link SubscriptionControl#subscribeByFilter subscribeByFilter} and
     * {@link SubscriptionControl#unsubscribeByFilter unsubscribeByFilter}.
     *
     * @since 6.0
     */
    interface SubscriptionByFilterResult {

        /**
         * @return number of sessions that satisfied the filter and qualified
         *         for subscription/unsubscription. This will be 0 if no
         *         sessions satisfied the filter. In a clustered environment
         *         this will be the sum of all sessions that matched across the
         *         cluster.
         */
        int numberSelected();
    }
}
