/*******************************************************************************
 * Copyright (c) 2018, 2025 DiffusionData Ltd., All Rights Reserved.
 *
 * Use is subject to licence terms.
 *
 * NOTICE: All information contained herein is, and remains the
 * property of DiffusionData. The intellectual and technical
 * concepts contained herein are proprietary to DiffusionData and
 * may be covered by U.S. and Foreign Patents, patents in process, and
 * are protected by trade secret or copyright law.
 *******************************************************************************/
package com.pushtechnology.diffusion.client.security.authentication;

import java.util.Map;

import com.pushtechnology.diffusion.client.Diffusion;
import com.pushtechnology.diffusion.client.features.Security;
import com.pushtechnology.diffusion.client.session.Session;
import com.pushtechnology.diffusion.client.types.Credentials;

/**
 * An authentication handler that processes authentication requests from the
 * server.
 * <p>
 * Instances can be registered with the server using the
 * {@link com.pushtechnology.diffusion.client.features.control.clients.AuthenticationControl#setAuthenticationHandler(String, com.pushtechnology.diffusion.client.features.control.clients.AuthenticationControl.ControlAuthenticator)
 * AuthenticationControl} feature.
 * <p>
 * The server calls an authenticator when a client application creates a
 * session, or re-authenticates a session, allowing the handler to veto
 * individual requests.
 * <P>
 * Authenticators are configured in precedence order. Authentication will
 * succeed if an authenticator responds by calling {@link Callback#allow()
 * allow()} or {@link Callback#allow(Map) allow(Map)} and authenticators with
 * higher precedence respond by calling {@link Callback#abstain() abstain()}.
 * <p>
 * Authentication will fail if an authenticator responds by calling
 * {@link Callback#deny() deny()} and all higher precedence authenticators
 * respond by calling {@link Callback#abstain() abstain()}.
 * <p>
 * If all authenticators respond by calling {@link Callback#abstain()
 * abstain()}, the request will be denied. Once the outcome is known, the server
 * can choose not to call any remaining authenticators.
 *
 * @author DiffusionData Limited
 * @since 6.2
 */
public interface Authenticator {

    /**
     * Processes an authentication request.
     * <P>
     * This method will be called to authenticate new sessions, and when a
     * session re-authenticates (using {@link Security#reauthenticate
     * reauthenticate}), when the principal for the session can optionally be
     * changed.
     * <P>
     * For each call to {@code authenticate}, the authenticator should respond
     * by calling one of the methods of the provided {@link Callback callback}.
     * The authenticator can return immediately and process the authentication
     * request asynchronously. The authentication will not proceed until a
     * callback method is called.
     * <p>
     * The content of the {@code sessionProperties} parameter depends upon
     * whether the authenticator is being called on initial authentication of a
     * session or as a result of a session re-authenticating using
     * {@link Security#reauthenticate reauthenticate}, as shown below:<br>
     *
     * <table border=1>
     * <tr>
     * <th style="text-align:left;"></th>
     * <th style="text-align:left;">Initial&nbsp;Authentication</th>
     * <th style="text-align:left;">Re-authentication</th>
     * </tr>
     * <tr style="vertical-align:top">
     * <th style="text-align:left;"><b>Fixed&nbsp;Properties</b></th>
     * <td>A full set of fixed session properties as defined in {@link Session}.
     * <p>
     * {@code $Principal} will be the same as the supplied {@code principal}.
     * <p>
     * {@code $Roles} will contain the configured default roles for the
     * principal.</td>
     * <td>A full set of fixed session properties as defined in {@link Session}.
     * <p>
     * {@code $Principal} will be the principal from the previously
     * authenticated session which can differ from the supplied
     * {@code principal}.
     * <p>
     * {@code $Roles} will contain the configured default roles for the new
     * principal.</td>
     * </tr>
     * <tr style="vertical-align:top">
     * <th style="text-align:left;"><b>User-defined&nbsp;Properties</b></th>
     * <td>None</td>
     * <td>Existing user-defined properties</td>
     * </tr>
     * </table>
     * <p>
     * On initial authentication the {@code proposedProperties} parameter will
     * provide any user-defined properties that the client supplied when opening
     * the client session. On re-authentication it will provide any user-defined
     * properties that the client supplied to the {@link Security#reauthenticate
     * reauthenticate} method.
     * <p>
     * The authenticator can choose to call {@link Callback#allow() allow()} to
     * accept the authentication request with default behaviour or
     * {@link Callback#allow(Map) allow(Map)} to accept the authentication
     * request with modifications to the session properties. Alternatively it
     * can call {@link Callback#deny() deny()} to deny the request or
     * {@link Callback#abstain() abstain()} to abstain from authentication, in
     * which case the request will be passed on to the next configured
     * authenticator.
     * <p>
     * If the authenticator calls {@link Callback#allow() allow()} then the
     * resulting session properties for the session will be as follows:
     * <table border=1>
     * <tr>
     * <th style="text-align:left;"></th>
     * <th style="text-align:left;">Initial&nbsp;Authentication</th>
     * <th style="text-align:left;">Re-authentication</th>
     * </tr>
     * <tr style="vertical-align:top">
     * <th style="text-align:left;"><b>Fixed&nbsp;Properties</b></th>
     * <td>As supplied plus those assigned by the server on connection.</td>
     * <td>As supplied but with {@code $Principal} replaced by the supplied
     * {@code principal}.</td>
     * </tr>
     * <tr style="vertical-align:top">
     * <th style="text-align:left;"><b>User-defined&nbsp;Properties</b></th>
     * <td>None</td>
     * <td>None</td>
     * </tr>
     * </table>
     * <p>
     * If the authenticator calls {@link Callback#allow(Map) allow(Map)} then
     * the map can contain values for any fixed properties that can be
     * changed/supplied (see below) and/or all user-defined properties to assign
     * to the session. The user-defined properties can be those proposed by the
     * client or any set of user-defined properties that the authenticator
     * chooses.
     * <h3>Permitted fixed property adjustments</h3>
     * <p>
     * An authenticator can set values for any of the following fixed properties
     * to {@link Callback#allow(Map) allow(Map)}:
     * <ul>
     * <li>{@link Session#COUNTRY $Country}
     * <li>{@link Session#EXPIRY_TIME $ExpiryTime}
     * <li>{@link Session#LANGUAGE $Language}
     * <li>{@link Session#LATITUDE $Latitude}
     * <li>{@link Session#LONGITUDE $Longitude}
     * <li>{@link Session#PRINCIPAL $Principal}
     * <li>{@link Session#ROLES $Roles}
     * </ul>
     * An authenticator can only set values of these fixed properties. Other
     * fixed properties provided by the authenticator will be ignored. If the
     * authenticator does not set a fixed property, the value will be as
     * supplied, or as assigned by the server.
     * <h3>Handling the {@code $Roles} property</h3>
     * <p>
     * The {@code $Roles} property is formatted as a quoted list of strings. To
     * make the handling of this property value easier there are methods on the
     * {@link Diffusion} singleton. Using these methods an authenticator can
     * adjust roles as follows:
     *
     * <pre>
     * final Set&lt;String&gt; roles =
     *     Diffusion.stringToRoles(sessionProperties.get(Session.ROLES));
     * ... changes roles are required ...
     * sessionProperties.put(Session.ROLES, Diffusion.rolesToString(roles));
     * callback.allow(sessionProperties);
     * </pre>
     *
     * @param principal the name of the proposed principal, or
     *        {@link Session#ANONYMOUS ANONYMOUS} if none was supplied
     *
     * @param credentials authenticating the principal; for example, a password
     *
     * @param sessionProperties supplies the currently known session properties
     *        for the client. On initial authentication this will be the known
     *        fixed property values. If the session is re-authenticating using
     *        {@link Security#reauthenticate reauthenticate}, this will be the
     *        full set of fixed property values plus any user-defined properties
     *        from the existing session. Modifications made to this map by the
     *        authenticator are ignored unless the map is passed to the
     *        {@link Callback#allow(Map) allow} method.
     *
     * @param proposedProperties provides any user-defined properties proposed
     *        by the client. The authenticator can choose to pass on these
     *        properties as they are, veto some properties, or add more
     *        properties before passing them to the result. The client can
     *        provide arbitrary keys and values. Supplied properties should be
     *        checked and filtered/constrained to ensure they do not affect the
     *        integrity of the application. Authenticators should not blindly
     *        pass proposed properties to the {@link Callback#allow(Map) allow}
     *        method. Modifications made to this map by the authenticator are
     *        ignored unless the map is passed to the {@link Callback#allow(Map)
     *        allow} method.
     *
     * @param callback single use callback
     */
    void authenticate(
        String principal,
        Credentials credentials,
        Map<String, String> sessionProperties,
        Map<String, String> proposedProperties,
        Callback callback);

    /**
     * Single-use callback provided to the {@link Authenticator#authenticate
     * authenticate} call.
     */
    interface Callback {

        /**
         * Authentication passed - allow the authentication request with
         * modified fixed session properties and/or specified user-defined
         * properties.
         * <p>
         * A map comprising fixed properties to be changed and/or user-defined
         * properties to be added to the session must be passed to this method.
         * Only user-defined properties that are explicitly passed via this map
         * will be added to the session.
         * <p>
         * For convenience, either the {@code sessionProperties} map or the
         * {@code proposedProperties} map passed to the
         * {@link Authenticator#authenticate authenticate} method can be
         * modified in order to create a suitable map to be passed to this
         * method. For example, if fixed properties only are to be changed then
         * the relevant properties in the {@code sessionProperties} map can be
         * altered before passing that map to this method (properties that are
         * not changed will be ignored). If no fixed properties are to be
         * changed but user-defined properties have been proposed and are
         * acceptable then the {@code proposedProperties} map can be passed to
         * this method. Properties can be added to or removed from the
         * {@code proposedProperties} map before passing to this method.
         *
         * @param properties specifies user defined properties and/or any
         *        changes to fixed session properties. User defined properties
         *        can include those from the proposed properties or additional
         *        properties. Proposed properties not included in this map will
         *        not be added to the session. See
         *        {@link Authenticator#authenticate authenticate}.
         *
         * @throws IllegalStateException if another method has already been
         *         invoked on this callback
         */
        void allow(Map<String, String> properties);

        /**
         * Authentication passed - allow the authentication request with fixed
         * properties as supplied but no user-defined properties.
         * <p>
         * If this method is used, any proposed properties passed to the
         * authenticator will not be added to the session.
         *
         * @throws IllegalStateException if another method has already been
         *         invoked on this callback
         */
        void allow();

        /**
         * The authentication has neither passed nor failed.
         *
         * @throws IllegalStateException if another method has already been
         *         invoked on this callback
         */
        void abstain();

        /**
         * Authentication failed - deny the authentication request.
         *
         * @throws IllegalStateException if another method has already been
         *         invoked on this callback
         */
        void deny();
    }
}
