/*******************************************************************************
 * Copyright (c) 2014, 2025 DiffusionData Ltd., All Rights Reserved.
 *
 * Use is subject to licence terms.
 *
 * NOTICE: All information contained herein is, and remains the
 * property of DiffusionData. The intellectual and technical
 * concepts contained herein are proprietary to DiffusionData and
 * may be covered by U.S. and Foreign Patents, patents in process, and
 * are protected by trade secret or copyright law.
 *******************************************************************************/
package com.pushtechnology.diffusion.client.session;

import java.net.SocketAddress;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;

import javax.net.ssl.SSLContext;

import com.pushtechnology.diffusion.client.Diffusion;
import com.pushtechnology.diffusion.client.features.Messaging;
import com.pushtechnology.diffusion.client.session.proxy.HTTPProxyAuthentication;
import com.pushtechnology.diffusion.client.session.proxy.ProxyAuthenticationFactory;
import com.pushtechnology.diffusion.client.session.reconnect.ReconnectionStrategy;
import com.pushtechnology.diffusion.client.session.retry.RetryStrategy;
import com.pushtechnology.diffusion.client.types.Credentials;
import com.pushtechnology.diffusion.client.types.Credentials.Type;

/**
 * Factory for client {@link Session sessions}.
 * <P>
 * Each instance is immutable, and has a number of session attributes that
 * determine how sessions should be established. An initial instance of this
 * factory with default values for all attributes can be obtained from
 * {@link Diffusion#sessions()}. Methods such as {@link #inputBufferSize(int)}
 * or {@link #maximumMessageSize(int)} that allow attributes to be changed do
 * not modify this instance. Each method returns a copy of this factory, with a
 * single changed attribute.
 *
 * <h3>Establishing a session</h3>
 * <P>
 * A new session can be created using {@link #openAsync(String)} specifying a
 * URL that identifies the server. This is a non-blocking method that returns
 * immediately, and provides the result asynchronously using a
 * {@link CompletableFuture}.
 * <p>
 * {@link #open(String)} is a blocking alternative to {@code openAsync} that
 * waits for a session to be established before returning.
 * <P>
 * The overloaded methods {@link #openAsync()} or {@link #open()} use the
 * session factory attributes to locate a server rather than a URL. The server
 * is identified by the {@link #serverHost(String)} and
 * {@link #serverPort(int)}, {@link #requestPath(String)},
 * {@link #secureTransport(boolean)} attributes. The
 * {@link #transports(SessionAttributes.Transport...)} attributes allows the
 * configuration of multiple transports for cascading.
 *
 * <p>
 * If a URL is specified, it takes precedence over the {@code serverHost},
 * {@code serverPort}, {@code requestPath}, and {@code secureTransport} session
 * factory attributes. A URL can only be used to configure a single transport
 * and cannot be used to configure cascading, so the {@code transports}
 * attribute is ignored.
 *
 * <h3>URL format</h3>
 * <P>
 * URLs should take the form <em>scheme://host:port</em>, where <em>scheme</em>
 * is chosen from the following table and determines the transport protocol used
 * to send Diffusion messages.
 *
 * <table>
 * <tr>
 * <th>Scheme</th>
 * <th>Transport Protocol</th>
 * </tr>
 * <tr>
 * <td>{@code ws}</td>
 * <td>WebSocket. See <a href="https://tools.ietf.org/html/rfc6455">RFC 6455</a>
 * .</td>
 * </tr>
 * <tr>
 * <td>{@code wss}</td>
 * <td>WebSocket over TLS.
 * </tr>
 * <tr>
 * <td>{@code xhr}</td>
 * <td>HTTP polling.</td>
 * </tr>
 * <tr>
 * <td>{@code xhrs}</td>
 * <td>HTTP polling over TLS.</td>
 * </tr>
 *
 * </table>
 *
 * <p>
 * We recommend using the WebSocket protocol options ({@code ws} or
 * {@code wss}).
 *
 * <p>
 * TLS is <em>Transport Layer Security</em>, commonly known as SSL. TLS-based
 * protocols use cryptography to provide transport-level privacy,
 * authentication, and integrity, and protects against network-sniffing and
 * man-in-the-middle attacks. We recommend using the TLS variants for all
 * communication. For a typical application, you should only consider
 * <em>not</em> using TLS for unauthenticated ("anonymous") client sessions.
 *
 * @author DiffusionData Limited
 * @since 5.0
 */
public interface SessionFactory {

    /**
     * Nominates a listener for session events from a session.
     * <P>
     * If not specified, there will be no session listener.
     *
     * @param listener specifies a listener to be used for subsequent sessions
     *        that are created. Null may be specified to remove any previously
     *        specified listener
     *
     * @return a new immutable instance of the factory with the listener changed
     */
    SessionFactory listener(Session.Listener listener);

    /**
     * Nominates an error handler for a session.
     * <P>
     * If not specified, an instance of {@link Session.ErrorHandler.Default}
     * will be used.
     *
     * @param errorHandler specifies the error handler to use for subsequent
     *        sessions that are created. Null may be specified to remove any
     *        previously specified handler and effectively revert to using the
     *        default handler
     *
     * @return a new immutable instance of the factory with the error handler
     *         changed
     */
    SessionFactory errorHandler(Session.ErrorHandler errorHandler);

    /**
     * Sets the security principal.
     * <P>
     * By default this will be {@link Session#ANONYMOUS}.
     *
     * @param principal the principal
     *
     * @return a new immutable instance of the factory with the principal
     *         changed
     */
    SessionFactory principal(String principal);

    /**
     * Set credentials to {@link Type#NONE}.
     * <P>
     * This is the default.
     *
     * @return a new immutable instance of the factory with the credentials
     *         changed
     */
    SessionFactory noCredentials();

    /**
     * Set credentials.
     * <P>
     * The default is {@link Type#NONE}.
     *
     * @param credentials the credentials
     *
     * @return a new immutable instance of the factory with the credentials
     *         changed
     */
    SessionFactory credentials(Credentials credentials);

    /**
     * Set credentials of type {@link Type#PLAIN_PASSWORD}.
     * <P>
     * The default is {@link Type#NONE}.
     *
     * @param password the password
     *
     * @return a new immutable instance of the factory with the credentials
     *         changed
     */
    SessionFactory password(String password);

    /**
     * Set credentials of type {@link Type#CUSTOM}.
     * <P>
     * The default is {@link Type#NONE}.
     *
     * @param bytes the credentials
     *
     * @return a new immutable instance of the factory with the credentials
     *         changed
     */
    SessionFactory customCredentials(byte[] bytes);

    /**
     * Sets the connection timeout.
     *
     * <p>
     * This constrains the time taken to establish an initial connection to the
     * server. The {@link #writeTimeout(int) write timeout} constrains the time
     * to send the connection request. The server is responsible for limiting
     * the overall time taken to complete a connection once it has received the
     * request.
     *
     * <P>
     * The default is {@link SessionAttributes#DEFAULT_CONNECTION_TIMEOUT}.
     *
     * @param timeout the default connection timeout in milliseconds. If this
     *        exceeds one hour (3600000ms) a warning will be logged and the
     *        time-out will be set to one hour.
     *
     * @return a new immutable instance of the factory with the connection
     *         timeout changed
     *
     * @throws IllegalArgumentException if an illegal timeout value is supplied
     */
    SessionFactory connectionTimeout(int timeout)
        throws IllegalArgumentException;

    /**
     * Disable reconnection. Sessions will not attempt to reconnect in the event
     * of unexpected connection loss.
     * <P>
     * This is equivalent to calling {@link SessionFactory#reconnectionTimeout}
     * with a timeout of <code>0</code>.
     *
     * @return a new immutable instance of the factory with reconnection
     *         disabled
     *
     * @since 5.5
     */
    SessionFactory noReconnection();

    /**
     * Sets the initial retry strategy.
     * <P>
     * The strategy will determine whether a failure to open a session due to a
     * {@link SessionEstablishmentTransientException} should be retried and if
     * so, at what interval and for how many attempts.
     * <p>
     * If no initial retry strategy is set there will be no attempt to retry
     * after such a failure.
     *
     * @param strategy the retry strategy to use
     *
     * @return a new immutable instance of the factory with the initial retry
     *         strategy changed
     *
     * @since 6.9
     */
    SessionFactory initialRetryStrategy(RetryStrategy strategy);

    /**
     * Sets the reconnection timeout. This determines the duration that sessions
     * will attempt to reconnect for, in the event of unexpected connection
     * loss.
     * <P>
     * The default is {@link SessionAttributes#DEFAULT_RECONNECTION_TIMEOUT}.
     * <P>
     * A timeout value of <code>0</code> or less is equivalent to to calling
     * {@link SessionFactory#noReconnection}.
     *
     * @param timeout the timeout duration to use when attempting to reconnect,
     *        in milliseconds
     *
     * @return a new immutable instance of the factory with the reconnection
     *         timeout changed
     *
     * @throws IllegalArgumentException if an illegal timeout value is supplied
     *
     * @since 5.5
     */
    SessionFactory reconnectionTimeout(int timeout)
        throws IllegalArgumentException;

    /**
     * Sets the reconnection strategy. If reconnection is enabled, this strategy
     * will be called in the event of an unexpected connection loss.
     * <P>
     * If the reconnection timeout has been set, but not a strategy, sessions
     * will use a default strategy that attempts to reconnect on a regular basis
     * of 5 seconds until the reconnection timeout is exceeded.
     *
     * @param strategy the reconnection strategy to use
     *
     * @return a new immutable instance of the factory with the reconnection
     *         strategy changed
     *
     * @since 5.5
     */
    SessionFactory reconnectionStrategy(ReconnectionStrategy strategy);

    /**
     * Sets the input buffer size for socket connection buffers and message
     * receiving buffers.
     * <P>
     * This controls both how much pending data the socket can accommodate and
     * how much available data will be read by the input tasks. In general
     * setting both this and {@link #outputBufferSize(int)} to the same
     * reasonable size will result in higher network throughput. The cost of the
     * receiving buffer is only incurred when actually reading data for
     * processing. The cost of the socket buffer is always incurred and may be
     * modified by the operating system.
     * <P>
     * The default is {@link SessionAttributes#DEFAULT_INPUT_BUFFER_SIZE}.
     *
     * @param size input buffer size in bytes. This may not be less than
     *        {@link SessionAttributes#MAXIMUM_MESSAGE_SIZE_MIN}.
     *
     * @return a new immutable instance of the factory with the input buffer
     *         size changed
     *
     * @throws IllegalArgumentException if the input buffer size is invalid
     */
    SessionFactory inputBufferSize(int size) throws IllegalArgumentException;

    /**
     * Sets the output buffer size for socket connection buffers and message
     * sending buffers.
     * <P>
     * This controls both how much pending data the socket can accommodate and
     * how much data will be batched by sending messages. In general setting
     * both this and {@link #inputBufferSize(int)} to the same reasonable size
     * will result in higher network throughput. The cost of the sending buffer
     * is only incurred when actually queuing data for output. The cost of the
     * socket buffer is always incurred and may be modified by the operating
     * system.
     * <P>
     * The default is {@link SessionAttributes#DEFAULT_OUTPUT_BUFFER_SIZE}.
     *
     * @param size output buffer size in bytes.
     *
     * @return a new immutable instance of the factory with the output buffer
     *         size changed
     *
     * @throws IllegalArgumentException if the output buffer size is invalid
     */
    SessionFactory outputBufferSize(int size) throws IllegalArgumentException;

    /**
     * Set the optional local socket address, used prior to connection.
     * <P>
     * By default no local socket address is used.
     *
     * @param address the local socket address. Setting this to null effectively
     *        removes any previous setting
     *
     * @return a new immutable instance of the factory with the local socket
     *         address changed
     */
    SessionFactory localSocketAddress(SocketAddress address);

    /**
     * Sets the SSL Context if a secure connection is required.
     * <P>
     * If not explicitly supplied, the {@link SSLContext#getDefault() default
     * context} will be used.
     * <P>
     * Setting the SSL Context will not enable transport layer security. It must
     * also be specified by the URL or {@link #secureTransport(boolean)}.
     *
     * @param context the SSL Context to use when making the connection
     *
     * @return a new immutable instance of the factory with the SSL context
     *         changed
     */
    SessionFactory sslContext(SSLContext context);

    /**
     * Sets the write timeout value for blocking writes.
     * <P>
     * Blocking writes are only used for the initial connection request. If it
     * is not possible to complete the write within this time, the connection is
     * closed.
     * <P>
     * The default is {@link SessionAttributes#DEFAULT_WRITE_TIMEOUT}.
     *
     * @param timeout the write timeout in milliseconds. If this exceeds one
     *        hour (3600000ms) a warning will be logged and the time-out will be
     *        set to one hour.
     *
     * @return a new immutable instance of the factory with the write timeout
     *         changed
     *
     * @throws IllegalArgumentException if {@code timeout} is invalid
     */
    SessionFactory writeTimeout(int timeout) throws IllegalArgumentException;

    /**
     * Set the maximum message size.
     * <P>
     * This constrains the size of messages that will be accepted from the
     * server and thus the size of any content that can be received. The limit
     * is simply to protect against unexpectedly large messages.
     * <P>
     * By default, the size of received messages is unconstrained – see
     * {@link SessionAttributes#DEFAULT_MAXIMUM_MESSAGE_SIZE}.
     *
     * @param size the maximum message size in bytes. This must not be less than
     *        {@link SessionAttributes#MAXIMUM_MESSAGE_SIZE_MIN}
     *
     * @return a new immutable instance of the factory with the maximum message
     *         size changed
     *
     * @throws IllegalArgumentException if the specified {@code size} is invalid
     */
    SessionFactory maximumMessageSize(int size) throws IllegalArgumentException;

    /**
     * Set the address of the HTTP Proxy that should be used to make connections
     * to the server.
     * <P>
     * This allows connection to a server using HTTP CONNECT tunnelling through
     * the specified proxy.
     *
     * @param host the host name of the HTTP proxy
     *
     * @param port the port of the HTTP proxy
     *
     * @return a new immutable instance of the factory that will use the
     *         provided HTTP proxy
     *
     * @throws IllegalArgumentException if the specified {@code host} or
     *         {@code port} are invalid
     * @see #httpProxy(String, int, HTTPProxyAuthentication)
     */
    SessionFactory httpProxy(String host, int port)
        throws IllegalArgumentException;

    /**
     * A variant of {@link #httpProxy(String, int)} that provides authentication
     * details to the HTTP proxy.
     * <P>
     * This allows connection to a server using HTTP CONNECT tunnelling through
     * the specified proxy, with authentication.
     *
     * <p>
     * The {@code authentication} parameter only affects communication with the
     * proxy, it is not passed on to the server.
     * {@link Diffusion#proxyAuthentication()} and
     * {@link ProxyAuthenticationFactory} can be used to create a suitable
     * instance. For example, to specify HTTP BASIC authentication with a user
     * name <em>joe</em> and password <em>s3cret</em>:
     * </p>
     *
     * <pre>
     * HTTPProxyAuthentication authentication =
     *   Diffusion.proxyAuthentication().basic(&quot;joe&quot;, &quot;s3cret&quot;)
     * </pre>
     *
     * <p>
     * The {@link #principal(String)} and {@link #credentials(Credentials)}
     * methods control authentication with the server and do not affect proxy
     * authentication.
     *
     * @param host the host name of the HTTP proxy
     *
     * @param port the port of the HTTP proxy
     *
     * @param authentication the authentication details
     *
     * @return a new immutable instance of the factory that will use the
     *         provided HTTP proxy and proxy authentication details
     *
     * @throws IllegalArgumentException if the specified {@code host} or
     *         {@code port} are invalid
     */
    SessionFactory httpProxy(
        String host,
        int port,
        HTTPProxyAuthentication authentication)
        throws IllegalArgumentException;

    /**
     * Set the host name of the server to connect the session to.
     * <P>
     * This value is only used if a URL is not provided when opening a session.
     *
     * @param host the host name of the server
     * @return a new immutable instance of the factory that will use the
     *         provided host
     * @throws IllegalArgumentException if the specified {@code host} is invalid
     * @since 5.8
     */
    SessionFactory serverHost(String host) throws IllegalArgumentException;

    /**
     * Set the port of the server to connect the session to.
     * <P>
     * This value is only used if a URL is not provided when opening a session.
     * If the port is not set using this method or a URL, the port will be
     * inferred based on the transport and security configuration.
     * <P>
     * The provided value must be within the range used for port numbers.
     *
     * @param port the port of the server
     * @return a new immutable instance of the factory that will use the
     *         provided port
     * @throws IllegalArgumentException if the specified {@code port} is invalid
     * @since 5.8
     */
    SessionFactory serverPort(int port) throws IllegalArgumentException;

    /**
     * Set the transports to use to connect the session to the server.
     * <P>
     * When a session is opened, it will try to establish a connection with each
     * transport in the list until successful. If a connection cannot be
     * established using any of the transports, the session will be closed.
     * <P>
     * This value is only used if a URL is not provided when opening a session.
     * <P>
     * The WebSocket transport will be used by default.
     *
     * @param transports the transports to use to connect the session to the
     *        server. The provided transports may not be null, contain null or
     *        be an empty array.
     * @return a new immutable instance of the factory that will use the
     *         provided transports
     * @throws IllegalArgumentException if the specified {@code transports} are
     *         invalid
     * @since 5.8
     */
    SessionFactory transports(SessionAttributes.Transport... transports)
        throws IllegalArgumentException;

    /**
     * Set if a secure transport should be used.
     * <P>
     * This value is only used if a URL is not provided when opening a session.
     * Enables or disables the use of transport layer security. TLS is disabled
     * by default because in most cases to correctly enable TLS an
     * {@link SSLContext} must also be provided using
     * {@link #sslContext(SSLContext)}.
     *
     * @param secureTransport If transport layer security is enabled
     * @return a new immutable instance of the factory that will use the
     *         provided setting
     * @since 5.8
     */
    SessionFactory secureTransport(boolean secureTransport);

    /**
     * Set the path used for HTTP requests.
     * <P>
     * This value is only used if a URL is not provided when opening a session.
     * The path must start with a '/' and end with '/diffusion'.
     * <P>
     * The default is {@link SessionAttributes#DEFAULT_REQUEST_PATH}.
     *
     * @param requestPath the path used for HTTP requests
     * @return a new immutable instance of the factory that will use the
     *         provided path
     * @throws IllegalArgumentException if the specified {@code requestPath} is
     *         invalid
     * @since 5.8
     */
    SessionFactory requestPath(String requestPath)
        throws IllegalArgumentException;

    /**
     * Set the recovery buffer size.
     *
     * <p>
     * If the server is configured to support reconnection, a session
     * established with a non-zero
     * {@link SessionAttributes#getReconnectionTimeout() reconnection} time
     * retains a buffer of sent messages. If the session disconnects and
     * reconnects, this buffer is used to re-send messages that the server has
     * not received.
     *
     * <p>
     * The default is {@link SessionAttributes#DEFAULT_RECOVERY_BUFFER_SIZE}.
     * Higher values increase the chance of successful reconnection, but
     * increase the per-session memory footprint.
     *
     * @param size the recovery buffer size in messages; can be 0
     * @return a new immutable instance of the factory that will use the
     *         provided recovery buffer size
     * @throws IllegalArgumentException if the specified {@code size} is
     *         negative
     * @since 5.8
     */
    SessionFactory recoveryBufferSize(int size) throws IllegalArgumentException;

    /**
     * Sets the maximum size of the outbound message queue for the connection.
     * <P>
     * The outbound message queue should be large enough to accommodate all the
     * messages sent to the server. This would include topic updates, messages
     * sent using {@link Messaging}, and service requests such as registering a
     * handler.
     * <P>
     * It may be necessary to increase this value for applications that send
     * messages in bursts, or continue to send messages when a session is
     * disconnected and reconnecting. Larger values allow more messages to be
     * queued, and increase the memory footprint of the session.
     * <P>
     * If the outbound message queue fills, sending a message will cause the
     * session to close with an error.
     * <P>
     * The default is {@link SessionAttributes#DEFAULT_MAXIMUM_QUEUE_SIZE}.
     *
     * @param size a positive integer indicating the maximum number of messages
     *        that may be queued
     *
     * @return a new immutable instance of the factory that will use the
     *         provided maximum queue size
     *
     * @throws IllegalArgumentException if the specified size is not a positive
     *         integer
     *
     * @since 5.9
     */
    SessionFactory maximumQueueSize(int size) throws IllegalArgumentException;

    /**
     * Sets a user-defined session property value.
     * <p>
     * Supplied session properties will be provided to the server when a session
     * is created using this session factory. The supplied properties will be
     * validated during authentication and may be discarded or changed.
     * <p>
     * The specified property will be added to any existing properties set for
     * this session factory.
     * <p>
     * For details of how session properties are used see {@link Session}.
     *
     * @param key the property key
     *
     * @param value the property value
     *
     * @return a new immutable instance of the factory with the named property
     *         set. If the property was already set it will be overwritten.
     *
     * @see #properties(Map)
     *
     * @since 6.2
     */
    SessionFactory property(String key, String value);

    /**
     * Sets user-defined session property values.
     * <p>
     * Supplied session properties will be provided to the server when a session
     * is created using this session factory. The supplied properties will be
     * validated during authentication and may be discarded or changed.
     * <p>
     * The specified properties will be added to any existing properties set for
     * this session factory. If any of the keys have been previously declared
     * then they will be overwritten with the new values.
     * <p>
     * For details of how session properties are used see {@link Session}.
     *
     * @param properties a map of user-defined session properties
     *
     * @return a new immutable instance of the factory with the supplied
     *         properties set
     *
     * @see #property(String, String)
     *
     * @since 6.2
     */
    SessionFactory properties(Map<String, String> properties);

    /**
     * Open a connection to a server and request a new session.
     *
     * <p>
     * The server location is configured using the {@link #serverHost(String)},
     * {@link #serverPort(int)},
     * {@link #transports(SessionAttributes.Transport...)},
     * {@link #secureTransport(boolean)} and {@link #requestPath(String)}
     * methods.
     *
     * <p>
     * This method is non-blocking. The result is provided asynchronously by the
     * returned {@link CompletableFuture}.
     *
     * @return a CompletableFuture that completes when a response is received
     *         from the server.
     *
     *         <p>
     *         If a session was established, the CompletableFuture will complete
     *         successfully with a {@link Session} value.
     *
     *         <p>
     *         Otherwise, the CompletableFuture will complete exceptionally with
     *         a {@link CompletionException}. Common reasons for failure, listed
     *         by the exception reported as the
     *         {@link CompletionException#getCause() cause}, include:
     *
     *         <ul>
     *         <li>{@link IllegalStateException} &ndash; if any of the session
     *         attributes are found to be inconsistent. For example, if the
     *         default SSL context could not be loaded;
     *
     *         <li>{@link SessionEstablishmentException} &ndash; if an initial
     *         connection could not be established;
     *
     *         <li>{@link AuthenticationException} &ndash;if the client is
     *         insufficiently authorized to start the session, based on the
     *         supplied client credentials.
     *         </ul>
     * @since 6.0
     * @see #open()
     */
    CompletableFuture<Session> openAsync();

    /**
     * Open a connection to a server and request a new session.
     *
     * <p>
     * This method is a variant of {@link #openAsync()} that uses a URL to
     * specify the server location, ignoring the {@link #serverHost(String)
     * serverHost} and {@link #serverPort(int) serverPort},
     * {@link #requestPath(String) requestPath},
     * {@link #secureTransport(boolean) secureTransport} and
     * {@link #transports(SessionAttributes.Transport...) transports} session
     * factory attributes.
     *
     * @param url the server location, see the {@link SessionFactory class
     *        documentation}
     *
     * @return a CompletableFuture that completes when a response is received
     *         from the server.
     *
     *         <p>
     *         If a session was established, the CompletableFuture will complete
     *         successfully with a {@link Session} value.
     *
     *         <p>
     *         Otherwise, the CompletableFuture will complete exceptionally with
     *         a {@link CompletionException}. Common reasons for failure, listed
     *         by the exception reported as the
     *         {@link CompletionException#getCause() cause}, include:
     *
     *         <ul>
     *         <li>{@link IllegalArgumentException} &ndash; if {@code url} is
     *         invalid;
     *
     *         <li>{@link IllegalStateException} &ndash; if any of the session
     *         attributes are found to be inconsistent. For example, if the
     *         default SSL context could not be loaded;
     *
     *         <li>{@link SessionEstablishmentException} &ndash; if an initial
     *         connection could not be established;
     *
     *         <li>{@link AuthenticationException} &ndash;if the client is
     *         insufficiently authorized to start the session, based on the
     *         supplied client credentials.
     *         </ul>
     * @since 6.0
     */
    CompletableFuture<Session> openAsync(String url);

    /**
     * Open a connection to a server and request a new client session.
     * <P>
     * This is a convenience version of {@link #openAsync(String)} which blocks
     * until the session has been established.
     *
     * @param url the server location, see the {@link SessionFactory class
     *        documentation}
     *
     * @return a new session
     *
     * @throws IllegalArgumentException if {@code url} is invalid
     *
     * @throws IllegalStateException if any of the session attributes are found
     *         to be inconsistent. For example, if the default SSL context could
     *         not be loaded.
     *
     * @throws SessionEstablishmentException if an initial connection could not
     *         be established
     *
     * @throws AuthenticationException if the client is insufficiently
     *         authorized to start the session, based on the supplied client
     *         credentials
     */
    Session open(String url)
        throws IllegalArgumentException, IllegalStateException,
        SessionEstablishmentException,
        AuthenticationException;

    /**
     * Open a connection to a server and request a new session.
     *
     * <P>
     * This is a convenience version of {@link #openAsync()} which blocks until
     * the session has been established.
     *
     * @return a new session
     *
     * @throws IllegalStateException if any of the session attributes are found
     *         to be inconsistent. For example, if the default SSL context could
     *         not be loaded.
     *
     * @throws SessionEstablishmentException if an initial connection could not
     *         be established
     *
     * @throws AuthenticationException if the client is insufficiently
     *         authorized to start the session, based on the supplied client
     *         credentials
     * @since 5.8
     */
    Session open() throws IllegalStateException,
        SessionEstablishmentException,
        AuthenticationException;

}
