/*******************************************************************************
 * Copyright (c) 2016, 2025 DiffusionData Ltd., All Rights Reserved.
 *
 * Use is subject to licence terms.
 *
 * NOTICE: All information contained herein is, and remains the
 * property of DiffusionData. The intellectual and technical
 * concepts contained herein are proprietary to DiffusionData and
 * may be covered by U.S. and Foreign Patents, patents in process, and
 * are protected by trade secret or copyright law.
 *******************************************************************************/
package com.pushtechnology.diffusion.client.topics.details;

import java.util.Map;

import com.pushtechnology.diffusion.client.Diffusion;
import com.pushtechnology.diffusion.client.features.Pings;
import com.pushtechnology.diffusion.client.features.TimeSeries;
import com.pushtechnology.diffusion.client.features.TopicUpdate;
import com.pushtechnology.diffusion.client.features.Topics.ValueStream;
import com.pushtechnology.diffusion.client.features.control.topics.TopicControl;
import com.pushtechnology.diffusion.client.features.control.topics.TopicControl.InvalidTopicSpecificationException;
import com.pushtechnology.diffusion.client.features.control.topics.TopicNotifications;
import com.pushtechnology.diffusion.client.session.Session;
import com.pushtechnology.diffusion.client.topics.TopicSelector;
import com.pushtechnology.diffusion.datatype.DataType;
import com.pushtechnology.diffusion.datatype.DataTypes;
import com.pushtechnology.diffusion.datatype.recordv2.schema.Schema;
import com.pushtechnology.diffusion.datatype.recordv2.schema.SchemaBuilder;

/**
 * Topic specifications provide the information required to create a topic.
 * Topics can be created from a topic specification using
 * {@link TopicControl#addTopic(String, TopicSpecification)}.
 *
 * <p>
 * Topic specifications allow an application to introspect the type and
 * capabilities of a topic. Topic specifications are provided to
 * {@link ValueStream value streams} and {@link TopicNotifications topic
 * notification listeners}.
 *
 * <p>
 * A topic specification has a {@link TopicType topic type} and a map of
 * property settings which define the behaviour of the topic. A default topic
 * specification for a topic type can be created using
 * {@link Diffusion#newTopicSpecification(TopicType)}. Topic specifications with
 * different properties can be derived from a default instance using the
 * {@link #withProperties(Map)} and {@link #withProperty(String, String)}
 * builder methods.
 *
 * <h2>Topic Properties</h2>
 * <p>
 * Depending on the topic type, some properties must be included in the
 * specification when creating a topic and some properties have no effect. The
 * required and optional properties for each the topic type are set out in the
 * following table. Properties unsupported by the topic type are ignored.
 *
 * <table>
 * <tr>
 * <th></th>
 * <th style="text-align:center;">Default when optional</th>
 * <th></th>
 * <th style="text-align:center;">{@link TopicType#STRING STRING}<br>
 * {@link TopicType#JSON JSON}<br>
 * {@link TopicType#BINARY BINARY}<br>
 * </th>
 * <th style="text-align:center;">{@link TopicType#DOUBLE DOUBLE}<br>
 * {@link TopicType#INT64 INT64}</th>
 * <th style="text-align:center;">{@link TopicType#RECORD_V2 RECORD_V2}</th>
 * <th style="text-align:center;">{@link TopicType#TIME_SERIES TIME_SERIES}</th>
 * </tr>
 * <tr>
 * <tr>
 * <th style="text-align:left;">{@link #COMPRESSION}</th>
 * <td style="text-align:center;">{@code low}</td>
 * <td></td>
 * <td style="text-align:center;">Optional</td>
 * <td style="text-align:center;">—</td>
 * <td style="text-align:center;">Optional</td>
 * <td style="text-align:center;">Optional</td>
 * </tr>
 * <tr>
 * <th style="text-align:left;">{@link #CONFLATION}</th>
 * <td style="text-align:center;">{@code conflate}</td>
 * <td></td>
 * <td style="text-align:center;">Optional</td>
 * <td style="text-align:center;">Optional</td>
 * <td style="text-align:center;">Optional</td>
 * <td style="text-align:center;">†</td>
 * </tr>
 * <tr>
 * <th style="text-align:left;">{@link #DONT_RETAIN_VALUE}</th>
 * <td style="text-align:center;">{@code false}</td>
 * <td></td>
 * <td style="text-align:center;">Optional</td>
 * <td style="text-align:center;">Optional</td>
 * <td style="text-align:center;">Optional</td>
 * <td style="text-align:center;">—</td>
 * </tr>
 * <tr>
 * <th style="text-align:left;">{@link #OWNER}</th>
 * <td style="text-align:center;"></td>
 * <td></td>
 * <td style="text-align:center;">Optional</td>
 * <td style="text-align:center;">Optional</td>
 * <td style="text-align:center;">Optional</td>
 * <td style="text-align:center;">Optional</td>
 * </tr>
 * <tr>
 * <th style="text-align:left;">{@link #PERSISTENT}</th>
 * <td style="text-align:center;">{@code true}</td>
 * <td></td>
 * <td style="text-align:center;">Optional</td>
 * <td style="text-align:center;">Optional</td>
 * <td style="text-align:center;">Optional</td>
 * <td style="text-align:center;">Optional</td>
 * </tr>
 * <tr>
 * <th style="text-align:left;">{@link #PRIORITY}</th>
 * <td style="text-align:center;">{@code default}</td>
 * <td></td>
 * <td style="text-align:center;">Optional</td>
 * <td style="text-align:center;">Optional</td>
 * <td style="text-align:center;">Optional</td>
 * <td style="text-align:center;">Optional</td>
 * </tr>
 * <tr>
 * <th style="text-align:left;">{@link #PUBLISH_VALUES_ONLY}</th>
 * <td style="text-align:center;">{@code false}</td>
 * <td></td>
 * <td style="text-align:center;">Optional</td>
 * <td style="text-align:center;">—</td>
 * <td style="text-align:center;">Optional</td>
 * <td style="text-align:center;">Optional</td>
 * </tr>
 * <tr>
 * <th style="text-align:left;">{@link #REMOVAL}</th>
 * <td style="text-align:center;"></td>
 * <td></td>
 * <td style="text-align:center;">Optional</td>
 * <td style="text-align:center;">Optional</td>
 * <td style="text-align:center;">Optional</td>
 * <td style="text-align:center;">Optional</td>
 * </tr>
 * <tr>
 * <th style="text-align:left;">{@link #SCHEMA}</th>
 * <td style="text-align:center;"></td>
 * <td></td>
 * <td style="text-align:center;">—</td>
 * <td style="text-align:center;">—</td>
 * <td style="text-align:center;">Optional</td>
 * <td style="text-align:center;">—</td>
 * </tr>
 * <tr>
 * <th style="text-align:left;">{@link #TIDY_ON_UNSUBSCRIBE}</th>
 * <td style="text-align:center;">{@code false}</td>
 * <td></td>
 * <td style="text-align:center;">Optional</td>
 * <td style="text-align:center;">Optional</td>
 * <td style="text-align:center;">Optional</td>
 * <td style="text-align:center;">Optional</td>
 * </tr>
 * <tr>
 * <th style="text-align:left;">{@link #TIME_SERIES_EVENT_VALUE_TYPE}</th>
 * <td style="text-align:center;"></td>
 * <td></td>
 * <td style="text-align:center;">—</td>
 * <td style="text-align:center;">—</td>
 * <td style="text-align:center;">—</td>
 * <td style="text-align:center;">Required</td>
 * </tr>
 * <tr>
 * <th style="text-align:left;">{@link #TIME_SERIES_RETAINED_RANGE}</th>
 * <td style="text-align:center;">{@code limit 10}</td>
 * <td></td>
 * <td style="text-align:center;">—</td>
 * <td style="text-align:center;">—</td>
 * <td style="text-align:center;">—</td>
 * <td style="text-align:center;">Optional</td>
 * </tr>
 * <tr>
 * <th style="text-align:left;">{@link #TIME_SERIES_SUBSCRIPTION_RANGE}</th>
 * <td style="text-align:center;"><em>{@link #TIME_SERIES_SUBSCRIPTION_RANGE As
 * documented}</em></td>
 * <td></td>
 * <td style="text-align:center;">—</td>
 * <td style="text-align:center;">—</td>
 * <td style="text-align:center;">—</td>
 * <td style="text-align:center;">Optional</td>
 * </tr>
 * <tr>
 * <th style="text-align:left;">{@link #VALIDATE_VALUES}</th>
 * <td style="text-align:center;">{@code false}</td>
 * <td></td>
 * <td style="text-align:center;">Optional</td>
 * <td style="text-align:center;">Optional</td>
 * <td style="text-align:center;">Optional</td>
 * <td style="text-align:center;">Optional</td>
 * </tr>
 * </table>
 * <dl>
 * <dt>†
 * <dd>{@code TIME_SERIES} topics have restricted values for the
 * {@code CONFLATION} property. They are only allowed to have the values
 * {@code off} or {@code unsubscribe}.
 * </dl>
 *
 * @author DiffusionData Limited
 * @since 5.7
 */
public interface TopicSpecification {

    /**
     * Returns the topic type.
     *
     * @return type
     */
    TopicType getType();

    /**
     * Returns the topic properties.
     *
     * @return a map of the topic properties. This may be empty depending upon
     *         the level of detail requested
     */
    Map<String, String> getProperties();

    /**
     * Creates a new specification with the specified property value set.
     *
     * @param key the property key
     *
     * @param value the property value
     *
     * @return a new TopicSpecification based upon this one but with the
     *         specified property set to the specified value. All other property
     *         settings will be copied to the new specification.
     *
     * @throws IllegalArgumentException if key is not a supported property key
     * @throws NullPointerException if either the key or value is null
     */
    TopicSpecification withProperty(String key, String value)
        throws IllegalArgumentException;

    /**
     * Creates a new specification with the specified properties.
     *
     * @param properties map of properties
     *
     * @return a new TopicSpecification based upon this one but with all
     *         properties replaced by those in the supplied map. Previous
     *         property settings will not be copied to the new specification.
     *
     * @throws NullPointerException if {@code properties} is null
     * @throws IllegalArgumentException any key is not a supported property key
     * @throws NullPointerException if any key or value in {@code properties} is
     *         null
     */
    TopicSpecification withProperties(Map<String, String> properties)
        throws IllegalArgumentException;

    /**
     * Returns a new specification that does not have the specified properties.
     *
     * @param propertyNames a list of the property names that should be excluded
     *
     * @return a new TopicSpecification based upon this one but without those
     *         specified
     * @since 6.2
     */
    TopicSpecification withoutProperties(String... propertyNames);

    /**
     * Key of the topic property that specifies whether a topic should publish
     * only values.
     *
     * <p>
     * By default, a topic that supports delta streams will publish the
     * difference between two values (a delta) when doing so is more efficient
     * than publishing the complete new value. Subscribing sessions can use a
     * {@link ValueStream value stream} to automatically apply the delta to a
     * local copy of the topic value to calculate the new value.
     *
     * <p>
     * Setting {@code PUBLISH_VALUES_ONLY} to {@code true} disables this
     * behaviour so that deltas are never published. Doing so is usually not
     * recommended because it will result in more data being transmitted, less
     * efficient use of network resources, and increased transmission latency.
     * On the other hand, calculating deltas can require significant CPU from
     * the server or, if update streams are used, from the updating client. The
     * CPU cost will be higher if there are many differences between successive
     * values, in which case delta streams confer fewer benefits. If successive
     * values are unrelated to each other, consider setting
     * {@code PUBLISH_VALUES_ONLY} to {@code true}. Also consider setting
     * {@code PUBLISH_VALUES_ONLY} to {@code true} if the network capacity is
     * high and the bandwidth savings of deltas are not required.
     *
     * @see #DONT_RETAIN_VALUE
     */
    String PUBLISH_VALUES_ONLY = "PUBLISH_VALUES_ONLY";

    /**
     * Key of the topic property that specifies whether a topic should validate
     * inbound values.
     * <P>
     * By default, the server does not validate received values before sending
     * them on to client sessions. Invalid or corrupt values will be stored in
     * the topic and passed on to sessions. If this property is set to "true",
     * the server will perform additional validation on values to check that
     * they are valid instances of the data type, and if it is not then it will
     * return an error to the updater and not update the topic.
     * <P>
     * If this value is not set (or set to something other than "true"), no
     * server validation of inbound values is performed. This is the recommended
     * setting as there is a performance overhead to validation and a session
     * using {@link TopicUpdate topic update} cannot send invalid values
     * anyway.
     */
    String VALIDATE_VALUES = "VALIDATE_VALUES";

    /**
     * Key of the topic property that specifies the 'tidy on unsubscribe' option
     * for a topic.
     * <P>
     * By default, if a session unsubscribes from a topic, it will receive any
     * updates for that topic that were previously queued but not sent.
     * <P>
     * If this property is set to "true", when a session unsubscribes from the
     * topic, any updates for the topic that are still queued for the session
     * are removed. There is a performance overhead to using this option as the
     * client queue must be scanned to find topic updates to remove, however it
     * may prove useful for preventing unwanted data being sent to sessions.
     *
     * @since 6.0
     */
    String TIDY_ON_UNSUBSCRIBE = "TIDY_ON_UNSUBSCRIBE";

    /**
     * Key of the topic property that specifies the event data type for a time
     * series topic.
     *
     * <p>
     * The value is the {@link DataType#getTypeName() type name} of a data type.
     *
     * @since 6.0
     * @see DataTypes#getByName(String)
     */
    String TIME_SERIES_EVENT_VALUE_TYPE = "TIME_SERIES_EVENT_VALUE_TYPE";

    /**
     * Key of the topic property that specifies the range of events retained by
     * a time series topic.
     *
     * <p>
     * When a new event is added to the time series, older events that fall
     * outside of the range are discarded.
     *
     * <p>
     * If the property is not specified, a time series topic will retain the ten
     * most recent events.
     *
     * <h4>Time series range expressions</h4>
     *
     * <p>
     * The property value is a time series <em>range expression</em> string
     * composed of one or more constraint clauses. Constraints are combined to
     * provide a range of events from the end of the time series.
     *
     * <dl>
     * <dt>limit constraint
     * <dd>A limit constraint specifies the maximum number of events from the
     * end of the time series.
     * <dt>last clause
     * <dd>A last constraint specifies the maximum duration of events from the
     * end of the time series. The duration is expressed as an integer followed
     * by one of the following time units. <br>
     * <br>
     * {@code MS} &ndash; milliseconds; <br>
     * {@code S} &ndash; seconds; <br>
     * {@code H} &ndash; hours.
     * </dl>
     *
     * <p>
     * If a range expression contains multiple constraints, the constraint that
     * selects the smallest range is used.
     *
     * <table>
     * <tr>
     * <th>Property value</th>
     * <th>Meaning</th>
     * </tr>
     * <tr>
     * <td>{@code limit 5}</td>
     * <td>The five most recent events</td>
     * </tr>
     * <tr>
     * <td>{@code last 10s}</td>
     * <td>All events that are no more than ten seconds older than the latest
     * event</td>
     * </tr>
     * <tr>
     * <td>{@code last 10s limit 5}</td>
     * <td>The five most recent events that are no more than ten seconds older
     * than the latest event</td>
     * </tr>
     * </table>
     *
     * <p>
     * Range expressions are not case sensitive: {@code limit 5 last 10s} is
     * equivalent to {@code LIMIT 5 LAST 10S}.
     *
     * @since 6.0
     */
    String TIME_SERIES_RETAINED_RANGE = "TIME_SERIES_RETAINED_RANGE";

    /**
     * Key of the topic property that specifies the range of time series topic
     * events to send to new subscribers.
     *
     * <p>
     * The property value is a time series range expression, following the
     * format used for {@link #TIME_SERIES_RETAINED_RANGE}.
     *
     * <p>
     * If the property is not specified, new subscribers will be sent the latest
     * event if delta streams are enabled and no events if delta streams are
     * disabled. See the description of <em>Subscription range</em> in the
     * {@link TimeSeries time series feature} documentation.
     *
     * @since 6.0
     */
    String TIME_SERIES_SUBSCRIPTION_RANGE = "TIME_SERIES_SUBSCRIPTION_RANGE";

    /**
     * Key of the topic property that specifies a schema which constrains topic
     * values.
     *
     * <p>
     * This property is only used by {@link TopicType#RECORD_V2 RECORD_V2}
     * topics. The property value can be generated using the
     * {@link Schema#asJSONString()} method of a {@link Schema} created using a
     * {@link SchemaBuilder}.
     *
     * @since 6.0
     */
    String SCHEMA = "SCHEMA";

    /**
     * Key of the topic property that specifies a topic should not retain its
     * last value.
     *
     * <p>
     * By default, a topic will retain its latest value. The latest value will
     * be sent to new subscribers. Setting this property to {@code true}
     * disables this behaviour. New subscribers will not be sent an initial
     * value. No value will be returned for fetch operations that select the
     * topic. This is useful for data streams where the values are only
     * transiently valid.
     *
     * <p>
     * Setting {@code DONT_RETAIN_VALUE} to {@code true} also disables delta
     * streams, regardless of the {@link #PUBLISH_VALUES_ONLY} value. If
     * subsequent values are likely to be related, delta streams usually provide
     * performance benefits (see {@link #PUBLISH_VALUES_ONLY}). Consider leaving
     * {@code DONT_RETAIN_VALUE} set to {@code false} to benefit from delta
     * streams, even if there is no other requirement to retain the last value.
     *
     * <p>
     * Bearing in mind the performance trade-offs of disabling delta streams,
     * there are two reasons to consider setting {@code DONT_RETAIN_VALUE} to
     * {@code true}. First, it stops the server and each subscribed client from
     * keeping a copy of the value, reducing their memory requirements. Second,
     * when a topic has a high update rate and is replicated across a cluster,
     * it can significantly improve throughput because the values need not be
     * persisted to the cluster.
     *
     * <p>
     * Time series topics ignore this property and always retain the latest
     * value.
     *
     * @since 6.0
     */
    String DONT_RETAIN_VALUE = "DONT_RETAIN_VALUE";

    /**
     * Key of the topic property that can be used to prevent a topic from being
     * persisted when the server is configured to enable persistence.
     * <P>
     * By default, a topic will be persisted if persistence is enabled at the
     * server and the topic type supports persistence.
     * <p>
     * Setting PERSISTENT to {@code false} will prevent the topic from being
     * persisted.
     *
     * @since 6.1
     */
    String PERSISTENT = "PERSISTENT";

    /**
     * Key of the topic property that specifies a removal policy for automatic
     * removal of the topic (and/or other topics).
     * <p>
     * This property is specified as an expression which defines one or more
     * conditions that are to be satisfied before automatic removal occurs.
     * <p>
     * The expression takes the form:
     * <p>
     * <code>
     * when <i>conditions</i> [remove "<i>selector</i>"]
     * </code>
     * <p>
     * At least one condition must be supplied. If more than one is supplied,
     * they must be separated by logical operators ({@code and} or {@code or}).
     * The natural evaluation order of the operators may be changed by
     * surrounding with parentheses (e.g. (<i>condition</i> {@code and}
     * <i>condition</i>)).
     * <p>
     * The {@code remove} clause is optional. It provides a
     * {@link TopicSelector} expression representing the topics to be removed.
     * If a {@code remove} clause is specified, the topic with the removal
     * policy will only be removed if its path matches the selector expression.
     * The selector must be surrounded by either double or single quotes.
     * <p>
     * When many topics have the same removal policy, it is better to
     * set the {@code REMOVAL} property for one of them, using a {@code remove}
     * clause that selects all of the topics. This is more efficient because it
     * allows the server to avoid evaluating the same condition many times.
     * </p>
     * <p>
     * The permissions that are applied at the time of removal are those defined
     * by the roles of the principal that created the topic at the time of
     * creation. The roles of that principal may therefore change before the
     * removal with no effect, but if the permissions given to the roles change
     * it may have an effect upon the final removal.
     * <p>
     * Only one occurrence of each of the following condition types may be
     * included within the expression:
     * <table>
     * <tr>
     * <th style="text-align:left;">Condition&nbsp;Type</th>
     * <th style="text-align:left;">Format</th>
     * <th style="text-align:left;">Usage</th>
     * </tr>
     * <tr style="vertical-align:top">
     * <th style="text-align:left;"><b>time&nbsp;after</b></th>
     * <td><code>time&nbsp;after&nbsp;<i>absoluteTime</i></code></td>
     * <td>Removal should occur after a specified absolute time. Absolute time
     * may be specified as a number of milliseconds since the epoch (00:00:00 on
     * 1 January 1970) <b>or</b> as a quoted date and time formatted in <a href=
     * "https://docs.oracle.com/javase/8/docs/api/java/time/format/DateTimeFormatter.html#RFC_1123_DATE_TIME">RFC_1123
     * date time format</a>. Either single or double quotes may be used.</td>
     * </tr>
     * <tr style="vertical-align:top">
     * <th style=
     * "text-align:left;"><b>subscriptions&nbsp;less&nbsp;than</b></th>
     * <td><code>[local]&nbsp;subscriptions&nbsp;&lt;&nbsp;<i>n</i>&nbsp;for&nbsp;<i>forPeriod</i>&nbsp;[after&nbsp;<i>afterPeriod</i>]</code></td>
     * <td>Removal should occur when the topic has had less than the specified
     * number (<i>n</i>) of subscriptions for a given period (<i>forPeriod</i>)
     * of time. Optionally, an initial period (<i>afterPeriod</i>) may be
     * specified by which to delay the initial checking of this condition. See
     * below for period formats.
     * <p>
     * The optional <code>local</code> keyword restricts evaluation to only count
     * subscriptions from sessions belonging to the local server or cluster,
     * ignoring subscriptions from sessions belonging to downstream remote servers that
     * host fanout replicas of the topic.</td>
     * </tr>
     * <tr style="vertical-align:top">
     * <th style="text-align:left;"><b>no&nbsp;updates&nbsp;for</b></th>
     * <td><code>no&nbsp;updates&nbsp;for&nbsp;<i>forPeriod</i>&nbsp;[after&nbsp;<i>afterPeriod</i>]</code></td>
     * <td>Removal should occur when the topic has had no updates for a given
     * period (<i>forPeriod</i>) of time. Optionally, an initial period
     * (<i>afterPeriod</i>) may be specified by which to delay the initial
     * checking of this condition. See below for period formats.</td>
     * </tr>
     * </table>
     * <p>
     * Multiple occurrences of the following condition types may be included
     * within the expression:
     * <table>
     * <tr>
     * <th style="text-align:left;">Condition&nbsp;Type</th>
     * <th style="text-align:left;">Format</th>
     * <th style="text-align:left;">Usage</th>
     * </tr>
     * <tr style="vertical-align:top">
     * <th style="text-align:left;"><b>no&nbsp;session&nbsp;has</b></th>
     * <td><code>no&nbsp;[local]&nbsp;session&nbsp;has&nbsp;<i>"criteria"</i>&nbsp;[for&nbsp;<i>forPeriod</i>]&nbsp;[after&nbsp;<i>afterPeriod</i>]</code></td>
     * <td>Removal should occur when there are no sessions satisfying certain
     * <i>criteria</i>. Optionally the criteria can be required to be satisfied
     * for a period of time (<i>forPeriod</i>). Optionally, an initial period
     * (<i>afterPeriod</i>) can be specified to delay the initial check of the
     * criteria. Session selection criteria are specified as defined in
     * {@link Session session filters} and must be surrounded by single or
     * double quotes. See below for period formats.
     * <p>The optional <code>local</code> keyword restricts evaluation to sessions
     * belonging to the local server or cluster, ignoring sessions belonging to
     * downstream remote servers that host fanout replicas of the topic.</td>
     * </tr>
     * <tr style="vertical-align:top">
     * <th></th>
     * <td><code>this&nbsp;session&nbsp;closes</code></td>
     * <td>This is a shorthand form of {@code no local session has} that may be
     * used to indicate that the topic is to be removed when the session that
     * created it closes.</td>
     * </tr>
     * </table>
     * Time periods are specified as a number followed (with no intermediate
     * space) by a single letter representing the time unit. The time unit may
     * be {@code s} (seconds), {@code m}(minutes), {@code h} (hours) or
     * {@code d} (days). For example, 10 minutes would be specified as
     * {@code 10m}.
     * <p>
     * If quotes or backslashes ({@code \}) are required within quoted values
     * such as selectors or session criteria then they may be escaped by
     * preceding with {@code \}. The convenience method
     * {@link Diffusion#escape(String)} is provided to escape such characters in
     * a value. The expression is validated only by the server and therefore if
     * an invalid expression is specified it will be reported as an
     * {@link InvalidTopicSpecificationException}.
     * <p>
     * <b>Examples:</b>
     * <p>
     * <code>
     * when time after 1518780068112
     * </code><br>
     * The topic will be removed when the date and time indicated by the
     * specified number of milliseconds since the epoch has passed.
     * <p>
     * <code>
     * when time after "Tue, 3 Jun 2018 11:05:30 GMT"
     * </code><br>
     * The topic will be removed when the specified date and time has passed.
     * <p>
     * <code>
     * when time after "Tue, 3 Jun 2018 11:05:30 GMT" remove "*alpha/beta//"
     * </code><br>
     * The topic alpha/beta and all topics subordinate to it will be removed
     * when the specified date and time has passed.
     * <p>
     * <code>
     * when subscriptions &lt; 1 for 20m
     * </code><br>
     * The topic will be removed when it has had no subscriptions for a
     * continuous period of 20 minutes.
     * <p>
     * <code>
     * when subscriptions &lt; 2 for 20m after 1h
     * </code><br>
     * The topic will be removed when it has had less than 2 subscriptions for a
     * continuous period of 20 minutes after one hour has passed since its
     * creation.
     * <p>
     * <code>
     * when no updates for 3h
     * </code><br>
     * The topic will be removed when it has had no updates for a continuous
     * period of 3 hours.
     * <p>
     * <code>
     * when no updates for 15m after 1d
     * </code><br>
     * The topic will be removed when it has had no updates for a continuous
     * period of 15 minutes after one day has passed since its creation.
     * <p>
     * <code>
     * when this session closes
     * </code><br>
     * The topic will be removed when the session creating it closes.
     * <p>
     * <code>
     * when no session has '$Principal is "Alice"'
     * </code><br>
     * The topic will be removed when there are no sessions with the principal
     * 'Alice'.
     * <p>
     * <code>
     * when no session has '$Principal is "Alice"' for 10m
     * </code><br>
     * The topic will be removed when there are no sessions with the principal
     * 'Alice' for a continuous period of 10 minutes.
     * <p>
     * <code>
     * when no session has 'Department is "Accounts"' for 30m after 2h
     * </code><br>
     * The topic will be removed when there have been no sessions from the
     * Accounts department for a continuous period of 30 minutes after 2 hours
     * have passed since its creation.
     * <p>
     * <code>
     * when time after "Tue, 3 Jun 2018 11:05:30 GMT" and subscriptions &lt; 1 for 30m
     * </code><br>
     * The topic will be removed when the specified date and time has passed and
     * the topic has had no subscriptions for a continuous period of 30 minutes
     * after that time.
     * <p>
     * <code>
     * when time after "Tue, 3 Jun 2018 11:05:30 GMT" and subscriptions &lt; 2 for 10m after 1h
     * </code><br>
     * The topic will be removed when the specified date and time has passed and
     * the topic has had less than 2 subscriptions for a continuous period of 10
     * minutes after that time plus one hour.
     * <p>
     * <code>
     * when time after "Tue, 3 Jun 2018 11:05:30 GMT" or subscriptions &lt; 2 for 10m after 1h
     * </code><br>
     * The topic will be removed when the specified date and time has passed or
     * the topic has had less than 2 subscriptions for a continuous period of 10
     * minutes after one hour from its creation.
     * <p>
     * <code>
     * when time after "Tue, 3 Jun 2018 11:05:30 GMT" and (subscriptions &lt; 2 for 10m after 1h or no updates for 20m)
     * </code><br>
     * The topic will be removed when the specified date and time has passed and
     * either the topic has had less than 2 subscriptions for a continuous
     * period of 10 minutes after that time plus one hour or it has had no
     * updates for a continuous period of 20 minutes. Note that the parentheses
     * are significant here as without them the topic would be removed if it had
     * had no updates for 20 minutes regardless of the time and subscriptions
     * clause.
     * <p>
     * <b>Notes and restrictions on use</b>
     * <p>
     * The {@code after} time periods refer to the period since the topic was
     * created or restored from persistence store after a server is restarted.
     * They are designed as a 'grace' period after the topic comes into
     * existence before the related condition starts to be evaluated. When not
     * specified the conditions start to be evaluated as soon as the topic is
     * created or restored.
     * <p>
     * The server will evaluate conditions on a periodic basis (every few
     * seconds) so the exact removal time will not be precise for low periodic
     * granularity.
     * <p>
     * The meaning of the {@code for} period in a {@code no session has}
     * condition is subtly different from its use in other conditions. It does
     * not guarantee that there has been no session satisfying the condition at
     * some point between evaluations, only that when evaluated the given period
     * of time has passed since it was last evaluated and found to have no
     * matching sessions.
     * <p>
     * Subscriptions is the number of subscriptions to a topic.
     * <p>
     * Automatic topic removal is supported for a topic that is replicated
     * across the local cluster, and for a topic with with fanout replicas on
     * downstream remote servers. A {@code subscriptions less than} condition
     * will be evaluated against the total number of subscriptions across the
     * cluster and on all fanout replicas on downstream remote servers. A
     * {@code no session has} condition will consider all sessions hosted across
     * the cluster and all sessions hosted by downstream remote servers that
     * have a fanout replica of the topic. The {@code local} keyword can be used
     * to restrict evaluation to the local cluster, ignoring fanout replicas.
     *
     * @since 6.1
     */
    String REMOVAL = "REMOVAL";

    /**
     * Key of the topic property that specifies the conflation policy of the
     * topic. The policy specifies how the server manages queued topic updates.
     * Conflation is applied individually to each session queue.
     * <p>
     * Conflation is the process of merging or discarding topic updates queued
     * for a session to reduce the server memory footprint and network data. The
     * server will conflate sessions that have a large number of queued messages
     * to meet configured queue size targets. The sessions with the largest
     * queues are typically slow consumers or have been disconnected – both will
     * benefit from conflation. This property allows conflation behaviour to be
     * tuned on a topic-by-topic basis.
     * <p>
     * The supported policies are:
     * <ul>
     * <li>{@code off}</li>
     * <li>{@code conflate}</li>
     * <li>{@code unsubscribe}</li>
     * <li>{@code always}</li>
     * </ul>
     * <p>The default policy used when the property is not specified and the
     * topic type is not time series is {@code conflate}. The default policy
     * used when the property is not specified and the topic type is time
     * series is {@code off}.
     * <p>
     * The policy {@code off} disables conflation for the topic. This policy
     * disables all conflation for the topic, so topic updates will never be
     * merged or discarded.
     * <p>
     * The policy {@code conflate} automatically conflates topic updates when
     * back pressure is detected by the server.
     * <p>
     * The policy {@code unsubscribe} automatically unsubscribes the topic when
     * back pressure is detected by the server. The unsubscription is not
     * persisted to the cluster. If a session fails over to a different server
     * it will be resubscribed to the topic.
     * <p>
     * The policy {@code always} automatically conflates topic updates as they
     * are queued for the session. This is an eager policy that ensures only the
     * latest update is queued for the topic, minimising the server memory and
     * network bandwidth used by the session.
     * <p>
     * The {@code conflate} and {@code unsubscribe} policies are applied when
     * the server detects back pressure for a session. The server configuration
     * places limits on the data queued for each session. If these limits are
     * breached, the server will conflate the session queue to attempt to reduce
     * its size. If the session queue still exceeds the limits after conflation,
     * the session will be terminated.
     * <p>
     * Conflation can be
     * {@link com.pushtechnology.diffusion.client.features.control.clients.ClientControl#setConflated(com.pushtechnology.diffusion.client.session.SessionId, boolean)
     * disabled on a session-by-session basis}. If conflation is disabled for a
     * session the policy will not be applied to topic updates queued for the
     * session but will be for other sessions that have conflation enabled.
     * <p>
     * The policies {@code conflate} and {@code always} are not supported for
     * time series topics as they would cause missing events. Attempts to enable
     * these policies with time series topics will cause the creation of the
     * topic to fail, reporting that the specification is invalid.
     *
     * @since 6.1
     */
    String CONFLATION = "CONFLATION";

    /**
     * Key of the topic property that allows the creator of a topic to extend
     * READ_TOPIC, MODIFY_TOPIC, and UPDATE_TOPIC permissions to a specific
     * principal, in addition to the permissions granted by the authorisation
     * rules in the security store.
     * <p>
     * A session that has authenticated using the principal can update and
     * remove the topic, so the principal can be considered the topic owner. To
     * fetch or subscribe to the topic, the principal must also be granted
     * the SELECT_TOPIC permission by the security store rules.
     * <p>
     * This may be used in the following cases:<br>
     * 1) A session creates a topic and makes its own principal the owner.<br>
     * 2) A session creates a topic and makes another principal the owner.
     * <p>
     * The format of the property value is:
     * <p>
     * <code>$Principal is "<i>name</i>"</code>
     * <p>
     * where <i>name</i> is the name of the principal. Single quotes may be used
     * instead of double quotes and special characters can be escaped using
     * {@link Diffusion#escape(String)} if required.
     * <p>
     * The purpose of this property is to allow a client to create topics on
     * behalf of other users. This can be used in conjunction with the
     * {@link #REMOVAL} property so that such topics are removed when there are
     * no longer any sessions for the named principal.
     * <p>
     * For example:
     * <p>
     * <code>
     * specification<br>
     * &nbsp;&nbsp;&nbsp;&nbsp;.withProperty(OWNER, "$Principal is 'myPrincipal'")<br>
     * &nbsp;&nbsp;&nbsp;&nbsp;.withProperty(REMOVAL, "when no session has '$Principal is \"myPrincipal\"' for 5s");
     * </code>
     *
     * @since 6.1
     */
    String OWNER = "OWNER";

    /**
     * Key of the topic property that allows the compression policy to be set
     * on a per-topic basis.
     *
     * <p>
     * Compression reduces the bandwidth required to broadcast topic updates to
     * subscribed sessions, at the cost of increased server CPU.
     *
     * <p>
     * Changes to a topic's value are published to each subscribed session as a
     * sequence of topic messages. A topic message can carry the latest value or
     * the difference between the latest value and the previous value (a delta).
     * The compression policy determines if and how published topic messages
     * are compressed. Topic messages are not exposed through the client API;
     * the client library handles decompression and decodes deltas
     * automatically, passing reconstructed values to the application.
     *
     * <p>
     * The compression policy for a topic is specified by setting this property
     * to one of several values:
     * <ul>
     * <li>{@code off}</li>
     * <li>{@code low}</li>
     * <li>{@code medium}</li>
     * <li>{@code high}</li>
     * </ul>
     *
     * <p>
     * The policies are listed in the order of increasing compression and
     * increasing CPU cost. {@code off} disables compression completely for the
     * topic and requires no additional CPU; {@code high} compresses the topic
     * messages to the smallest number of bytes, but has the highest CPU cost.
     * Generally some compression is beneficial, so the default value for this
     * property is {@code low}.
     *
     * <p>
     * Prior to version 6.4, only two values were allowed: {@code true}
     * (equivalent to {@code medium}, and the previous default policy) and
     * {@code false} (equivalent to {@code off}). These values are still
     * supported.
     *
     * <p>
     * This property is only one factor that determines whether a topic message
     * will be compressed. Other factors include:
     * <ul>
     * <li>Compression must be enabled in the server configuration.
     * <li>The client library must support the server's compression scheme. In
     * this release, the server supports zlib compression, and also allows
     * compression to be disabled on a per-connector basis. From 6.4, all client
     * libraries are capable of zlib compression. A JavaScript client may or may
     * not support zlib compression, depending on whether the zlib library can
     * be loaded. The zlib library is packaged separately to reduce the download
     * size of the core library.
     * </ul>
     *
     * @since 6.2
     */
    String COMPRESSION = "COMPRESSION";

    /**
     * Key of the topic property that specifies the topic delivery priority.
     * <p>
     * The supported delivery priorities are:
     * <ul>
     * <li>{@code low}</li>
     * <li>{@code default}</li>
     * <li>{@code high}</li>
     * </ul>
     *
     * <p>
     * The delivery priority affects the order of topic updates sent to a
     * subscribed client session. When there are multiple topic updates for
     * topics with different priorities in a session's outbound queue, updates
     * for {@code high} priority topics will be delivered first, followed by
     * updates for {@code default} priority topics, followed by updates for
     * {@code low} priority topics. Topic subscription and unsubscription
     * notifications are also delivered according to the topic delivery
     * priority.
     *
     * <p>
     * Using different delivery priorities is most beneficial when there is a
     * large backlog of queued updates to deliver to a client session. On
     * lightly loaded systems, updates typically remain in the outbound queue
     * for a few milliseconds and so there is a lower chance of topic updates
     * being reordered based on their priority. The backlog will be larger if
     * the topic update rate is higher; the server or the client are more
     * heavily loaded; the client session becomes temporarily disconnected; or
     * if there is poor network connectivity between the server and the client.
     *
     * <p>
     * Messages from the server to the client that are not topic updates, for
     * example {@link Pings ping} requests and responses, are queued with the
     * {@code default} delivery priority.
     *
     * @since 6.4
     */
    String PRIORITY = "PRIORITY";
}
