/*******************************************************************************
 * Copyright (c) 2014, 2023 DiffusionData Ltd., All Rights Reserved.
 *
 * Use is subject to licence terms.
 *
 * NOTICE: All information contained herein is, and remains the
 * property of DiffusionData. The intellectual and technical
 * concepts contained herein are proprietary to DiffusionData and
 * may be covered by U.S. and Foreign Patents, patents in process, and
 * are protected by trade secret or copyright law.
 *******************************************************************************/
package com.pushtechnology.diffusion.client.topics.details;

import com.pushtechnology.diffusion.client.features.TimeSeries;
import com.pushtechnology.diffusion.datatype.DataType;
import com.pushtechnology.diffusion.datatype.DataTypes;
import com.pushtechnology.diffusion.datatype.recordv2.RecordV2;
import com.pushtechnology.diffusion.datatype.recordv2.RecordV2DataType;

/**
 * The topic type determines the type of the data values a topic publishes to
 * subscribers.
 *
 * <h3>Source Topics</h3>
 *
 * Most topics are source topics. The characteristics of each type of source
 * topic are summarized in the following table.
 *
 * <table>
 * <tr>
 * <th>Topic type</th>
 * <th>State</th>
 * <th>Data type</th>
 * </tr>
 * <tr>
 * <th>{@link #STRING}</th>
 * <td>Single scalar value.</td>
 * <td>String.</td>
 * </tr>
 * <tr>
 * <th>{@link #INT64}</th>
 * <td>Single scalar value.</td>
 * <td>64-bit integer.</td>
 * </tr>
 * <tr>
 * <th>{@link #DOUBLE}</th>
 * <td>Single scalar value.</td>
 * <td>Double precision floating point number.</td>
 * </tr>
 * <tr>
 * <th>{@link #BINARY}</th>
 * <td>Single scalar value.</td>
 * <td>Arbitrary binary data.</td>
 * </tr>
 * <tr>
 * <th>{@link #RECORD_V2}</th>
 * <td>Single composite value.</td>
 * <td>{@link RecordV2} &ndash; Diffusion-specific data type. A list of records,
 * each composed of field values, with an optional schema.</td>
 * </tr>
 * <tr>
 * <th>{@link #JSON}</th>
 * <td>Single composite value.</td>
 * <td>{@link com.pushtechnology.diffusion.datatype.json.JSON JSON}, backed by
 * CBOR-format binary.</td>
 * </tr>
 * <tr>
 * <th>{@link #TIME_SERIES}</th>
 * <td>Append-only log of events.</td>
 * <td>{@link com.pushtechnology.diffusion.client.features.TimeSeries.Event
 * TimeSeries.Event} containing a value of a {@link DataTypes well-known data
 * type}.</td>
 * </tr>
 *
 * </table>
 *
 * @author DiffusionData Limited
 * @since 5.0
 * @see TopicSpecification
 */
public enum TopicType {

    /**
     * Topic that stores and publishes JSON (JavaScript Object Notation) values.
     * Based on the {@link DataTypes#json JSON data type}.
     * <p>
     * Supports delta-streams.
     *
     * @since 5.7
     */
    JSON,

    /**
     * Topic that stores and publishes binary values. Based on the
     * {@link DataTypes#binary binary data type}.
     * <p>
     * Supports delta-streams.
     *
     * @since 5.7
     */
    BINARY,

    /**
     * Topic that stores and publishes data in the form of records and fields.
     * Based on the {@link RecordV2DataType RecordV2} data type.
     * <p>
     * Supports delta-streams.
     *
     * @since 6.0
     */
    RECORD_V2,

    /**
     * Topic that stores and publishes IEEE 754 double-precision floating point
     * numbers. Based on the {@link DataTypes#doubleFloat() double data type}.
     * <p>
     * Supports null Double values.
     * <p>
     * The topic does not support delta-streams — only complete values are
     * transmitted.
     *
     * @since 6.0
     */
    DOUBLE,

    /**
     * Topic that stores and publishes 64-bit integer values. Based on the
     * {@link DataTypes#int64() int64 data type}.
     * <p>
     * Supports null Long values.
     * <p>
     * The topic does not support delta-streams — only complete values are
     * transmitted.
     *
     * @since 6.0
     */
    INT64,

    /**
     * Topic that stores and publishes String values. Based on the
     * {@link DataTypes#string() string data type}.
     * <p>
     * Supports null String values.
     * <p>
     * Supports delta-streams.
     *
     * @since 6.0
     */
    STRING,

    /**
     * Time Series Topic.
     *
     * <p>
     * A <em>time series</em> is a sequence of events. Each event contains a
     * value and has server-assigned metadata comprised of a sequence number,
     * timestamp, and author.
     * <p>
     * A time series topic allows sessions to access a time series that is
     * maintained by the server. A time series topic has an associated
     * {@link DataType event data type}, such as {@code Binary}, {@code String},
     * or {@code JSON}, that determines the type of value associated with each
     * event.
     *
     * <h4>Retained range</h4>
     *
     * <p>
     * The {@link TopicSpecification#TIME_SERIES_SUBSCRIPTION_RANGE
     * TIME_SERIES_SUBSCRIPTION_RANGE} property configures the range of historic
     * events retained by a time series topic. If the property is not specified,
     * a time series topic will retain the ten most recent events.
     *
     * <h4>Subscription range</h4>
     *
     * <p>
     * The {@link TopicSpecification#TIME_SERIES_SUBSCRIPTION_RANGE
     * TIME_SERIES_SUBSCRIPTION_RANGE} property configures a time series topic
     * to send a range of historic events from the end of the time series to new
     * subscribers. This is a convenient way to synchronize new subscribers
     * without requiring the use of a {@link TimeSeries#rangeQuery() range
     * query}.
     * <p>
     * By default, new subscribers will be sent the latest event if delta
     * streams are enabled and no events if delta streams are disabled. See the
     * description of <em>Subscription range</em> in the {@link TimeSeries time
     * series feature} documentation.
     *
     *
     * <h4>Mandatory properties</h4>
     * <p>
     * The {@link TopicSpecification#TIME_SERIES_EVENT_VALUE_TYPE
     * TIME_SERIES_EVENT_VALUE_TYPE} property must be provided when creating a
     * time series topic.
     *
     *
     * @since 6.0
     * @see TimeSeries
     */
    TIME_SERIES,

    /**
     * A topic type that is unsupported by the session.
     *
     * @since 6.1
     */
    UNKNOWN_TOPIC_TYPE,
}
