package com.xxdb.data;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import com.fasterxml.jackson.annotation.JsonIgnore;
import com.xxdb.io.ExtendedDataInput;
import com.xxdb.io.ExtendedDataOutput;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;


public class BasicArrayVector extends AbstractVector {
	private DATA_TYPE type;
	private int[] rowIndices;
	private Vector valueVec;
	private int baseUnitLength_;
	private int rowIndicesSize;
	private int capacity;
	private int scale_ = -1;

	private static final Logger log = LoggerFactory.getLogger(BasicArrayVector.class);

	public BasicArrayVector(DATA_TYPE type, int size){
		this(type, size, -1);
	}

	public BasicArrayVector(DATA_TYPE type, int size, int extra){
		super(DATA_FORM.DF_VECTOR);
		rowIndices = new int[size];
		if (type.getValue() == 81 || type.getValue() == 82)
			throw new RuntimeException("ArrayVector do not support String and Symbol");
		this.type = DATA_TYPE.valueOf(type.getValue());
		valueVec = BasicEntityFactory.instance().createVectorWithDefaultValue(DATA_TYPE.valueOf(type.getValue() - 64), 0, extra);
		if (extra >= 0){
			((AbstractVector)valueVec).setExtraParamForType(extra);
			this.scale_ = extra;
		}
		baseUnitLength_ = valueVec.getUnitLength();
		rowIndicesSize = 0;
		capacity = rowIndices.length;
	}

	public BasicArrayVector(List<Vector> value) throws Exception{
		super(DATA_FORM.DF_VECTOR);
		if (value.get(0).getDataType().getValue() + 64 == 81 || value.get(0).getDataType().getValue() + 64 == 82)
			throw new RuntimeException("ArrayVector do not support String and Symbol");
		else{
			this.type = DATA_TYPE.valueOf(value.get(0).getDataType().getValue() + 64);
			DATA_TYPE valueType = DATA_TYPE.valueOf(value.get(0).getDataType().getValue());
			if (valueType == DATA_TYPE.DT_DECIMAL32 || valueType == DATA_TYPE.DT_DECIMAL64 || valueType == DATA_TYPE.DT_DECIMAL128){
				int scale = ((AbstractVector)value.get(0)).getExtraParamForType();
				for (int i = 0; i < value.size(); i++){
					int scaleCopy = ((AbstractVector)value.get(i)).getExtraParamForType();
					if (scaleCopy != scale){
						throw new RuntimeException("The scale of decimal arrayVector's value is not same.");
					}
				}
				this.scale_ = scale;
			}
			int len = 0;
			for (Vector one : value){
				if(one.rows()>0)
					len += one.rows();
				else
					len++;
			}
			int indexPos = 0;
			int indexCount = value.size();
			this.rowIndices = new int[indexCount];
			this.valueVec = BasicEntityFactory.instance().createVectorWithDefaultValue(valueType, len, this.scale_);
			if (valueType == DATA_TYPE.DT_DECIMAL32){
				((BasicDecimal32Vector)valueVec).setScale(scale_);
			}else if (valueType == DATA_TYPE.DT_DECIMAL64){
				((BasicDecimal64Vector)valueVec).setScale(scale_);
			} else if (valueType == DATA_TYPE.DT_DECIMAL128) {
				((BasicDecimal128Vector)valueVec).setScale(scale_);
			}
			int index = 0;
			int curRows = 0;
			for (int valuePos = 0; valuePos < indexCount; valuePos++){
				Vector temp = value.get(valuePos);
				int size = temp.rows();
				if (size > 0){
					for (int i = 0; i < size ; i++){
						try {
							this.valueVec.set(index, temp.get(i));
							index++;
						}catch (Exception e){
							throw new RuntimeException("Failed to insert data, invalid data for "+((Vector)temp).get(i)+" error "+ e.toString());
						}
					}
					curRows += size;
				}
				else{
					this.valueVec.setNull(index);
					index++;
					curRows ++;
				}
				this.rowIndices[indexPos++] = curRows;
			}
			rowIndicesSize = rowIndices.length;
			capacity = rowIndices.length;
			this.baseUnitLength_ = (this.valueVec).getUnitLength();
		}
	}

	public BasicArrayVector(int[] index, Vector value) {
		super(DATA_FORM.DF_VECTOR);
		DATA_TYPE dataType = value.getDataType();
		this.type = DATA_TYPE.valueOf(dataType.getValue() + 64);
		this.valueVec = value;
		int indexCount = index.length;
		rowIndices = new int[indexCount];
		System.arraycopy(index, 0, this.rowIndices, 0, indexCount);
		this.baseUnitLength_ = value.getUnitLength();
		rowIndicesSize = rowIndices.length;
		capacity = rowIndices.length;
	}

	public BasicArrayVector(DATA_TYPE type, ExtendedDataInput in, int rows, int cols, int scale) throws IOException {
		super(DATA_FORM.DF_VECTOR);
		this.type = type;
		this.scale_ = scale;
		rowIndices = new int[rows];
		DATA_TYPE valueType = DATA_TYPE.valueOf(type.getValue() - 64);
		deserialize(valueType, in, rows, cols, this.scale_);
	}

	public BasicArrayVector(DATA_TYPE type, ExtendedDataInput in) throws IOException {
		super(DATA_FORM.DF_VECTOR);
		this.type = type;
		int rows = in.readInt();
		int cols = in.readInt();
		rowIndices = new int[rows];
		DATA_TYPE valueType = DATA_TYPE.valueOf(type.getValue() - 64);

		if (valueType == DATA_TYPE.DT_DECIMAL32 || valueType == DATA_TYPE.DT_DECIMAL64 || valueType == DATA_TYPE.DT_DECIMAL128) {
			this.scale_ = in.readInt();
			// ((AbstractVector)valueVec).setExtraParamForType(scale_);
		}

		deserialize(valueType, in, rows, cols, this.scale_);
	}

	public void deserialize(DATA_TYPE valueType,  ExtendedDataInput in, int rows, int cols, int scale) throws IOException {
		valueVec = BasicEntityFactory.instance().createVectorWithDefaultValue(valueType, 0, scale);

		ByteOrder bo = in.isLittleEndian() ? ByteOrder.LITTLE_ENDIAN : ByteOrder.BIG_ENDIAN;
		this.baseUnitLength_ = valueVec.getUnitLength();
		byte[] buf = new byte[4096];
		int rowsRead = 0;
		int rowsReadInBlock = 0;
		int prevIndex = 0;
		int totalBytes = 0;
		while (rowsRead < rows) {
			//read block header
			int blockRows = in.readUnsignedShort();
			int countBytes = in.readUnsignedByte();
			in.skipBytes(1);

			//read array of counts
			totalBytes = blockRows * countBytes;
			rowsReadInBlock = 0;
			int offset = 0;
			while(offset < totalBytes){
				int len = Math.min(4096, totalBytes - offset);
				in.readFully(buf, 0, len);
				int curRows = len / countBytes;
				if(countBytes == 1){
					for (int i = 0; i < curRows; i++){
						int curRowCells = Byte.toUnsignedInt(buf[i]);
						rowIndices[rowsRead + rowsReadInBlock + i] = prevIndex + curRowCells;
						prevIndex += curRowCells;
					}
				}
				else if(countBytes == 2){
					ByteBuffer byteBuffer = ByteBuffer.wrap(buf, 0, len).order(bo);
					for (int i = 0; i < curRows; i++){
						int curRowCells = Short.toUnsignedInt(byteBuffer.getShort(i * 2));
						rowIndices[rowsRead + rowsReadInBlock + i] = prevIndex + curRowCells;
						prevIndex += curRowCells;
					}
				}
				else {
					ByteBuffer byteBuffer = ByteBuffer.wrap(buf, 0, len).order(bo);
					for (int i = 0; i < curRows; i++){
						int curRowCells = byteBuffer.getInt(i * 4);
						rowIndices[rowsRead + rowsReadInBlock + i] = prevIndex + curRowCells;
						prevIndex += curRowCells;
					}
				}
				rowsReadInBlock += curRows;
				offset += len;
			}

			//read array of values
			int rowStart =  rowsRead == 0 ? 0 : rowIndices[rowsRead - 1];
			int valueCount = rowIndices[rowsRead + rowsReadInBlock - 1] - rowStart;
			Vector subVector = BasicEntityFactory.instance().createVectorWithDefaultValue(valueType, valueCount, scale);
			subVector.deserialize(0, valueCount, in);
			try {
				valueVec.Append(subVector);
			} catch (Exception e) {
				throw new RuntimeException(e);
			}

			rowsRead += rowsReadInBlock;
		}
		rowIndicesSize = rowIndices.length;
		capacity = rowIndices.length;
	}
	
	@Override
	public String getString(int index){
		StringBuilder sb = new StringBuilder("[");
		
		int startPosValueVec = index == 0 ? 0 : rowIndices[index - 1];
		int rows = rowIndices[index] - startPosValueVec;
		if(rows > 0)
			sb.append(valueVec.getString(startPosValueVec));
		for(int i=1; i<rows; ++i){
			sb.append(',');
			sb.append(valueVec.getString(startPosValueVec + i));
		}
		sb.append("]");
		return sb.toString();
	}
	
	@Override
	public Vector combine(Vector vector) {
		// TODO Auto-generated method stub
		return null;
	}

	@Override
	public Vector getSubVector(int[] indices) {
		// TODO Auto-generated method stub
		if (indices.length == 0)
			return null;
		List<Vector> value = new ArrayList<>();
		for (int i = 0; i < indices.length; i++){
			int start = indices[i] == 0 ? 0 : rowIndices[indices[i]-1];
			int end = rowIndices[indices[i]];
			int[] indexs = new int[end - start];
			for (int j = 0; j < indexs.length; j++){
				indexs[j] = start+j;
			}
			Vector subValue = valueVec.getSubVector(indexs);
			value.add(subValue);
		}
		try {
			return new BasicArrayVector(value);
		} catch (Exception e) {
			log.error(e.getMessage());
			e.printStackTrace();
			return null;
		}
	}

	@Override
	public int asof(Scalar value) {
		// TODO Auto-generated method stub
		return 0;
	}

	@Override
	public boolean isNull(int index) {
		// TODO Auto-generated method stub
		return false;
	}

	@Override
	public void setNull(int index) {
	}

	@Override
	public Entity get(int index) {
		int startPosValueVec = index == 0 ? 0 : rowIndices[index - 1];
		int rows = rowIndices[index] - startPosValueVec;
		DATA_TYPE valueType = DATA_TYPE.valueOf(type.getValue()-64);
		Vector value = BasicEntityFactory.instance().createVectorWithDefaultValue(valueType, rows, scale_);
		if (valueType == DATA_TYPE.DT_DECIMAL32){
			((BasicDecimal32Vector)value).setScale(scale_);
		} else if (valueType == DATA_TYPE.DT_DECIMAL64) {
			((BasicDecimal64Vector) value).setScale(scale_);
		} else if (valueType == DATA_TYPE.DT_DECIMAL128) {
			((BasicDecimal128Vector) value).setScale(scale_);
		}

		if (rows > 0){
			try {
				value.set(0, valueVec.get(startPosValueVec));
			}catch (Exception e){
				throw new RuntimeException("Failed to insert data, invalid data for "+valueVec.get(startPosValueVec)+" error "+ e.toString());
			}
		}
		for (int i = 1;i<rows;i++){
			try {
				value.set(i, valueVec.get(startPosValueVec + i));
			}catch (Exception e){
				throw new RuntimeException("Failed to insert data, invalid data for "+valueVec.get(startPosValueVec)+" error "+ e.toString());
			}
		}
		return value;
	}

	public Vector getVectorValue(int index) {
		return (Vector) get(index);
	}

	@Override
	public void set(int index, Entity value) throws Exception {
		if (!(value instanceof Vector)) {
			throw new RuntimeException("BasicArrayVector.set requires a Vector value, got: " + value.getClass().getSimpleName());
		}

		Vector newValue = (Vector) value;

		if (index < 0 || index >= capacity) {
			throw new IndexOutOfBoundsException("Index " + index + " out of bounds for capacity " + capacity);
		}

		DATA_TYPE currentValueType = DATA_TYPE.valueOf(type.getValue() - 64);
		if (newValue.getDataType() != currentValueType) {
			throw new RuntimeException("BasicArrayVector.set requires same data type. Expected: " + currentValueType + ", Got: " + newValue.getDataType());
		}

		if (currentValueType == DATA_TYPE.DT_DECIMAL32 || currentValueType == DATA_TYPE.DT_DECIMAL64 || currentValueType == DATA_TYPE.DT_DECIMAL128) {
			int currentScale = scale_;
			int newScale = ((AbstractVector)newValue).getExtraParamForType();
			if (currentScale != newScale) {
				throw new RuntimeException("BasicArrayVector.set requires same scale for decimal types. Expected: " + currentScale + ", Got: " + newScale);
			}
		}

		if (index >= rowIndicesSize) {
			for (int i = rowIndicesSize; i <= index; i++) {
				if (i == 0) {
					rowIndices[i] = 0;
				} else {
					rowIndices[i] = rowIndices[i-1];
				}
			}
			rowIndicesSize = index + 1;
		}

		int startPosValueVec = index == 0 ? 0 : rowIndices[index - 1];
		int currentRows = rowIndices[index] - startPosValueVec;
		int newRows = newValue.rows();

		if (currentRows > 0 && newRows != currentRows) {
			throw new RuntimeException("BasicArrayVector.set requires same length. Current: " + currentRows + ", New: " + newRows);
		}

		if (currentRows == 0) {
			try {
				if (newRows > 0) {
					for (int i = 0; i < newRows; i++) {
						valueVec.Append((Scalar) newValue.get(i));
					}
				}

				rowIndices[index] = startPosValueVec + newRows;

				for (int i = index + 1; i < rowIndicesSize; i++) {
					rowIndices[i] += newRows;
				}
			} catch (Exception e) {
				throw new RuntimeException("Failed to set data in BasicArrayVector: " + e.getMessage(), e);
			}
		} else {
			try {
				for (int i = 0; i < newRows; i++) {
					Scalar scalar = (Scalar) newValue.get(i);
					valueVec.set(startPosValueVec + i, scalar);
				}
			} catch (Exception e) {
				throw new RuntimeException("Failed to set data in BasicArrayVector: " + e.getMessage(), e);
			}
		}
	}

	@Override
	public void set(int index, Object value) {
		try {
			if (value instanceof Vector) {
				set(index, (Entity) value);
			} else {
				throw new RuntimeException("BasicArrayVector.set(Object) requires a Vector, got: " +
						(value == null ? "null" : value.getClass().getSimpleName()));
			}
		} catch (Exception e) {
			throw new RuntimeException("Error in BasicArrayVector.set(Object): " + e.getMessage(), e);
		}
	}

	@Override
	public Class<?> getElementClass() {
		return Entity.class;
	}

	@Override
	public void serialize(int start, int count, ExtendedDataOutput out) throws IOException {
		throw new RuntimeException("BasicAnyVector.serialize not supported.");
	}

	@Override
	public DATA_CATEGORY getDataCategory() {
		return Entity.DATA_CATEGORY.ARRAY;
	}

	@Override
	public DATA_TYPE getDataType() {
		return type;
	}

	@Override
	public int rows() {
		return rowIndicesSize;
	}

	@JsonIgnore
	@Override
	public int getUnitLength(){
		throw new RuntimeException("BasicArrayVector.getUnitLength() not supported.");
	}

	public void addRange(Object[] valueList) {
		throw new RuntimeException("ArrayVector not support addRange");
	}

	@Override
	public void Append(Scalar value) throws Exception{
		throw new RuntimeException("ArrayVector not support append scalar value");
	}

	@Override
	public void Append(Vector value) throws Exception{
		if (value.isVector()){
			int indexCount = rowIndicesSize;
			int prev = indexCount == 0? 0 : rowIndices[indexCount - 1];
			if (rowIndicesSize + 1 > capacity){
				rowIndices = Arrays.copyOf(rowIndices, Math.max(2, rowIndices.length * 2));
				capacity = rowIndices.length;
			}
			if (value.rows() != 0){
				rowIndices[rowIndicesSize] = prev + value.rows();
				valueVec.Append(value);
			}else {
				rowIndices[rowIndicesSize] = prev + 1;
				Scalar scalar = BasicEntityFactory.instance().createScalarWithDefaultValue(value.getDataType());
				scalar.setNull();
				valueVec.Append(scalar);
			}
			rowIndicesSize++;
		}else
			throw new RuntimeException("Append to arrayctor must be a vector. ");
	}

	@Override
	public void checkCapacity(int requiredCapacity) {
		throw new RuntimeException("BasicArrayVector not support checkCapacity.");
	}

	@Override
	public String getJsonString(int rowIndex) {
		StringBuilder sb = new StringBuilder("[");
		Vector value = (Vector) get(rowIndex);
		for (int j = 0; j < value.rows(); j++){
			sb.append(((Scalar)(value.get(j))).getJsonString());
			if (j != value.rows() - 1)
				sb.append(",");
		}
		sb.append("]");
		return sb.toString();
	}

	@Override
	public int serialize(int indexStart, int offect, int targetNumElement, NumElementAndPartial numElementAndPartial,  ByteBuffer out) throws IOException{
		numElementAndPartial.numElement = 0;
		numElementAndPartial.partial = 0;
		int byteSent = 0;
		NumElementAndPartial tempNumElementAndPartial = new NumElementAndPartial(0, 0);

		if(offect > 0)
		{
			int rowStart = indexStart > 0 ? rowIndices[indexStart - 1] : 0;
			int cellCount = rowIndices[indexStart] - rowStart;
			int cellCountToSerialize = Math.min(out.remaining() / baseUnitLength_, cellCount - offect);
			byteSent += valueVec.serialize(rowStart + offect, cellCount, numElementAndPartial.numElement, tempNumElementAndPartial, out);
			if(cellCountToSerialize < cellCount - offect)
			{
				numElementAndPartial.partial = offect + cellCountToSerialize;
				return byteSent;
			}
			else
			{
				--targetNumElement;
				++numElementAndPartial.numElement;
				++indexStart;
			}
		}

		int remainingBytes = out.remaining() - 4;
		int curCountBytes = 1;
		int maxCount = 255;
		int prestart = indexStart == 0 ? 0 : rowIndices[indexStart - 1];

		//one block can't exeed 65535 rows
		if (targetNumElement > 65535)
			targetNumElement = 65535;


		int i = 0;
		for(; i < targetNumElement && remainingBytes > 0; ++i)
		{
			int curStart = rowIndices[indexStart + i];
			int curCount = curStart - prestart;
			int oldCountBytes = curCountBytes;
			prestart = curStart;
			int byteRequired = 0;

			while(curCount > maxCount)
			{
				byteRequired += i * curCountBytes;
				curCountBytes *= 2;
				maxCount = Math.min(Integer.MAX_VALUE, (111 << (8 * curCountBytes)) - 1);
			}
			byteRequired += curCountBytes + curCount * baseUnitLength_;
			if(byteRequired > remainingBytes)
			{
				if(numElementAndPartial.numElement == 0)
				{
					numElementAndPartial.partial = (remainingBytes - curCountBytes) / baseUnitLength_;
					if(numElementAndPartial.partial <= 0)
					{
						numElementAndPartial.partial = 0;
					}
					else
					{
						++i;
						remainingBytes -= (curCountBytes + numElementAndPartial.partial * baseUnitLength_);
					}
				}
				else {
					if (oldCountBytes != curCountBytes)
						curCountBytes = oldCountBytes;
				}
				break;
			}
			else
			{
				remainingBytes -= byteRequired;
				++numElementAndPartial.numElement;
			}
		}

		if(i == 0)
			return byteSent;

		short rows = (short) i;
		out.putShort(rows);
		out.put((byte)curCountBytes);
		out.put((byte)0);
		byteSent += 4;

		//output array of counts
		prestart = indexStart == 0 ? 0 : rowIndices[indexStart - 1];
		for (int k = 0; k < rows; ++k)
		{
			int curStart = rowIndices[indexStart + k];
			int index = curStart - prestart;
			prestart = curStart;
			if (curCountBytes == 1)
				out.put((byte)index);
			else if (curCountBytes == 2)
				out.putShort((short)index);
			else
				out.putInt(index);
		}

		byteSent += curCountBytes * i;
		prestart = indexStart == 0 ? 0 : rowIndices[indexStart - 1];
		int count = (indexStart + numElementAndPartial.numElement == 0 ? 0 : rowIndices[indexStart + numElementAndPartial.numElement - 1]) + numElementAndPartial.partial - prestart;
		int bytes;
		bytes = valueVec.serialize(prestart, 0, count, tempNumElementAndPartial, out);
		return byteSent + bytes;
	}

	public void add(Object value) {
		throw new RuntimeException("ArrayVector not support add");
	}

	@Override
	public int getExtraParamForType(){
		return scale_;
	}

	@Override
	protected void writeVectorToOutputStream(ExtendedDataOutput out) throws IOException {
		// TODO Auto-generated method stub
		int indexCount = rowIndicesSize;
		int maxCount = 255;
		int countBytes = 1;
		int indicesPos = 0;
		int valuesOffect = 0;
		while (indicesPos < indexCount){
			int byteRequest = 4;
			int curRows = 0;
			int indiceCount = 1;
			while (byteRequest < 4096 && indicesPos + indiceCount - 1 < indexCount && indiceCount < 65536){// && indiceCount < 65536
				int curIndiceOffect = indicesPos + indiceCount - 1;
				int index = curIndiceOffect == 0 ? rowIndices[curIndiceOffect] : rowIndices[curIndiceOffect] - rowIndices[curIndiceOffect - 1];
				while(index > maxCount)
				{
					byteRequest += (indiceCount - 1) * countBytes;
					countBytes *= 2;
					maxCount = Math.min(Integer.MAX_VALUE, (111 << (8 * countBytes)) - 1);
				}
				curRows += index;
				indiceCount++;
				byteRequest += countBytes + baseUnitLength_ * index;
			}
			indiceCount --;
			out.writeShort(indiceCount);
			out.writeByte(countBytes);
			out.writeByte(0);
			for (int i = 0; i < indiceCount; i++){
				int index = indicesPos + i == 0 ? rowIndices[indicesPos + i] : rowIndices[indicesPos + i] - rowIndices[indicesPos + i - 1];
				if (countBytes == 1)
					out.writeByte(index);
				else if (countBytes == 2)
					out.writeShort(index);
				else
					out.writeInt(index);
			}
			valueVec.serialize(valuesOffect, curRows, out);
			indicesPos += indiceCount;
			valuesOffect += curRows;
		}
	}

	@Override
	public int columns() {
		return valueVec.rows();
	}

	public int getScale(){
		return scale_;
	}
}
