package timber.log;

/**
 * A {@link Tree Tree} for debug builds. Automatically infers the tag from the calling class.
 */
public class DebugTree extends Tree {

    private static final int MAX_LOG_LENGTH = 4000;
    private static final int CALL_STACK_INDEX = 8;
    private final Level MIN_LEVEL;

    /**
     * Create a DebugTree with minimum level set to {@link Level#TRACE TRACE}.
     */
    public DebugTree() {
        this(Level.TRACE);
    }

    /**
     * Create a DebugTree with minimum level set to {@code level}.
     */
    public DebugTree(Level level) {
        MIN_LEVEL = level;
    }

    @Override
    protected boolean isLoggeable(Level level) {
        return Level.compare(MIN_LEVEL, level) >= 0;
    }

    @Override
    protected String getTag() {
        String tag = super.getTag();
        if (tag != null) return tag;

        StackTraceElement[] stackTrace = Thread.currentThread().getStackTrace();
        if (stackTrace.length <= CALL_STACK_INDEX) {
            throw new IllegalStateException(
                    "Not enough elements in stacktrace. Are you using synthetic stacktrace ?");
        }
        return createStackElementTag(stackTrace[CALL_STACK_INDEX]);
    }

    /**
     * Break up {@code message} into maximum-length chunks (if needed) and send to either
     * {@link System#out stdout} or
     * {@link System#err stderr} for logging.
     * <p>
     * {@inheritDoc}
     */
    @Override
    protected void log(Level level, String tag, String message, Throwable t) {
        if (!isLoggeable(level)) return;
        if (message.length() < MAX_LOG_LENGTH) {
            System.out.println(formatLine(level, tag, message));
            return;
        }

        for (int i = 0, length = message.length(); i < length; i++) {
            int newline = message.indexOf('\n', i);
            newline = newline != -1 ? newline : length;
            do {
                int end = Math.min(newline, i + MAX_LOG_LENGTH);
                System.out.println(formatLine(level, tag, message.substring(i, end)));
                i = end;
            } while (i < newline);
        }

    }

}
