package timber.log;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.PrintWriter;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class FileTree extends Tree {

    private final PrintWriter PRINT_WRITER;
    private static final Pattern ANONYMOUS_CLASS = Pattern.compile("(\\$\\d+)+$");
    private final Level MIN_LEVEL;

    @Override
    protected String getTag() {
        String tag = super.getTag();
        if (tag != null) return tag;

        StackTraceElement[] stackTrace = Thread.currentThread().getStackTrace();
        if (stackTrace.length <= 8) {
            throw new IllegalStateException(
                    "Not enough elements in stacktrace. Are you using an obfuscator?");
        }
        return createStackElementTag(stackTrace[8]);
    }

    @Override
    protected boolean isLoggeable(Level level) {
        return Level.compare(MIN_LEVEL, level) >= 0;
    }

    /**
     * Create a new FileTree instance.
     *
     * @throws IOException
     */
    public FileTree(Level level, File logFile, boolean append) throws IOException {
        if (level == null) level = Level.TRACE;
        MIN_LEVEL = level;
        if (logFile == null) logFile = new File("timber.log");
        if (!logFile.exists()) logFile.createNewFile();
        if (!append || logFile.isDirectory()) {
            logFile.delete();
            logFile.createNewFile();
        }
        PRINT_WRITER = new PrintWriter(new FileOutputStream(logFile));
    }

    /**
     * Create a new FileTree instance.
     *
     * @throws IOException
     */
    public FileTree(Level level, File logFile) throws IOException {
        this(level, logFile, false);
    }

    @Override
    protected void log(Level level, String tag, String message, Throwable t) {
        if (!isLoggeable(level)) return;
        PRINT_WRITER.println(formatLine(level, tag, message));
        PRINT_WRITER.flush();
    }

    protected String createStackElementTag(StackTraceElement element) {
        String tag = element.getClassName();
        Matcher m = ANONYMOUS_CLASS.matcher(tag);
        if (m.find()) {
            tag = m.replaceAll("");
        }
        return tag.substring(tag.lastIndexOf('.') + 1).replaceAll("\\$", ".");
    }

}
