package timber.log;

import java.io.PrintWriter;
import java.io.StringWriter;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * A facade for handling logging calls. Install instances via {@link Timber#plant Timber.plant()}.
 */
public abstract class Tree {

    final ThreadLocal<String> TAG = new ThreadLocal<>();
    private static final Pattern ANONYMOUS_CLASS = Pattern.compile("(\\$\\d+)+$");

    protected final String NL = System.lineSeparator();

    protected String getTag() {
        String tag = TAG.get();
        if (tag != null)
            TAG.remove();
        return tag;
    }

    /**
     * Log a trace message with optional format args.
     */
    public void trace(String message, Object... args) {
        prepareLog(Level.TRACE, null, message, args);
    }

    /**
     * Log a trace exception and a message with optional format args.
     */
    public void trace(Throwable t, String message, Object... args) {
        prepareLog(Level.TRACE, t, message, args);
    }

    /**
     * Log a debug message with optional format args.
     */
    public void debug(String message, Object... args) {
        prepareLog(Level.DEBUG, null, message, args);
    }

    /**
     * Log a debug exception and a message with optional format args.
     */
    public void debug(Throwable t, String message, Object... args) {
        prepareLog(Level.DEBUG, t, message, args);
    }

    /**
     * Log an info message with optional format args.
     */
    public void info(String message, Object... args) {
        prepareLog(Level.INFO, null, message, args);
    }

    /**
     * Log an info exception and a message with optional format args.
     */
    public void info(Throwable t, String message, Object... args) {
        prepareLog(Level.INFO, t, message, args);
    }

    /**
     * Log a warn message with optional format args.
     */
    public void warn(String message, Object... args) {
        prepareLog(Level.WARN, null, message, args);
    }

    /**
     * Log a warn exception and a message with optional format args.
     */
    public void warn(Throwable t, String message, Object... args) {
        prepareLog(Level.WARN, t, message, args);
    }

    /**
     * Log an error message with optional format args.
     */
    public void error(String message, Object... args) {
        prepareLog(Level.ERROR, null, message, args);
    }

    /**
     * Log an error exception and a message with optional format args.
     */
    public void error(Throwable t, String message, Object... args) {
        prepareLog(Level.ERROR, t, message, args);
    }

    /**
     * Log a fatal message with optional format args.
     */
    public void fatal(String message, Object... args) {
        prepareLog(Level.FATAL, null, message, args);
    }

    /**
     * Log a fatal exception and a message with optional format args.
     */
    public void fatal(Throwable t, String message, Object... args) {
        prepareLog(Level.FATAL, t, message, args);
    }

    /**
     * Log at {@code level} a message with optional format args.
     */
    public void log(Level level, String message, Object... args) {
        prepareLog(level, null, message, args);
    }

    /**
     * Log at {@code level} an exception and a message with optional format args.
     */
    public void log(Level level, Throwable t, String message, Object... args) {
        prepareLog(level, t, message, args);
    }

    /**
     * Return whether a message at {@code level} should be logged.
     */
    protected boolean isLoggeable(Level level) {
        return true;
    }

    protected void prepareLog(Level level, Throwable t, String message, Object... args) {
        if (!isLoggeable(level)) return;
        if (message != null && message.length() == 0) message = null;
        if (message == null) {
            if (t == null) return;
            message = getStackTraceString(t);
        } else {
            if (args.length > 0) message = String.format(message, args);
            if (t != null) message += NL + getStackTraceString(t);
        }
        log(level, getTag(), message, t);
    }

    protected final String getStackTraceString(Throwable t) {
        StringWriter stringWriter = new StringWriter(256);
        PrintWriter printWriter = new PrintWriter(stringWriter, false);
        t.printStackTrace(printWriter);
        printWriter.flush();
        printWriter.close();
        return stringWriter.toString();
    }

    /**
     * Extract the tag which should be used for the message from the {@code element}. By default
     * this will use the class name without any anonymous class suffixes (e.g., {@code Foo$1}
     * becomes {@code Foo}).
     * <p>
     * Note: This will not be called if a {@linkplain Timber#tag(String) manual tag} was specified.
     */
    protected String createStackElementTag(StackTraceElement element) {
        String tag = element.getClassName();
        Matcher m = ANONYMOUS_CLASS.matcher(tag);
        if (m.find()) {
            tag = m.replaceAll("");
        }
        return tag.substring(tag.lastIndexOf('.') + 1).replaceAll("\\$", ".");
    }

    private static final SimpleDateFormat SIMPLE_DATE_FORMAT = new SimpleDateFormat("dd/MM/yyyy HH:mm:ss.SSS");

    protected String formatLine(Level level, String tag, String message) {
        return String.format("[%s][%s/%s]: %s", SIMPLE_DATE_FORMAT.format(new Date()), level.getLevelName().toUpperCase(), tag, message);
    }

    public final void setDateFormatPattern(String pattern) {
        SIMPLE_DATE_FORMAT.applyPattern(pattern);
    }

    /**
     * Write a log message to its destination. Called for all level-specific methods by default.
     *
     * @param level   Log level. See {@link Level} for constants.
     * @param tag     Explicit or inferred tag. May be {@code null}.
     * @param message Formatted log message. May be {@code null}, but then {@code t} will not be.
     * @param t       Accompanying exceptions. May be {@code null}, but then {@code message} will not be.
     */
    protected abstract void log(Level level, String tag, String message, Throwable t);

}
