/*
 * Decompiled with CFR 0.152.
 */
package com.github.isaichkindanila.command.framework;

import com.github.isaichkindanila.command.framework.CommandRunner;
import com.github.isaichkindanila.command.framework.ExternalConfig;
import com.github.isaichkindanila.command.framework.FixedFileHandler;
import com.github.isaichkindanila.command.framework.FrameworkConfig;
import com.github.isaichkindanila.command.framework.predefined.ConfigCommand;
import com.github.isaichkindanila.command.framework.predefined.HelpCommand;
import com.github.isaichkindanila.command.framework.stuff.Command;
import com.github.isaichkindanila.command.framework.stuff.ConfigHandler;
import com.github.isaichkindanila.command.framework.stuff.ConsoleIO;
import com.github.isaichkindanila.command.framework.stuff.StringDistanceAlgorithm;
import com.github.isaichkindanila.command.framework.util.ShouldNeverHappen;
import com.github.isaichkindanila.command.framework.util.cmd.CommandWrapper;
import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.DataInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Locale;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import java.util.function.IntUnaryOperator;
import java.util.logging.Level;
import java.util.logging.LogManager;
import java.util.logging.Logger;

public final class CommandFramework {
    private static final String GET_HELP = "use \"help\" command to get list of all available commands";
    private static final String NOT_FOUND_TEMPLATE = "command \"%s\" not found";
    private static final String SIMILAR_TEMPLATE = "perhaps you meant %s?";
    private static final Logger LOGGER = Logger.getLogger(CommandFramework.class.getName());
    private static List<CommandWrapper> wrappers;
    private static StringDistanceAlgorithm stringDistanceAlgorithm;
    private static IntUnaryOperator maxSimilarStringDistance;
    private static ConsoleIO consoleIO;
    private static File dataDirectory;
    private static BiConsumer<CommandWrapper, Command> showDetailedInformation;
    private static ExternalConfig externalConfig;
    private static File configFile;

    private CommandFramework() {
    }

    private static void initLogging(FrameworkConfig config) {
        Logger globalLogger = Logger.getLogger("");
        Logger frameworkLogger = Logger.getLogger(CommandFramework.class.getPackage().getName());
        LogManager.getLogManager().reset();
        globalLogger.addHandler(consoleIO.createHandler(config.consoleLoggingLevel()));
        frameworkLogger.setLevel(config.loggingLevel());
        try {
            File global = CommandFramework.getDataFile("logs", "global.log");
            File framework = CommandFramework.getDataFile("logs", "framework.log");
            globalLogger.addHandler(new FixedFileHandler(global, config.fileLogFormatter()));
            frameworkLogger.addHandler(new FixedFileHandler(framework, config.fileLogFormatter()));
        }
        catch (Exception e) {
            LOGGER.log(Level.WARNING, "error on initializing logging", e);
        }
    }

    private static void initExternalConfig(FrameworkConfig config) {
        Function<ConfigHandler, ExternalConfig> constructor = config.externalConfigConstructor();
        if (constructor == null) {
            LOGGER.info("external config constructor is not provided");
            return;
        }
        ConfigHandler configHandler = config.configHandler();
        if (configHandler == null) {
            LOGGER.warning("external config constructor is provided but ConfigHandler is not");
            return;
        }
        String configName = "config." + configHandler.extension();
        configFile = CommandFramework.getDataFile(configName);
        CommandFramework.runAndLog("parsed config", () -> {
            if (configFile.exists()) {
                try (BufferedInputStream in = new BufferedInputStream(new FileInputStream(configFile));){
                    configHandler.parse(in);
                }
                catch (FileNotFoundException e) {
                    LOGGER.log(Level.WARNING, "config file \"" + configName + "\" exists but not found", e);
                }
                catch (IOException e) {
                    throw new RuntimeException(e);
                }
            }
            externalConfig = (ExternalConfig)constructor.apply(configHandler);
        });
    }

    private static void init(FrameworkConfig config) {
        stringDistanceAlgorithm = config.stringDistanceAlgorithm();
        maxSimilarStringDistance = config.maxSimilarStringDistance();
        consoleIO = config.consoleIO();
        dataDirectory = config.dataDirectory();
        showDetailedInformation = config.showDetailedInformation();
        CommandFramework.initLogging(config);
        CommandFramework.initExternalConfig(config);
        CommandFramework.runAndLog("parsed commands", () -> CommandFramework.parseCommandWrappers(config));
    }

    private static void parseCommandWrappers(FrameworkConfig config) {
        InputStream inputStream = CommandFramework.class.getResourceAsStream("commands.bin");
        if (inputStream == null) {
            LOGGER.log(Level.SEVERE, "command info file not found");
            System.exit(-1);
        }
        DataInputStream in = new DataInputStream(new BufferedInputStream(inputStream));
        try {
            int count = in.readInt();
            wrappers = new ArrayList<CommandWrapper>(count + 2);
            LOGGER.fine("wrappers count: " + count);
            for (int i = 0; i < count; ++i) {
                wrappers.add(new CommandWrapper(in));
            }
            wrappers.add(0, HelpCommand.WRAPPER);
            if (externalConfig != null) {
                wrappers.add(ConfigCommand.getWrapper(config));
            }
        }
        catch (IOException e) {
            LOGGER.log(Level.SEVERE, "failed to parse command info file", e);
            System.exit(-1);
        }
    }

    public static CommandWrapper[] getAllWrappers() {
        return wrappers.toArray(new CommandWrapper[0]);
    }

    public static ConsoleIO getConsoleIO() {
        return consoleIO;
    }

    public static ExternalConfig getExternalConfig() {
        return externalConfig;
    }

    public static File getConfigFile() {
        return configFile;
    }

    public static void saveExternalConfig() {
        if (externalConfig == null) {
            return;
        }
        LOGGER.fine("saving config to " + configFile.getAbsolutePath());
        CommandFramework.runAndLog("config saved", () -> {
            try (BufferedOutputStream out = new BufferedOutputStream(new FileOutputStream(configFile));){
                externalConfig.save(out);
            }
            catch (FileNotFoundException e) {
                LOGGER.log(Level.WARNING, "failed to open '" + configFile + "'", e);
            }
            catch (IOException e) {
                LOGGER.log(Level.WARNING, "failed to save external config", e);
            }
        });
    }

    public static File getDataFile(String ... pathSegments) {
        if (pathSegments == null) {
            throw new NullPointerException("pathSegments is null");
        }
        if (pathSegments.length == 0) {
            throw new IllegalArgumentException("pathSegments is empty");
        }
        for (int i = 0; i < pathSegments.length; ++i) {
            String segment = pathSegments[i];
            if (segment == null) {
                throw new NullPointerException("pathSegments[" + i + "] is null");
            }
            if (!segment.isEmpty()) continue;
            throw new IllegalArgumentException("pathSegments[" + i + "] is empty");
        }
        File file = new File(dataDirectory, String.join((CharSequence)File.separator, pathSegments));
        File parent = file.getParentFile();
        if (!parent.exists() && !parent.mkdirs()) {
            throw new IllegalStateException("failed to create directory: " + parent.getAbsolutePath());
        }
        LOGGER.finer("returning file: " + file.getAbsolutePath());
        return file;
    }

    public static Optional<CommandWrapper> getWrapperForName(String name) {
        for (CommandWrapper wrapper : wrappers) {
            for (String s : wrapper.getNames()) {
                if (!s.equals(name)) continue;
                LOGGER.log(Level.FINE, "command for \"{0}\" is {1}", new String[]{name, wrapper.getClassName()});
                return Optional.of(wrapper);
            }
        }
        LOGGER.log(Level.FINE, "command for \"{0}\" not found", name);
        return Optional.empty();
    }

    public static Command commandFromWrapper(CommandWrapper wrapper) {
        try {
            Class<Command> commandClass = Class.forName(wrapper.getClassName()).asSubclass(Command.class);
            return commandClass.newInstance();
        }
        catch (Exception e) {
            throw new IllegalStateException(e);
        }
    }

    public static void showDetailedInformation(CommandWrapper wrapper, Command command) {
        showDetailedInformation.accept(wrapper, command);
    }

    public static boolean showDetailedInformation(CommandWrapper wrapper) {
        try {
            showDetailedInformation.accept(wrapper, CommandFramework.commandFromWrapper(wrapper));
            return true;
        }
        catch (IllegalStateException e) {
            return false;
        }
    }

    public static boolean showDetailedInformation(Command command) {
        try {
            String className = command.getClass().getName();
            Optional<CommandWrapper> optional = wrappers.stream().filter(w -> w.getClassName().equals(className)).findAny();
            if (optional.isPresent()) {
                showDetailedInformation.accept(optional.get(), command);
                return true;
            }
            return false;
        }
        catch (Exception e) {
            return false;
        }
    }

    public static boolean showDetailedInformation(String name) {
        Optional<CommandWrapper> optional = CommandFramework.getWrapperForName(name);
        if (optional.isPresent()) {
            return CommandFramework.showDetailedInformation(optional.get());
        }
        return false;
    }

    private static List<String> getSimilarCommandNames(String name) {
        ArrayList<String> similar = new ArrayList<String>();
        int minDistance = maxSimilarStringDistance.applyAsInt(name.length());
        LOGGER.logp(Level.FINER, CommandFramework.class.getName(), "getSimilarCommandNames(\"" + name + "\")", "max similar distance: {0}", minDistance);
        for (CommandWrapper wrapper : wrappers) {
            for (String s : wrapper.getNames()) {
                int distance = stringDistanceAlgorithm.compute(s, name);
                if (distance < minDistance) {
                    minDistance = distance;
                    similar.clear();
                    similar.add(s);
                    continue;
                }
                if (distance != minDistance) continue;
                similar.add(s);
            }
        }
        String tmp = "distance = " + minDistance + ", commands = [%s]";
        LOGGER.logp(Level.FINER, CommandFramework.class.getName(), "getSimilarCommandNames(\"" + name + "\")", () -> {
            if (similar.size() == 0) {
                return "similar commands not found";
            }
            return String.format(tmp, String.join((CharSequence)", ", similar));
        });
        return similar;
    }

    public static void showNotFound(String name) {
        consoleIO.printLine(String.format(NOT_FOUND_TEMPLATE, name));
        List<String> similar = CommandFramework.getSimilarCommandNames(name);
        if (!similar.isEmpty()) {
            CharSequence[] arr = (String[])similar.stream().map(s -> '\"' + s + '\"').toArray(String[]::new);
            String str = String.join((CharSequence)", ", arr);
            consoleIO.printLine(String.format(SIMILAR_TEMPLATE, str));
        }
        consoleIO.printLine(GET_HELP);
    }

    static void runAndLog(String finishedMsg, Runnable runnable) {
        long start = System.nanoTime();
        runnable.run();
        long end = System.nanoTime();
        LOGGER.fine(() -> String.format(Locale.US, "%s in %.3f seconds", finishedMsg, (double)(end - start) * 1.0E-9));
    }

    public static void run(String[] args, FrameworkConfig config, Runnable appInit) {
        try {
            CommandFramework.runAndLog("framework initialization finished", () -> CommandFramework.init(config));
            if (appInit != null) {
                CommandFramework.runAndLog("app initialization finished", appInit);
            }
            if (args.length == 0) {
                LOGGER.info("no arguments passed");
                consoleIO.printLine(GET_HELP);
                return;
            }
            String command = args[0];
            List<String> argList = Arrays.asList(args).subList(1, args.length);
            LOGGER.info(String.format("command = \"%s\", args = [%s]", command, String.join((CharSequence)", ", argList)));
            Optional<CommandWrapper> optional = CommandFramework.getWrapperForName(command);
            if (optional.isPresent()) {
                CommandRunner commandRunner = new CommandRunner(optional.get());
                commandRunner.run(argList);
            } else {
                CommandFramework.showNotFound(command);
            }
        }
        catch (ShouldNeverHappen e) {
            LOGGER.log(Level.SEVERE, "if you see this, something went HORRIBLY wrong", e);
        }
        catch (Throwable t) {
            LOGGER.log(Level.SEVERE, "unexpected error occurred", t);
        }
    }
}

