package cdc.issues.impl;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import cdc.issues.IssueId;
import cdc.issues.Project;
import cdc.issues.answers.IssueAnswer;
import cdc.issues.rules.DataSource;
import cdc.util.lang.Checks;

public class ProjectImpl implements Project {
    private String name;
    private ProfileImpl profile;
    private final List<SnapshotImpl> snapshots = new ArrayList<>();
    private final Map<IssueId, IssueAnswerImpl> map = new HashMap<>();
    private final Set<IssueAnswerImpl> answers = new HashSet<>();
    private final List<DataSource<?>> dataSources = new ArrayList<>();

    public ProjectImpl(String name) {
        this.name = name;
    }

    public ProjectImpl setName(String name) {
        this.name = name;
        return this;
    }

    public ProjectImpl setProfile(ProfileImpl profile) {
        Checks.isNotNull(profile, "profile");
        this.profile = profile;
        return this;
    }

    void addSnapshot(SnapshotImpl snapshot) {
        Checks.isNotNull(snapshot, "snapshot");
        snapshots.add(snapshot);
    }

    public SnapshotImpl createSnapshot() {
        return new SnapshotImpl(this);
    }

    public ProjectImpl addAnswer(IssueAnswerImpl answer) {
        Checks.isNotNull(answer, "answer");
        final IssueAnswer current = map.get(answer.getIssueId());
        answers.remove(current);
        answers.add(answer);
        map.put(answer.getIssueId(), answer);
        return this;
    }

    public ProjectImpl addAnswers(Collection<IssueAnswerImpl> answers) {
        for (final IssueAnswerImpl answer : answers) {
            addAnswer(answer);
        }
        return this;
    }

    public ProjectImpl addDataSource(DataSource<?> dataSource) {
        Checks.isNotNull(dataSource, "dataSource");
        dataSources.add(dataSource);
        return this;
    }

    @Override
    public String getName() {
        return name;
    }

    @Override
    public Optional<ProfileImpl> getProfile() {
        return Optional.ofNullable(profile);
    }

    @Override
    public Iterable<DataSource<?>> getDataSources() {
        return dataSources;
    }

    @Override
    public List<SnapshotImpl> getSnapshots() {
        return snapshots;
    }

    @Override
    public Set<IssueAnswerImpl> getAnswers() {
        return answers;
    }

    @Override
    public Optional<IssueAnswerImpl> getAnswer(IssueId id) {
        return Optional.ofNullable(map.get(id));
    }
}