package cdc.util.cli;

import java.io.BufferedInputStream;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.function.Function;
import java.util.stream.Stream;

import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.DefaultParser;
import org.apache.commons.cli.HelpFormatter;
import org.apache.commons.cli.Option;
import org.apache.commons.cli.OptionGroup;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;
import org.apache.logging.log4j.Level;
import org.apache.logging.log4j.Logger;

import cdc.util.files.Files;
import cdc.util.lang.Checks;

/**
 * Utility that can be used to create main programs.
 * <p>
 * The purpose is to force the adoption of a common pattern for such classes.
 * <p>
 * Some options are automatically defined and handled:
 * <ul>
 * <li><b>help:</b> if passed, help is printed and program is exited.
 * <li><b>version:</b> if passed, version is printed and program is exited.
 * <li><b>args-list &lt;FILE&gt;:</b> if passed, command line arguments
 * are read from the designated file and added to other command line arguments.<br>
 * This is useful when command line is too long.
 * </ul>
 *
 * @author Damien Carbonne
 *
 * @param <A> MainArgs type.
 * @param <R> Returned type.
 */
public abstract class AbstractMainSupport<A, R> {
    private final Class<?> mainClass;
    private final Logger logger;

    public static final String DEFAULT_PARTS_SEPARATOR = "::";

    public static final String ARGS_FILE = "args-file";
    public static final String CHARSET = "charset";
    public static final String DRIVER = "driver";
    public static final String HELP = "help";
    public static final String INPUT = "input";
    public static final String INPUT_DIR = "input-dir";
    public static final String OUTPUT = "output";
    public static final String OUTPUT_DIR = "output-dir";
    public static final String PASSWORD = "password";
    public static final String PATH = "path";
    public static final String PREFIX = "prefix";
    public static final String TMP_DIR = "tmp-dir";
    public static final String URL = "url";
    public static final String USER = "user";
    public static final String VERSION = "version";

    protected AbstractMainSupport(Class<?> mainClass,
                                  Logger logger) {
        this.mainClass = mainClass;
        this.logger = logger;
    }

    protected void addStandardOptions(Options options) {
        final boolean addArgsFileOption = hasMultipleValuesOption(options);

        final Option help = Option.builder("h")
                                  .longOpt(HELP)
                                  .desc("Prints this help and exits.")
                                  .build();
        options.addOption(help);

        if (getVersion() != null) {
            final Option version = Option.builder("v")
                                         .longOpt(VERSION)
                                         .desc("Prints version and exists.")
                                         .build();
            options.addOption(version);

            final OptionGroup group = new OptionGroup();
            group.addOption(help);
            group.addOption(version);
            options.addOptionGroup(group);
        }
        if (addArgsFileOption) {
            options.addOption(Option.builder()
                                    .longOpt(ARGS_FILE)
                                    .desc("Name of the file from which options can be read.\n"
                                            + "There must be one argument per line.")
                                    .hasArg()
                                    .build());
        }
    }

    protected final Logger getLogger() {
        return logger;
    }

    /**
     * Returns the version.
     * <p>
     * If {@code null} is returned (default implementation), no version option is available.
     *
     * @return The version.
     */
    protected String getVersion() {
        return null;
    }

    /**
     * Returns the help header.
     * <p>
     * If {@code null} is returned (default implementation), no help header is printed.
     *
     * @return The help header.
     */
    protected String getHelpHeader() {
        return null;
    }

    /**
     * Returns the help footer.
     * <p>
     * If {@code null} is returned (default implementation), no help footer is printed.
     *
     * @return The help footer.
     */
    protected String getHelpFooter() {
        return null;
    }

    /**
     * Creates specific options and add them to an options collection.
     * <p>
     * The standard options must not be added.
     *
     * @param options The options.
     */
    protected abstract void addSpecificOptions(Options options);

    /**
     * Analyzes the command line.
     * <p>
     * These options are already handled:
     * <ul>
     * <li>help
     * <li>version
     * <li>args-file
     * </ul>
     *
     * @param cl The command line.
     * @return A MainARgs instance.
     * @throws ParseException When command line parsing has a problem.
     */
    protected abstract A analyze(CommandLine cl) throws ParseException;

    /**
     * Executes the main program.
     *
     * @param margs The main arguments.
     * @return The optional result.
     * @throws Exception When program has a problem.
     */
    protected abstract R execute(A margs) throws Exception;

    public static <E extends Enum<E> & OptionEnum> void addNoArgOptions(Options options,
                                                                        Class<E> enumClass) {
        for (final E e : enumClass.getEnumConstants()) {
            options.addOption(Option.builder()
                                    .longOpt(e.getName())
                                    .desc(e.getDescription())
                                    .build());
        }
    }

    @SafeVarargs
    public static <E extends Enum<E> & OptionEnum> void createGroup(Options options,
                                                                    E... values) {
        final OptionGroup group = new OptionGroup();
        for (final E value : values) {
            group.addOption(options.getOption(value.getName()));
        }
        options.addOptionGroup(group);
    }

    @FunctionalInterface
    public static interface Maskable<E extends Enum<E>> {
        public void setEnabled(E e,
                               boolean enabled);
    }

    public static <E extends Enum<E> & OptionEnum> void setMask(CommandLine cl,
                                                                Class<E> enumClass,
                                                                Maskable<E> maskable) {
        for (final E e : enumClass.getEnumConstants()) {
            maskable.setEnabled(e, cl.hasOption(e.getName()));
        }
    }

    private static String[] loadArgs(File file) throws IOException {
        try (final InputStream in = new BufferedInputStream(new FileInputStream(file));
                final BufferedReader reader = new BufferedReader(new InputStreamReader(in))) {
            final List<String> list = new ArrayList<>();
            String line = null;
            while ((line = reader.readLine()) != null) {
                if (!line.isEmpty()) {
                    list.add(line);
                }
            }
            return list.toArray(new String[list.size()]);
        }
    }

    /**
     * Default main program.
     * <p>
     * This does the following things:
     * <ol>
     * <li>Build options.
     * <li>Parse the command line strings and creates a CommandLine instance.
     * <li>Analyze the CommandLine instance and create a MainArgs instance.
     * <li>Calls execute with the MainARgs instance.
     * </ol>
     *
     * @param args The command line arguments.
     * @return The optional result.
     */
    public R main(String[] args) {
        // Build all options as declared
        final Options allOptions = buildOptions();
        // Build all options as optional
        final Options optionalOptions = getOptionsAsOptional();
        final DefaultParser parser = new DefaultParser();
        try {
            final String[] effectiveArgs;
            // Analysis of optional args file
            final CommandLine cl0 = parser.parse(optionalOptions, args);
            final CommandLine cl1;
            final File argsFile = getValueAsNullOrExistingFile(cl0, ARGS_FILE, null);
            if (argsFile != null) {
                final String[] fileArgs = loadArgs(argsFile);
                // Concatenate file args followed by normal args
                effectiveArgs = Stream.concat(Arrays.stream(fileArgs), Arrays.stream(args))
                                      .toArray(String[]::new);
                cl1 = parser.parse(optionalOptions, effectiveArgs);
            } else {
                effectiveArgs = args;
                cl1 = cl0;
            }

            // First parsing and processing
            if (cl1.hasOption(HELP)) {
                printHelp(allOptions, null);
                return null;
            }
            if (cl1.hasOption(VERSION)) {
                printVersion();
                return null;
            }

            // Normal parsing and processing
            final CommandLine cl2 = parser.parse(allOptions, effectiveArgs);
            final A margs = analyze(cl2);
            try {
                return execute(margs);
            } catch (final Exception e) {
                logger.catching(e);
                return null;
            }
        } catch (final Exception e) {
            printHelp(allOptions, e);
            return null;
        }
    }

    /**
     * @param options The options.
     * @return {@code true} if {@code options} contains an options having multiple values.
     */
    private static boolean hasMultipleValuesOption(Options options) {
        for (final Option option : options.getOptions()) {
            if (option.hasArgs()) {
                return true;
            }
        }
        return false;
    }

    /**
     * Creates options, add standard and specific options and returns them.
     *
     * @return The options.
     */
    protected Options buildOptions() {
        final Options options = new Options();
        addSpecificOptions(options);
        addStandardOptions(options);
        return options;
    }

    /**
     * @return Build all options and make them optional.
     */
    protected Options getOptionsAsOptional() {
        final Options result = new Options();
        for (final Option option : buildOptions().getOptions()) {
            option.setRequired(false);
            result.addOption(option);
        }
        return result;
    }

    protected void printHelp(Options options,
                             Exception e) {
        final HelpFormatter formatter = new HelpFormatter();
        final StringBuilder header = new StringBuilder();
        if (getHelpHeader() != null) {
            header.append("\n");
            header.append(getHelpHeader());
        }
        header.append("\n");
        if (e != null && e.getMessage() != null) {
            header.append(e.getMessage());
            header.append("\n");
        }

        formatter.printHelp(mainClass.getSimpleName(),
                            header.toString(),
                            options,
                            getHelpFooter(),
                            true);
        if (e != null) {
            if (e instanceof ParseException) {
                if (logger.isEnabled(Level.DEBUG)) {
                    logger.catching(e);
                }
            } else {
                logger.catching(e);
            }
        }
    }

    protected void printVersion() {
        System.out.println(getVersion());
    }

    private static ParseException invalidArg(CommandLine cl,
                                             String opt,
                                             String message) {
        final StringBuilder builder = new StringBuilder();
        builder.append("Invalid arg for ");
        builder.append(opt);
        builder.append(" option: '");
        builder.append(cl.getOptionValue(opt));
        builder.append("'");
        if (message != null) {
            builder.append(" ");
            builder.append(message);
        }
        return new ParseException(builder.toString());
    }

    public static <T> T getValue(CommandLine cl,
                                 String opt,
                                 T def,
                                 Function<String, T> converter) throws ParseException {
        if (cl.hasOption(opt)) {
            try {
                return converter.apply(cl.getOptionValue(opt));
            } catch (final Exception e) {
                throw invalidArg(cl, opt, e.getMessage());
            }
        } else {
            return def;
        }
    }

    public static void fillValues(CommandLine cl,
                                  String opt,
                                  Collection<String> values) {
        if (cl.hasOption(opt)) {
            for (final String s : cl.getOptionValues(opt)) {
                values.add(s);
            }
        }
    }

    public static <T> void fillValues(CommandLine cl,
                                      String opt,
                                      Collection<T> values,
                                      Function<String, T> converter) throws ParseException {
        Checks.isNotNull(values, "values");
        if (cl.hasOption(opt)) {
            try {
                for (final String s : cl.getOptionValues(opt)) {
                    values.add(converter.apply(s));
                }
            } catch (final Exception e) {
                throw invalidArg(cl, opt, e.getMessage());
            }
        }
    }

    public static <T> List<T> getValues(CommandLine cl,
                                        String opt,
                                        Function<String, T> converter) throws ParseException {
        final List<T> values = new ArrayList<>();
        fillValues(cl, opt, values, converter);
        return values;
    }

    /**
     * Returns an option value as a File.
     * <p>
     * If option is absent, returns a default value.<br>
     * Result is null only when option does not exist and default value is null.
     *
     * @param cl The command line.
     * @param opt The option name.
     * @param def Default value.
     * @return The option value as a File.
     * @throws ParseException If result could not be produced.
     */
    public static File getValueAsFile(CommandLine cl,
                                      String opt,
                                      File def) throws ParseException {
        return getValue(cl, opt, def, File::new);
    }

    public static File getValueAsExistingFile(CommandLine cl,
                                              String opt,
                                              File def) throws ParseException {
        final File file = getValueAsFile(cl, opt, def);

        if (file == null || !file.isFile()) {
            throw invalidArg(cl, opt, "is not an existing file. Current dir: '" + Files.currentDir() + "'");
        }
        return file;
    }

    public static File getValueAsDirectory(CommandLine cl,
                                           String opt,
                                           File def) throws ParseException {
        final File result = getValueAsFile(cl, opt, def);
        if (result == null || (result.exists() && !result.isDirectory())) {
            throw invalidArg(cl, opt, "is not a directory. Current dir: '" + Files.currentDir() + "'");
        }
        if (!result.exists()) {
            final boolean done = result.mkdirs();
            if (!done) {
                throw new ParseException("Failed to create directory: " + result);
            }
        }
        return result;
    }

    /**
     * Returns an option value as an existing file or null.
     * <p>
     * If option is absent, returns a default value (possibly null).<br>
     * If result is not null and does not exist as a file, throws an exception.<br>
     * To obtain a null result :
     * <ul>
     * <li>default value must be set to null.</li>
     * <li>option must be absent.
     * </ul>
     *
     * @param cl The command line.
     * @param opt The option name.
     * @param def Default value. <em>MAY BE</em> null.
     * @return The option value or default value. Is either an existing file or null.
     * @throws ParseException If result is not null and does not exist.
     */
    public static File getValueAsNullOrExistingFile(CommandLine cl,
                                                    String opt,
                                                    File def) throws ParseException {
        final File file = getValueAsFile(cl, opt, def);

        if (file != null && (!file.exists() || !file.isFile())) {
            throw invalidArg(cl, opt, "is not an existing file. Current dir: '" + Files.currentDir() + "'");
        }

        return file;
    }

    public static File getValueAsExistingDirectory(CommandLine cl,
                                                   String opt,
                                                   File def) throws ParseException {
        final File file = getValueAsFile(cl, opt, def);

        if (file == null || !file.isDirectory()) {
            throw invalidArg(cl, opt, "is not an existing directory. Current dir: '" + Files.currentDir() + "'");
        }
        return file;
    }

    /**
     * Returns an option value as an existing directory or null.
     * <p>
     * If option is absent, returns a default value (possibly null).<br>
     * If result is not null and does not exist as a directory, throws an exception.<br>
     * To obtain a null result :
     * <ul>
     * <li>default value must be set to null.</li>
     * <li>option must be absent.
     * </ul>
     *
     * @param cl The command line.
     * @param opt The option name.
     * @param def Default value.
     * @return The option value or default value. Is either an existing directory or null.
     * @throws ParseException If result is not null and does not exist.
     */
    public static File getValueAsNullOrExistingDirectory(CommandLine cl,
                                                         String opt,
                                                         File def) throws ParseException {
        final File file = getValueAsFile(cl, opt, def);

        if (file != null && (!file.exists() || !file.isDirectory())) {
            throw invalidArg(cl, opt, "is not an existing directory. Current dir: '" + Files.currentDir() + "'");
        }

        return file;
    }

    public static File getValueAsExistingFileOrDirectory(CommandLine cl,
                                                         String opt,
                                                         File def) throws ParseException {
        final File file = getValueAsFile(cl, opt, def);

        if (file == null || !file.exists()) {
            throw invalidArg(cl, opt, "is not an existing file or directory. Current dir: '" + Files.currentDir() + "'");
        }
        return file;
    }

    /**
     * Returns an option value as an existing file or directory or null.
     * <p>
     * If option is absent, returns a default value (possibly null).<br>
     * If result is not null and does not exist as a file or directory, throws an exception.<br>
     * To obtain a null result :
     * <ul>
     * <li>default value must be set to null.</li>
     * <li>option must be absent.
     * </ul>
     *
     * @param cl The command line.
     * @param opt The option name.
     * @param def Default value.
     * @return The option value or default value. Is either an existing file or directory or null.
     * @throws ParseException If result is not null and does not exist.
     */
    public static File getValueAsNullOrExistingFileOrDirectory(CommandLine cl,
                                                               String opt,
                                                               File def) throws ParseException {
        final File file = getValueAsFile(cl, opt, def);

        if (file != null && !file.exists()) {
            throw invalidArg(cl, opt, "is not an existing file or directory. Current dir: '" + Files.currentDir() + "'");
        }

        return file;
    }

    /**
     * Returns an option value as an URL.
     * <p>
     * If option is absent, returns a default value (possibly null).<br>
     * If option is present and can be converted to an URL, returns this conversion.<br>
     * Otherwise, raises an exception.
     *
     * @param cl The command line.
     * @param opt The option name.
     * @param def Default value.
     * @return The option value or default value.
     * @throws ParseException If option is present and can not be converted to a valid URL.
     */
    public static URL getValueAsURL(CommandLine cl,
                                    String opt,
                                    URL def) throws ParseException {
        if (cl.hasOption(opt)) {
            final String s = cl.getOptionValue(opt);
            try {
                return new URL(s);
            } catch (final MalformedURLException e) {
                throw new ParseException("Failed to convert '" + s + "' to URL (" + e.getMessage() + ")");
            }
        } else {
            return def;
        }
    }

    /**
     * Returns an option value as a string.
     * <p>
     * If option is absent, returns a default value.
     *
     * @param cl The command line.
     * @param opt The option name.
     * @param def Default value.
     * @return The option value as a string.
     */
    public static String getValueAsString(CommandLine cl,
                                          String opt,
                                          String def) {
        return cl.getOptionValue(opt, def);
    }

    private static char parseChar(String s) {
        if (s != null && s.length() == 1) {
            return s.charAt(0);
        } else {
            throw new IllegalArgumentException("Invalid char '" + s + "'");
        }
    }

    /**
     * Returns an option value as a char.
     * <p>
     * If option is absent, returns a default value.
     *
     * @param cl The command line.
     * @param opt The option name.
     * @param def Default value.
     * @return The option value as a char.
     * @throws ParseException If the option value can not be parsed as a char.
     */
    public static char getValueAsChar(CommandLine cl,
                                      String opt,
                                      char def) throws ParseException {
        return getValue(cl, opt, def, AbstractMainSupport::parseChar);
    }

    public static Character getValueAsChar(CommandLine cl,
                                           String opt,
                                           Character def) throws ParseException {
        return getValue(cl, opt, def, AbstractMainSupport::parseChar);
    }

    /**
     * Returns an option value as a long.
     * <p>
     * If option is absent, returns a default value.
     *
     * @param cl The command line.
     * @param opt The option name.
     * @param def Default value.
     * @return The option value as a long.
     * @throws ParseException If the option value can not be parsed as a long.
     */
    public static long getValueAsLong(CommandLine cl,
                                      String opt,
                                      long def) throws ParseException {
        return getValue(cl, opt, def, Long::parseLong);
    }

    public static Long getValueAsLong(CommandLine cl,
                                      String opt,
                                      Long def) throws ParseException {
        return getValue(cl, opt, def, Long::parseLong);
    }

    /**
     * Returns an option value as an int.
     * <p>
     * If option is absent, returns a default value.
     *
     * @param cl The command line.
     * @param opt The option name.
     * @param def Default value.
     * @return The option value as an int.
     * @throws ParseException If the option value can not be parsed as an int.
     */
    public static int getValueAsInt(CommandLine cl,
                                    String opt,
                                    int def) throws ParseException {
        return getValue(cl, opt, def, Integer::parseInt);
    }

    public static Integer getValueAsInt(CommandLine cl,
                                        String opt,
                                        Integer def) throws ParseException {
        return getValue(cl, opt, def, Integer::parseInt);
    }

    /**
     * Returns an option value as a short.
     * <p>
     * If option is absent, returns a default value.
     *
     * @param cl The command line.
     * @param opt The option name.
     * @param def Default value.
     * @return The option value as a short.
     * @throws ParseException If the option value can not be parsed as a short.
     */
    public static short getValueAsShort(CommandLine cl,
                                        String opt,
                                        short def) throws ParseException {
        return getValue(cl, opt, def, Short::parseShort);
    }

    public static Short getValueAsShort(CommandLine cl,
                                        String opt,
                                        Short def) throws ParseException {
        return getValue(cl, opt, def, Short::parseShort);
    }

    /**
     * Returns an option value as a byte.
     * <p>
     * If option is absent, returns a default value.
     *
     * @param cl The command line.
     * @param opt The option name.
     * @param def Default value.
     * @return The option value as a byte.
     * @throws ParseException If the option value can not be parsed as a byte.
     */
    public static byte getValueAsByte(CommandLine cl,
                                      String opt,
                                      byte def) throws ParseException {
        return getValue(cl, opt, def, Byte::parseByte);
    }

    public static Byte getValueAsByte(CommandLine cl,
                                      String opt,
                                      Byte def) throws ParseException {
        return getValue(cl, opt, def, Byte::parseByte);
    }

    /**
     * Returns an option value as a double.
     * <p>
     * If option is absent, returns a default value.
     *
     * @param cl The command line.
     * @param opt The option name.
     * @param def Default value.
     * @return The option value as a double.
     * @throws ParseException If the option value can not be parsed as a double.
     */
    public static double getValueAsDouble(CommandLine cl,
                                          String opt,
                                          double def) throws ParseException {
        return getValue(cl, opt, def, Double::parseDouble);
    }

    public static Double getValueAsDouble(CommandLine cl,
                                          String opt,
                                          Double def) throws ParseException {
        return getValue(cl, opt, def, Double::parseDouble);
    }

    /**
     * Returns an option value as a float.
     * <p>
     * If option is absent, returns a default value.
     *
     * @param cl The command line.
     * @param opt The option name.
     * @param def Default value.
     * @return The option value as a float.
     * @throws ParseException If the option value can not be parsed as a float.
     */
    public static float getValueAsFloat(CommandLine cl,
                                        String opt,
                                        float def) throws ParseException {
        return getValue(cl, opt, def, Float::parseFloat);
    }

    public static Float getValueAsFloat(CommandLine cl,
                                        String opt,
                                        Float def) throws ParseException {
        return getValue(cl, opt, def, Float::parseFloat);
    }

    /**
     * Returns an option value as a enum.
     * <p>
     * If option is absent, returns a default value.
     *
     * @param <E> The enum type.
     * @param cl The command line.
     * @param opt The option name.
     * @param enumClass Enum class
     * @param def Default value.
     * @return The option value as an enum.
     * @throws ParseException If the option value can not be parsed as an enum value.
     */
    public static <E extends Enum<E>> E getValueAsEnum(CommandLine cl,
                                                       String opt,
                                                       Class<E> enumClass,
                                                       E def) throws ParseException {
        return getValue(cl, opt, def, s -> {
            for (final E e : enumClass.getEnumConstants()) {
                if (e.name().equals(s)) {
                    return e;
                }
            }
            throw new IllegalArgumentException("Invalid enum value");
        });
    }

    /**
     * Returns the part of a string that is before a separator.
     *
     * @param s The string.
     * @param sep The separator.
     * @return The part of {@code s} that is before {@code sep} or {@code s} if {@code sep} is not found.
     */
    public static String getPart1(String s,
                                  String sep) {
        final int pos = s.indexOf(sep);
        if (pos >= 0) {
            return s.substring(0, pos);
        } else {
            return s;
        }
    }

    /**
     * Returns the part of a string that is before {@link #DEFAULT_PARTS_SEPARATOR}.
     *
     * @param s The string.
     * @return The part of {@code s} that is before {@link #DEFAULT_PARTS_SEPARATOR} or {@code s} if separator is not found.
     */
    public static String getPart1(String s) {
        return getPart1(s, DEFAULT_PARTS_SEPARATOR);
    }

    /**
     * Returns the part of a string that is after a separator.
     *
     * @param s The string.
     * @param sep The separator.
     * @param def The default value.
     * @return The part of {@code s} that is after {@code sep} or {@code def} if {@code sep} is not found.
     */
    public static String getPart2(String s,
                                  String sep,
                                  String def) {
        final int pos = s.indexOf(sep);
        if (pos >= 0 && pos < s.length()) {
            return s.substring(pos + sep.length());
        } else {
            return def;
        }
    }

    /**
     * Returns the part of a string that is after {@link #DEFAULT_PARTS_SEPARATOR}.
     *
     * @param s The string.
     * @param def The default value.
     * @return The part of {@code s} that is after {@link #DEFAULT_PARTS_SEPARATOR} or {@code def} if separator is not found.
     */
    public static String getPart2(String s,
                                  String def) {
        return getPart2(s, DEFAULT_PARTS_SEPARATOR, def);
    }
}