package cdc.util.cli;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.logging.log4j.Logger;

import cdc.util.lang.Checks;
import cdc.util.lang.FailureReaction;

public class OptionEnumMask {
    private final OptionEnumType type;
    private final Set<OptionEnum> options = new HashSet<>();

    public OptionEnumMask(OptionEnumType type) {
        this.type = type;
    }

    public final void setEnabled(OptionEnum option,
                                 boolean enabled) {
        Checks.isTrue(type.contains(option), "Invalid option");
        if (enabled) {
            options.add(option);
        } else {
            options.remove(option);
        }
    }

    public final void add(OptionEnum option) {
        options.add(option);
    }

    public final void addAll(OptionEnum... options) {
        for (final OptionEnum feature : options) {
            add(feature);
        }
    }

    public final void addAll(Collection<? extends OptionEnum> options) {
        this.options.addAll(options);
    }

    public final void addAll(OptionEnumMask mask) {
        addAll(mask.options);
    }

    public final void remove(OptionEnum option) {
        this.options.remove(option);
    }

    public final boolean isEnabled(OptionEnum option) {
        return options.contains(option);
    }

    public boolean isEmpty() {
        return options.isEmpty();
    }

    public final boolean contains(OptionEnum option) {
        return options.contains(option);
    }

    public final Set<OptionEnum> toSet() {
        return options.stream().collect(Collectors.toSet());
    }

    public final List<OptionEnum> toList() {
        return options.stream().toList();
    }

    /**
     * Check that at most one feature is enabled in a group of features.
     *
     * @param logger The logger to use.
     * @param reaction The reaction to adopt in case of failure.
     * @param options The options.
     * @return {@code true} if at most one feature among {@code features}
     *         is enabled in this FeatureMask.
     */
    public final boolean checkAtMostOne(Logger logger,
                                        FailureReaction reaction,
                                        OptionEnum... options) {
        final Set<OptionEnum> other = new HashSet<>();
        Collections.addAll(other, options);
        other.retainAll(this.options);
        if (other.size() <= 1) {
            return true;
        } else {
            FailureReaction.onError("Expected at most one enabled feature among " + Arrays.toString(options),
                                    logger,
                                    reaction,
                                    IllegalArgumentException::new);
            return false;
        }
    }

    /**
     * Check that exactly one option is enabled in a group of options.
     *
     * @param logger The logger to use.
     * @param reaction The reaction to adopt in case of failure.
     * @param options The options.
     * @return {@code true} if exactly one option among {@code options}
     *         is enabled in this OptionEnumMask.
     */
    public final boolean checkExactlyOne(Logger logger,
                                         FailureReaction reaction,
                                         OptionEnum... options) {
        final Set<OptionEnum> other = new HashSet<>();
        Collections.addAll(other, options);
        other.retainAll(this.options);
        if (other.size() == 1) {
            return true;
        } else {
            FailureReaction.onError("Expected exactly one enabled feature among " + Arrays.toString(options),
                                    logger,
                                    reaction,
                                    IllegalArgumentException::new);
            return false;
        }
    }
}