/*
 * The MIT License
 * 
 * Copyright (c) 2014- High-Mobility GmbH (https://high-mobility.com)
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
package com.highmobility.autoapi;

import com.highmobility.autoapi.property.Property;
import javax.annotation.Nullable;

/**
 * The Messaging capability
 */
public class Messaging {
    public static final int IDENTIFIER = Identifier.MESSAGING;

    public static final byte PROPERTY_TEXT = 0x01;
    public static final byte PROPERTY_HANDLE = 0x02;

    /**
     * Message received
     */
    public static class MessageReceived extends SetCommand {
        Property<String> text = new Property<>(String.class, PROPERTY_TEXT);
        Property<String> handle = new Property<>(String.class, PROPERTY_HANDLE);
    
        /**
         * @return The text
         */
        public Property<String> getText() {
            return text;
        }
        
        /**
         * @return The handle
         */
        public Property<String> getHandle() {
            return handle;
        }
        
        /**
         * Message received
         * 
         * @param text The text
         * @param handle The optional handle of message
         */
        public MessageReceived(String text, @Nullable String handle) {
            super(IDENTIFIER);
        
            addProperty(this.text.update(text));
            addProperty(this.handle.update(handle));
            createBytes();
        }
    
        MessageReceived(byte[] bytes) throws PropertyParseException {
            super(bytes);
            while (propertyIterator.hasNext()) {
                propertyIterator.parseNextSetter(p -> {
                    switch (p.getPropertyIdentifier()) {
                        case PROPERTY_TEXT: return text.update(p);
                        case PROPERTY_HANDLE: return handle.update(p);
                    }
        
                    return null;
                });
            }
            if (this.text.getValue() == null) {
                throw new PropertyParseException(mandatoryPropertyErrorMessage(getClass()));
            }
        }
    }

    /**
     * The messaging state
     */
    public static class State extends SetCommand {
        Property<String> text = new Property<>(String.class, PROPERTY_TEXT);
        Property<String> handle = new Property<>(String.class, PROPERTY_HANDLE);
    
        /**
         * @return The text
         */
        public Property<String> getText() {
            return text;
        }
    
        /**
         * @return The optional handle of message
         */
        public Property<String> getHandle() {
            return handle;
        }
    
        State(byte[] bytes) {
            super(bytes);
            while (propertyIterator.hasNext()) {
                propertyIterator.parseNextState(p -> {
                    switch (p.getPropertyIdentifier()) {
                        case PROPERTY_TEXT: return text.update(p);
                        case PROPERTY_HANDLE: return handle.update(p);
                    }
    
                    return null;
                });
            }
        }
    
        public static final class Builder extends SetCommand.Builder<Builder> {
            public Builder() {
                super(IDENTIFIER);
            }
    
            public State build() {
                SetCommand baseSetCommand = super.build();
                Command resolved = CommandResolver.resolve(baseSetCommand.getByteArray());
                return (State) resolved;
            }
    
            /**
             * @param text The text
             * @return The builder
             */
            public Builder setText(Property<String> text) {
                Property property = text.setIdentifier(PROPERTY_TEXT);
                addProperty(property);
                return this;
            }
            
            /**
             * @param handle The optional handle of message
             * @return The builder
             */
            public Builder setHandle(Property<String> handle) {
                Property property = handle.setIdentifier(PROPERTY_HANDLE);
                addProperty(property);
                return this;
            }
        }
    }
}