/*
 * The MIT License
 * 
 * Copyright (c) 2014- High-Mobility GmbH (https://high-mobility.com)
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
package com.highmobility.autoapi.value.measurement

import com.highmobility.autoapi.CommandParseException
import com.highmobility.autoapi.property.Property
import com.highmobility.value.Bytes

class Duration : MeasurementType {
    val value: Double
    val unit: Unit

    constructor(value: Double, unit: Unit) : super() {
        this.value = value
        this.unit = unit
    
        bytes[0] = 0x07
        bytes[1] = unit.id
        set(2, Property.doubleToBytes(value))
    }

    constructor(valueBytes: Bytes) : super(valueBytes, 0x07) {
        unit = Unit.fromInt(valueBytes[1])
        value = Property.getDouble(valueBytes, 2)
    }

    fun inSeconds(): Double {
        return when (unit) {
            Unit.SECONDS -> value * 1.0
            Unit.MINUTES -> value * 60.0
            Unit.HOURS -> value * 3600.0
            Unit.DAYS -> value * 86400.0
            Unit.WEEKS -> value * 604800.0
            Unit.MONTHS -> value * 2629800.0
            Unit.MILLISECONDS -> value * 0.001
        }
    }
    
    fun inMinutes() = inSeconds() / 60.0
    
    fun inHours() = inSeconds() / 3600.0
    
    fun inDays() = inSeconds() / 86400.0
    
    fun inWeeks() = inSeconds() / 604800.0
    
    fun inMonths() = inSeconds() / 2629800.0
    
    fun inMilliseconds() = inSeconds() / 0.001
    
    enum class Unit(val id: Byte) {
        SECONDS(0x00),
        MINUTES(0x01),
        HOURS(0x02),
        DAYS(0x03),
        WEEKS(0x04),
        MONTHS(0x05),
        MILLISECONDS(0x06);
    
        companion object {
            private val map = values().associateBy(Unit::id)
            fun fromInt(type: Byte) = map[type] ?: throw CommandParseException()
        }
    }
}