/**
 * (c) Copyright 2018, 2019 IBM Corporation
 * 1 New Orchard Road, 
 * Armonk, New York, 10504-1722
 * United States
 * +1 914 499 1900
 * support: Nathaniel Mills wnm3@us.ibm.com
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */

package com.api.jsonata4java.expressions.functions;

import java.time.Instant;

import com.api.jsonata4java.expressions.EvaluateRuntimeException;
import com.api.jsonata4java.expressions.ExpressionsVisitor;
import com.api.jsonata4java.expressions.generated.MappingExpressionParser.Function_callContext;
import com.api.jsonata4java.expressions.utils.Constants;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.LongNode;

/**
 * http://docs.jsonata.org/numeric-functions.html
 *
 * $millis()
 * 
 * Returns the number of milliseconds since the Unix Epoch (1 January, 1970 UTC)
 * as a number. All invocations of $millis() within an evaluation of an
 * expression will all return the same value
 * 
 * Examples
 * 
 * $millis()==1502700297574
 *
 */
public class MillisFunction extends FunctionBase implements Function {

	public static String ERR_ARG1BADTYPE = String.format(Constants.ERR_MSG_ARG1_BAD_TYPE, Constants.FUNCTION_MILLIS);

	public JsonNode invoke(ExpressionsVisitor expressionVisitor, Function_callContext ctx) {
		// Create the variable to return
		JsonNode result = null;

		// Retrieve the number of arguments
		int argCount = getArgumentCount(ctx);

		// Make sure that we have the right number of arguments
		if (argCount == 0) {
			long millis = Instant.now().toEpochMilli();
			result = new LongNode(millis);
		} else {
			throw new EvaluateRuntimeException(ERR_ARG1BADTYPE);
		}

		return result;
	}

	@Override
	public String getSignature() {
		// returns a number
		return "<:n>";
	}
}
