(ns
    ^{:doc "Prototype implementation of TopClock
            Experimental and subject to change!"
      :author "Sam Aaron"}
  overtone.studio.clocks.topclock
  (:require [overtone.osc :as osc]
            [overtone.music.time :refer [now]]))



;; ;;;;;;; proposed OSC protocol ;;;;;;;;;;;

;; OSC protocol:
;; ("/topclock/register" ipaddress rate)
;; ("/topclock/sync/request" ipaddress t1)
;; ("/topclock/sync/response" t1 t2)
;; ("/topclock/stream" time bpm beat numerator denominator)
;; ("/topclock/bpm" time bpm)

;; ipaddress   - osc string (i.e. "192.168.1.1")
;; t1 & t2     - osc double (time in seconds UTC epoch Jan 1 1900)
;; rate        - osc double (stream rate in beats (i.e. 0.125 for 1/8 notes))
;; beat        - osc double (time in beats - and subdivisions of beats - since server started)
;; numerator   - osc int32  (i.e. 4/4 for common time)
;; denominator - osc int32  (i.e. 4/4 for common time)


;; ;;;;;;;; osc breakdown ;;;;;;;;;;;;;;;;;;;

;; ** data stream **

;; ("/topclock/register" ipaddress rate)
;; ("/topclock/stream" time bpm beat numerator denominator)

;; clients can register to be called back at a given rate - as specified
;; when registering. The message rate is relative to the current BPM and
;; is specified in beats. A rate of 0.125 would send 8 messages in sub
;; divisions of a beat - 10.0 10.125 10.25 etc..

;; an ipaddress is provided to (a) avoid broadcasting which is often
;; blocked by switches on internal networks and (b) to remove the need to
;; do any name resolution. providing an ip supports the simplest case.

;; the server will send osc messages to each registered ip address at
;; whatever rate the client requested. Each message contains the current
;; UTC system time (Epoch 1900), the current BPM, the current beat (with
;;                                                                  subdivisions of the beat) since the server started. And optionally a
;; numerator and denominator for managing metre.

;; At it's most basic the system can potentially be used without using
;; the timestamp provided in the stream. This assumes that the client
;; selects a high enough rate to receive updates promptly. Higher
;; accuracy can be achieved by using the timestamp to accurately schedule
;; tempo changes against the stream. when using timestamp there is an
;; assumption that all systems share a synchronised view of time - see **
;; clock sync ** section below.


;; ** clock sync **

;; ("/topclock/sync/request" ipaddress t1)
;; ("/topclock/sync/response" t1 t2)

;; clock sync is two steps, client sends request with a current UTC
;; system timestamp adjusted to an Epoch of Jan 1st 1900 (same as NTP).
;; Server then sends client a message with both the clients initial
;; timestamp + a new system timestamp (Epoch 1900 also) from the server.
;; Client can then establish an offset for its system clock using the
;; following NTP algorithm. Where t1 is the client time when messaging
;; THE server. t2 is the server when messaging the client. And t3 is a
;; new timestamp on the client when receiving the servers message (which
;;                                                                 includes t1 and t2)

;; offset = ((t2 - t1) + (t2 - t3)) / 2

;; Offset is then summed with your system clock to provide a syncd clock.
;; In the simple 'one off' case this will work - as demonstrated in the audio
;; example attached. However a significantly more accurate timestamp
;; with a confidence guide can be established by maintaining a past
;; history of offsets and then applying a std deviation which will
;; provide a guide for a confidence rating. The client is responsible for
;; scheduling regular sync request calls.


;; ** update bpm **

;; ("/topclock/bpm" time bpm)

;; Send BPM change to server with the current client system time (UTC
;;                                                                epoch 1900), which is assumed to be synchronised to the server (see **
;;                                                                                                                                    clock sync ** section) and beats-per-minute as a double. The client
;; should then wait to get the updated BPM from the /topclock/stream
;; rather than applying it locally.


;; ;;;;;;;;;; Trivial Reference Implementation ;;;;;;;;;;
;; ;;
;; ;; this code is provided only as a simple
;; ;; reference implementation.
;; ;;
;; ;; note that this same code runs on both
;; ;; server and client.
;; ;;

;; ;; list of registered users
;; (define *topclock-registered-address* '())
;; ;; ip client
;; (define *iplocal* (sys:ipaddress))
;; ;; topclock port
;; (define *topclock-port* 5555)
;; ;; epoch adjustments for NTP Jan 1st 1900
;; (define *Epoch-1900-1970* 2208988800.0)

;; (define 1900->1970
;;   (lambda (time)
;;           (- time *Epoch-1900-1970*)))

;; (define 1970->1900
;;   (lambda (time)
;;           (+ time *Epoch-1900-1970*)))

;; ;; OSC reciever (for both client and server)
;; (define topclock-receive
;;   (let ((oldbpm 0.0))
;;     (lambda (timestamp address . args)
;;             (cond ((string=? address "/topclock/sync/request")
;;                    (io:osc:send (now) (cons (car args) *topclock-port*) "/topclock/sync/response"
;;                                 (cadr args) (1970->1900 (clock:clock))))
;;                   ((string=? address "/topclock/sync/response")
;;                    (let* ((t1 (1900->1970 (car args)))
;;                           (t2 (1900->1970 (cadr args))) ;; t2
;;                           (t3 (1900->1970 (cadr args))) ;; and t3 the same
;;                           (t4 (clock:clock))
;;                           (msg-delay (- t4 t1))
;;                           (offset (/ (+ (- t2 t1) (- t3 t4)) 2.0)))
;;                          ;; offset is adjustment from local system UTC time.
;;                          (clock:adjust-offset offset)))
;;                   ((string=? address "/topclock/register/request")
;;                    (println 'registering 'client (car args) 'at 'rate (cadr args))
;;                    (set! *topclock-registered-address*
;;                          (cons (cons (car args) (cadr args))
;;                                *topclock-registered-address*)))
;;                   ((string=? address "/topclock/stream")
;;                    (let ((time (1900->1970 (car args)))
;;                          (bpm (cadr args))
;;                          (beat (caddr args))
;;                          (num (cadddr args))
;;                          (denom (car (cddddr args))))
;;                      ;; don't worry about metre (num/denom) for testing
;;                      (if (<> bpm oldbpm)
;;                        (begin (set! oldbpm bpm)
;;                               (*metro* 'set-tempo bpm (clock->samples time) beat)))))
;;                   ((string=? address "/topclock/bpm")
;;                    (let ((time (1900->1970 (car args)))
;;                          (bpm (cadr args)))
;;                      (*metro* 'set-tempo bpm (clock->samples time))))
;;                   (else (println 'bad 'osc 'message: address))))))

;; ;; start topclock osc receiver (both client and server)
;; (io:osc:start-server *topclock-port* "topclock-receive")
;; ;; use doubles for OSC real nums:
;; (io:osc:set-real-64bit? #t)

;; ;; sends OSC message /topclock/sync/request
;; ;; of type <string,double>
;; ;; 'string' is ipaddress string (client must convert hostname to IP)
;; ;; 'double' is the system time in seconds (UTC with epoch of Jan 1 1900 (i.e. NTP))
;; (define topclock-sync
;;   (lambda (server)
;;           (io:osc:send (now) (cons server *topclock-port*) "/topclock/sync/request"
;;                        *iplocal* (1970->1900 (clock:clock)))))


;; ;; sends OSC message /topclock/register
;; ;; of type <string,double>
;; ;; 'string' is ipaddress string (client must convert hostname to IP)
;; ;; 'double' is the 'stream' rate which is in beats at current BPM (0.125 for 8th notes as example)
;; (define topclock-register
;;   (lambda (server)
;;           (io:osc:send (now) (cons server *topclock-port*) "/topclock/register"
;;                        *iplocal* 1/32)))


;; ;; sends OSC message /topclock/bpm
;; ;; of type <double,double>
;; ;; 'double' is time (UTC epoch 1900)
;; ;; 'bpm' beats per minute
;; (define topclock-bpm
;;   (lambda (server time bpm)
;;           (io:osc:send (- time 1000) (cons server *topclock-port*) "/topclock/bpm"
;;                        (1970->1900 (samples->clock time)) bpm)))


;; ;; server proc streams to all registered ip addresses
;; ;; at whatever specified rate (in beats) the client provided
;; (define topclock-streamer
;;   (lambda (beat dur)
;;           (for-each (lambda (client)
;;                             (if (modulo beat (cdr client))
;;                               (io:osc:send (now) (cons (car client) *topclock-port*) "/topclock/stream"
;;                                            (1970->1900 (samples->clock (*metro* beat)))
;;                                            (*metro* 'get-tempo)
;;                                            (rational->real beat)
;;                                            0 0))) ;; don't worry about num and denum for tests
;;                     *topclock-registered-address*)
;;           (callback (*metro* (+ beat (* .5 dur))) 'topclock-streamer
;;                     (+ beat dur) dur)))

;; ;; start topclock streamer
;; (callback (+ (now) *second*) 'topclock-streamer (*metro* 'get-beat 4) 1/64)


;; )))))))))

(defn resolve-ip-address []
  (.getHostAddress (java.net.InetAddress/getLocalHost)))

(def sync-request-path "/topclock/sync/request")
(def sync-response-path "/topclock/sync/response")
(def register-path "/topclock/register")
(def stream-path "/topclock/stream")
(def bpm-path "/topclock/bpm")

(def NTP-1900-epoch-diff 2208988800)

(defn convert-1900->1970
  [time]
  (- time NTP-1900-epoch-diff))

(defn convert-1970->1900
  [time]
  (+ time NTP-1900-epoch-diff))

(defn server-sync-request-handler
  "Handle client request for a sync. Request contains a current UTC
   system timestamp adjusted to an Epoch of Jan 1st 1900 (same as NTP).
   Server then sends client a message with both the clients initial
   timestamp + a new system timestamp (Epoch 1900 also) from the server.
   Client can then establish an offset for its system clock using the
   following NTP algorithm. Where t1 is the client time when messaging
   the server. t2 is the server when messaging the client. And t3 is a
   new timestamp on the client when receiving the servers message (which
   includes t1 and t2)

   offset = ((t2 - t1) + (t2 - t3)) / 2

   Offset is then summed with your system clock to provide a syncd
   clock.  In the simple 'one off' case this will work - as demonstrated
   in the audio example attached. However a significantly more accurate
   timestamp with a confidence guide can be established by maintaining a
   past history of offsets and then applying a std deviation which will
   provide a guide for a confidence rating. The client is responsible
   for scheduling regular sync request calls.

   OSC arguments:
   * ipaddress (string)
   * time (double)"
  [topclock-server osc-msg bar]
  (let [[address time] (:args osc-msg)
        osc-client     (osc/osc-client address (:port topclock-server))]
    (osc/osc-send osc-client sync-reponse-path time (convert-1970->1900 (now)))))

(defn server-register-handler
  "Register to be called back at a given rate. The message rate is
   relative to the current BPM and is specified in beats. A rate of
   0.125 would send 8 messages in sub divisions of a beat - 10.0 10.125
   10.25 etc..

   An ipaddress is provided to (a) avoid broadcasting which is often
   blocked by switches on internal networks and (b) to remove the need
   to do any name resolution. providing an ip supports the simplest
   case.

   OSC arguments:
   * ipaddress (string)
   * rate      (double)"
  [topclock-server osc-msg]
  (let [[address rate] (:args osc-msg)]
    (swap! (:clients topclock-server-server) assoc address rate)))

(defn server-set-bpm-handler
  [topclock-server osc-msg]
  (let [[_ new-bpm] (:args osc-msg)]
    (reset! (:bpm topclock-server) new-bpm)))

(defn client-sync-response
  [topclock-client osc-msg])

(defn client-stream-response
  [topclock-client osc-msg])

(defn register-server-handlers
  [topclock-server]
  (osc/osc-handle topclock-server
                  sync-request-path
                  (partial server-sync-request-handler topclock-server))

  (osc/osc-handle topclock-server
                  register-path
                  (partial server-register-handler topclock-server))

  (osc/osc-handle topclock-server
                  set-bpm-path
                  (partial server-bpm-handler topclock-server)))

(defn register-client-handlers
  [topclock-client]
  (osc-osc-handle topclock-client
                  sync-response-path
                  (partial client-sync-response handler topclock-client))
  (osc-osc-handle topclock-client
                  stream-path
                  (partial client-stream-response handler topclock-client)))

(defrecord TopClockServer [osc-server port bpm clients])
(defrecord TopClockClient [osc-server port current-bpm current-offset server-address server-port])

(defn topclock-server
  "Initialise a new topclock server running on the specific
   port (defaults to 5555). A server is also a client."
  ([] (topclock-server 5555))
  ([port]
     (let [osc-server      (osc/osc-server port)
           clients         (atom {})
           topclock-server (TopClockServer. osc-server clients)]
       (register-server-handlers topclock-server)
       topclock-server)))

(defn topclock-client
  ([topclock-server-address] (topclock-client topclock-server-address 5556))
  ([topclock-server-address client-port] (topclock-client topclock-server-address client-port 5555))
  ([topclock-server-address client-port topclock-server-port]
     (let [osc-server (osc/osc-server client-port)
           client     (TopClockClient. osc-server client-port (atom 0) (atom 0) topclock-server-address topclock-server-port)]
       (register-client-handlers client)
       client)))

(osc/osc-server)
