package com.cdata;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;

public class CData implements AutoCloseable {
    static {
        try {
            // First try loading directly from java.library.path
            try {
                System.loadLibrary("cdatajni");
            } catch (UnsatisfiedLinkError e) {
                // If that fails, try to extract from JAR and load
                loadFromJar();
            }
        } catch (Exception e) {
            throw new RuntimeException("Failed to load native library: " + e.getMessage(), e);
        }
    }
    
    private static void loadFromJar() throws IOException {
        String archName = getArchitectureName();
        String libraryName = System.mapLibraryName("cdatajni");
        String tempDir = System.getProperty("java.io.tmpdir");
        File tempFile = new File(tempDir, "cdatajni-" + archName + "-" + System.currentTimeMillis() + "-" + libraryName);
        tempFile.deleteOnExit();
        // Extract library from JAR
        String name = "/com/cdata/" + libraryName;
		try (InputStream in = CData.class.getResourceAsStream(name)) {
            if (in == null) {
                throw new IOException("Library not found in JAR: /com/cdata/" + libraryName);
            }
            Files.copy(in, tempFile.toPath(), StandardCopyOption.REPLACE_EXISTING);
        }
        
        // Load the extracted library
        System.load(tempFile.getAbsolutePath());
    }

    private static String getArchitectureName() {
        String arch = System.getProperty("os.arch").toLowerCase();
        String os = System.getProperty("os.name").toLowerCase();
        
        if (os.contains("linux")) {
            if (arch.contains("amd64") || arch.contains("x86_64")) {
                return "x86_64-linux";
            } else if (arch.contains("aarch64") || arch.contains("arm64")) {
                return "arm64-linux";
            }
        }
        
        throw new UnsatisfiedLinkError("Unsupported architecture: " + arch + " on " + os);
    }
    
    private long nativeHandle;
    
    // Constructor
    public CData(String schemaString, String mappedFile, int timeout, boolean isProducer) {
        nativeHandle = createNative(schemaString, mappedFile, timeout, isProducer);
        if (nativeHandle == 0) {
            throw new RuntimeException("Failed to create native CData object");
        }
    }
    
    // Destructor - implementing AutoCloseable
    @Override
    public void close() {
        if (nativeHandle != 0) {
            destroyNative(nativeHandle);
            nativeHandle = 0;
        }
    }
    
    // Required by garbage collector
    @Override
    protected void finalize() throws Throwable {
        close();
        super.finalize();
    }
    
    // Native methods
    private native long createNative(String schemaString, String mappedFile, int timeout, boolean isProducer);
    private native void destroyNative(long handle);
    
    // Data manipulation methods
    public native void setData();
    public native int waitOnStateChange();
    public native boolean updateData();
    public native void trigger();
    
    // Value manipulation by ID
    public native void setValueFloatById(int id, float value);
    public native void setValueDoubleById(int id, double value);
    public native void setValueIntById(int id, int value);
    public native void setValueBoolById(int id, boolean value);
    public native void setValueUint64ById(int id, long value);

    public native float getValueFloatById(int id);
    public native double getValueDoubleById(int id);
    public native int getValueIntById(int id);
    public native boolean getValueBoolById(int id);
    public native long getValueUint64ById(int id);

    // Value manipulation by key
    public native void setValueFloatByKey(String key, float value);
    public native void setValueDoubleByKey(String key, double value);
    public native void setValueIntByKey(String key, int value);
    public native void setValueBoolByKey(String key, boolean value);
    public native void setValueUint64ByKey(String key, long value);
    
    public native float getValueFloatByKey(String key);
    public native double getValueDoubleByKey(String key);
    public native int getValueIntByKey(String key);
    public native boolean getValueBoolByKey(String key);
    public native long getValueUint64ByKey(String key);
}
