package com.cdata;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;

public class CData implements AutoCloseable {
	static {
		try {
			// First try loading directly from java.library.path
//			try {
//				System.loadLibrary("cdatajni");
//			} catch (UnsatisfiedLinkError e) {
//				// If that fails, try to extract from JAR and load
				loadFromJar();
//			}
		} catch (Exception e) {
			throw new RuntimeException("Failed to load native library: " + e.getMessage(), e);
		}
	}

	private static class OSUtil {
		public static boolean is64Bit() {
			//// System.out.println("Arch: "+getOsArch());
			return getOsArch().startsWith("x86_64") || getOsArch().startsWith("amd64")
					|| getOsArch().startsWith("aarch64");
		}

		public static boolean isARM() {
			return getOsArch().startsWith("arm") || getOsArch().startsWith("aarch");
		}

		public static boolean isPPC() {
			return getOsArch().toLowerCase().contains("ppc");
		}

		public static boolean isWindows() {
			//// System.out.println("OS name: "+getOsName());
			return getOsName().toLowerCase().startsWith("windows") || getOsName().toLowerCase().startsWith("microsoft")
					|| getOsName().toLowerCase().startsWith("ms");
		}

		public static boolean isLinux() {
			return getOsName().toLowerCase().startsWith("linux");
		}

		public static boolean isFreeBSD() {
			return getOsName().toLowerCase().startsWith("freebsd");
		}

		public static boolean isOSX() {
			return getOsName().toLowerCase().startsWith("mac");
		}

		public static String getExtension() {
			if (isWindows()) {
				return ".dll";
			}

			if (isLinux() || isFreeBSD()) {
				return ".so";
			}

			if (isOSX()) {
				return ".jnilib";
			}

			return "";
		}

		public static String getOsName() {
			return System.getProperty("os.name");
		}

		public static String getOsArch() {
			return System.getProperty("os.arch");
		}

		@SuppressWarnings("unused")
		public static String getIdentifier() {
			return getOsName() + " : " + getOsArch();
		}
	}

	private static void loadFromJar() throws IOException {
		String archName = getArchitectureName();

		String libraryName = System.mapLibraryName("cdatajni-x86_64");
		String tempDir = System.getProperty("java.io.tmpdir");
		File tempFile = new File(tempDir,
				 libraryName);
		//tempFile.deleteOnExit();
		// Extract library from JAR
		String name = "/com/cdata/" + libraryName;
		try (InputStream in = CData.class.getResourceAsStream(name)) {
			if (in == null) {
				throw new IOException("Library not found in JAR: /com/cdata/" + libraryName);
			}
			Files.copy(in, tempFile.toPath(), StandardCopyOption.REPLACE_EXISTING);
		}
		System.out.println("Attempting to load "+name+" from file "+tempFile+" exists: "+tempFile.exists());

		// Load the extracted library
		System.load(tempFile.getAbsolutePath());
	}

	private static String getArchitectureName() {
		String arch = System.getProperty("os.arch").toLowerCase();
		String os = System.getProperty("os.name").toLowerCase();

		if (os.contains("linux")) {
			if (arch.contains("amd64") || arch.contains("x86_64")) {
				return "x86_64-linux";
			} else if (arch.contains("aarch64") || arch.contains("arm64")) {
				return "arm64-linux";
			}
		}

		throw new UnsatisfiedLinkError("Unsupported architecture: " + arch + " on " + os);
	}

	private long nativeHandle;

    // Producer constructor
    public CData(String schemaString, String mappedFile) {
		nativeHandle = createNative(schemaString, mappedFile, 0, true);
        if (nativeHandle == 0) {
            throw new RuntimeException("Failed to create native CData object");
        }
    }
    
    // Consumer constructor
    public CData(String mappedFile, int timeout) {
		nativeHandle = createNative("", mappedFile, timeout, false);
        if (nativeHandle == 0) {
            throw new RuntimeException("Failed to create native CData object");
        }
    }
    

	// Constructor
	public CData(String schemaString, String mappedFile, int timeout, boolean isProducer) {
		nativeHandle = createNative(schemaString, mappedFile, timeout, isProducer);
		if (nativeHandle == 0) {
			throw new RuntimeException("Failed to create native CData object");
		}
	}

	// Destructor - implementing AutoCloseable
	@Override
	public void close() {
		if (nativeHandle != 0) {
			destroyNative(nativeHandle);
			nativeHandle = 0;
		}
	}

	// Required by garbage collector
	@Override
	protected void finalize() throws Throwable {
		close();
		super.finalize();
	}

	// Native methods
	private native long createNative(String schemaString, String mappedFile, int timeout, boolean isProducer);

	private native void destroyNative(long handle);

	// Data manipulation methods
	public native void setData();
	public native int waitOnStateChange();
	public native boolean updateData();
	public native void trigger();
	public native boolean openMap();

	// Value manipulation by ID
	public native void setValueFloatById(int id, float value);
	public native void setValueDoubleById(int id, double value);
	public native void setValueIntById(int id, int value);
	public native void setValueBoolById(int id, boolean value);
	public native void setValueUint64ById(int id, long value);
	
	public native float getValueFloatById(int id);
	public native double getValueDoubleById(int id);
	public native int getValueIntById(int id);
	public native boolean getValueBoolById(int id);
	public native long getValueUint64ById(int id);

	// Value manipulation by key
	public native void setValueFloatByKey(String key, float value);
	public native void setValueDoubleByKey(String key, double value);
	public native void setValueIntByKey(String key, int value);
	public native void setValueBoolByKey(String key, boolean value);
	public native void setValueUint64ByKey(String key, long value);
	
	public native float getValueFloatByKey(String key);
	public native double getValueDoubleByKey(String key);
	public native int getValueIntByKey(String key);
	public native boolean getValueBoolByKey(String key);
	public native long getValueUint64ByKey(String key);
}


    