/*
 * Decompiled with CFR 0.152.
 */
package com.sap.cloud.security.test;

import com.sap.cloud.security.config.Service;
import com.sap.cloud.security.json.JsonObject;
import com.sap.cloud.security.json.JsonParsingException;
import com.sap.cloud.security.token.SapIdToken;
import com.sap.cloud.security.token.Token;
import com.sap.cloud.security.token.XsuaaToken;
import com.sap.cloud.security.token.validation.validators.JwtSignatureAlgorithm;
import java.io.IOException;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.Signature;
import java.security.SignatureException;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Base64;
import java.util.GregorianCalendar;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import javax.annotation.Nonnull;
import org.apache.commons.io.IOUtils;
import org.json.JSONException;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class JwtGenerator {
    public static final Instant NO_EXPIRE_DATE = new GregorianCalendar(2190, 11, 31).getTime().toInstant();
    private static final Logger LOGGER = LoggerFactory.getLogger(JwtGenerator.class);
    private static final String DEFAULT_JWKS_URL = "http://localhost/token_keys";
    private static final String DEFAULT_KEY_ID = "default-kid";
    private static final char DOT = '.';
    private final JSONObject jsonHeader = new JSONObject();
    private final JSONObject jsonPayload = new JSONObject();
    private final Set<String> unsupportedClaims = Arrays.asList("aud").stream().collect(Collectors.toSet());
    private SignatureCalculator signatureCalculator;
    private Service service;
    private JwtSignatureAlgorithm signatureAlgorithm;
    private PrivateKey privateKey;
    private String appId;
    private List<String> scopes = new ArrayList<String>();
    private List<String> localScopes = new ArrayList<String>();

    private JwtGenerator() {
    }

    public static JwtGenerator getInstance(Service service, String clientId) {
        return JwtGenerator.getInstance(service, JwtGenerator::calculateSignature, clientId);
    }

    static JwtGenerator getInstance(Service service, SignatureCalculator signatureCalculator, String clientId) {
        JwtGenerator instance = new JwtGenerator();
        instance.service = service;
        instance.signatureCalculator = signatureCalculator;
        instance.signatureAlgorithm = JwtSignatureAlgorithm.RS256;
        JwtGenerator.setTokenDefaults(clientId, instance);
        return instance;
    }

    private static void setTokenDefaults(String clientId, JwtGenerator instance) {
        instance.withHeaderParameter("alg", instance.signatureAlgorithm.value());
        instance.withHeaderParameter("kid", DEFAULT_KEY_ID);
        instance.withClaimValue("cid", clientId);
        instance.withExpiration(NO_EXPIRE_DATE);
        if (instance.service == Service.XSUAA) {
            instance.withHeaderParameter("jku", DEFAULT_JWKS_URL);
        }
    }

    public JwtGenerator withHeaderParameter(String parameterName, String value) {
        this.jsonHeader.put(parameterName, (Object)value);
        return this;
    }

    public JwtGenerator withClaimValue(String claimName, String value) {
        this.assertClaimIsSupported(claimName);
        this.jsonPayload.put(claimName, (Object)value);
        return this;
    }

    public JwtGenerator withClaimValue(String claimName, JsonObject object) {
        this.assertClaimIsSupported(claimName);
        try {
            this.jsonPayload.put(claimName, (Object)new JSONObject(object.asJsonString()));
        }
        catch (JSONException e) {
            throw new JsonParsingException(e.getMessage());
        }
        return this;
    }

    public JwtGenerator withClaimValues(String claimName, String ... values) {
        this.assertClaimIsSupported(claimName);
        this.jsonPayload.put(claimName, (Object)values);
        return this;
    }

    public JwtGenerator withClaimsFromFile(String claimsJsonResource) throws IOException {
        JSONObject claimsAsJsonObject;
        String claimsJson = IOUtils.resourceToString((String)claimsJsonResource, (Charset)StandardCharsets.UTF_8);
        try {
            claimsAsJsonObject = new JSONObject(claimsJson);
        }
        catch (JSONException e) {
            throw new JsonParsingException(e.getMessage());
        }
        for (String key : claimsAsJsonObject.keySet()) {
            Object value = claimsAsJsonObject.get(key);
            this.jsonPayload.put(key, value);
        }
        return this;
    }

    private void assertClaimIsSupported(String claimName) {
        if (this.unsupportedClaims.contains(claimName)) {
            throw new UnsupportedOperationException("generic method for claim " + claimName + " is not supported");
        }
    }

    public JwtGenerator withExpiration(@Nonnull Instant expiration) {
        this.jsonPayload.put("exp", expiration.getEpochSecond());
        return this;
    }

    public JwtGenerator withSignatureAlgorithm(JwtSignatureAlgorithm signatureAlgorithm) {
        if (signatureAlgorithm != JwtSignatureAlgorithm.RS256) {
            throw new UnsupportedOperationException(signatureAlgorithm + " is not supported yet");
        }
        this.signatureAlgorithm = signatureAlgorithm;
        return this;
    }

    public JwtGenerator withPrivateKey(PrivateKey privateKey) {
        this.privateKey = privateKey;
        return this;
    }

    public JwtGenerator withScopes(String ... scopes) {
        if (this.service != Service.XSUAA) {
            throw new UnsupportedOperationException("Scopes are not supported for service " + this.service);
        }
        this.scopes = Arrays.asList(scopes);
        this.putScopesInJsonPayload();
        return this;
    }

    public JwtGenerator withLocalScopes(String ... scopes) {
        if (this.appId == null) {
            throw new IllegalStateException("Cannot create local scopes because appId has not been set!");
        }
        if (this.service != Service.XSUAA) {
            throw new UnsupportedOperationException("Scopes are not supported for service " + this.service);
        }
        this.localScopes = Stream.of(scopes).map(scope -> this.appId + "." + scope).collect(Collectors.toList());
        this.putScopesInJsonPayload();
        return this;
    }

    public JwtGenerator withAppId(String appId) {
        this.appId = appId;
        return this;
    }

    public Token createToken() {
        if (this.privateKey == null) {
            throw new IllegalStateException("Private key was not set!");
        }
        this.createAudienceClaim();
        switch (this.service) {
            case IAS: {
                return new SapIdToken(this.createTokenAsString());
            }
            case XSUAA: {
                return new XsuaaToken(this.createTokenAsString());
            }
        }
        throw new UnsupportedOperationException("Identity Service " + this.service + " is not supported.");
    }

    private void putScopesInJsonPayload() {
        List resultingScopes = Stream.concat(this.localScopes.stream(), this.scopes.stream()).collect(Collectors.toList());
        this.jsonPayload.put("scope", resultingScopes);
    }

    private void createAudienceClaim() {
        if (this.service == Service.IAS) {
            this.jsonPayload.put("aud", (Object)this.jsonPayload.getString("cid"));
        } else {
            this.jsonPayload.put("aud", Arrays.asList(this.jsonPayload.getString("cid")));
        }
    }

    private String createTokenAsString() {
        String header = this.base64Encode(this.jsonHeader.toString().getBytes());
        String payload = this.base64Encode(this.jsonPayload.toString().getBytes());
        String headerAndPayload = header + '.' + payload;
        String signature = this.calculateSignature(headerAndPayload);
        return headerAndPayload + '.' + signature;
    }

    private static byte[] calculateSignature(PrivateKey privateKey, JwtSignatureAlgorithm signatureAlgorithm, byte[] dataToSign) throws SignatureException, InvalidKeyException, NoSuchAlgorithmException {
        Signature signature = Signature.getInstance(signatureAlgorithm.javaSignature());
        signature.initSign(privateKey);
        signature.update(dataToSign);
        return signature.sign();
    }

    private String calculateSignature(String headerAndPayload) {
        try {
            return this.base64Encode(this.signatureCalculator.calculateSignature(this.privateKey, this.signatureAlgorithm, headerAndPayload.getBytes()));
        }
        catch (NoSuchAlgorithmException e) {
            LOGGER.error("Algorithm '{}' not found.", (Object)this.signatureAlgorithm.javaSignature());
            throw new UnsupportedOperationException(e);
        }
        catch (SignatureException e) {
            LOGGER.error("Error creating JWT signature.");
            throw new UnsupportedOperationException(e);
        }
        catch (InvalidKeyException e) {
            LOGGER.error("Invalid private key.");
            throw new UnsupportedOperationException(e);
        }
    }

    private String base64Encode(byte[] bytes) {
        return Base64.getUrlEncoder().withoutPadding().encodeToString(bytes);
    }

    static interface SignatureCalculator {
        public byte[] calculateSignature(PrivateKey var1, JwtSignatureAlgorithm var2, byte[] var3) throws InvalidKeyException, SignatureException, NoSuchAlgorithmException;
    }
}

