/*
 * Decompiled with CFR 0.152.
 */
package com.sap.cloud.security.test;

import com.sap.cloud.security.config.Service;
import com.sap.cloud.security.json.JsonObject;
import com.sap.cloud.security.json.JsonParsingException;
import com.sap.cloud.security.test.RSAKeys;
import com.sap.cloud.security.token.SapIdToken;
import com.sap.cloud.security.token.Token;
import com.sap.cloud.security.token.XsuaaToken;
import com.sap.cloud.security.token.validation.validators.JwtSignatureAlgorithm;
import java.io.IOException;
import java.lang.reflect.Field;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.Signature;
import java.security.SignatureException;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Base64;
import java.util.Collections;
import java.util.GregorianCalendar;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Stream;
import javax.annotation.Nonnull;
import org.apache.commons.io.IOUtils;
import org.json.JSONException;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class JwtGenerator {
    public static final Instant NO_EXPIRE_DATE = new GregorianCalendar(2190, 11, 31).getTime().toInstant();
    public static final String DEFAULT_KEY_ID = "default-kid";
    public static final String DEFAULT_KEY_ID_IAS = "default-kid-ias";
    public static final String DEFAULT_ZONE_ID = "the-zone-id";
    public static final String DEFAULT_APP_TID = "the-app-tid";
    public static final String DEFAULT_USER_ID = "the-user-id";
    private static final Logger LOGGER = LoggerFactory.getLogger(JwtGenerator.class);
    private static final String DEFAULT_JWKS_URL = "http://localhost/token_keys";
    private static final char DOT = '.';
    private final JSONObject jsonHeader = JwtGenerator.newPredictableOrderingJSONObject();
    private final JSONObject jsonPayload = JwtGenerator.newPredictableOrderingJSONObject();
    private final SignatureCalculator signatureCalculator;
    private final Service service;
    private JwtSignatureAlgorithm signatureAlgorithm;
    private PrivateKey privateKey = RSAKeys.generate().getPrivate();
    private String appId;
    private List<String> scopes = new ArrayList<String>();
    private List<String> localScopes = new ArrayList<String>();

    private JwtGenerator(Service service, SignatureCalculator signatureCalculator) {
        this.service = service;
        this.signatureCalculator = signatureCalculator;
        this.predefineTokenClaims();
    }

    private static JSONObject newPredictableOrderingJSONObject() {
        JSONObject jsonObject = new JSONObject();
        try {
            Field declaredMapField = jsonObject.getClass().getDeclaredField("map");
            declaredMapField.setAccessible(true);
            declaredMapField.set(jsonObject, new LinkedHashMap());
            declaredMapField.setAccessible(false);
        }
        catch (IllegalAccessException | NoSuchFieldException e) {
            LOGGER.info("Couldn't create a JSONObject with a LinkedHashMap field. {}", (Object)e.getMessage());
        }
        return jsonObject;
    }

    public static JwtGenerator getInstance(Service service, String clientId) {
        JwtGenerator instance = new JwtGenerator(service, JwtGenerator::calculateSignature);
        instance.setDefaultsForNewToken(clientId);
        return instance;
    }

    public static JwtGenerator getInstanceFromFile(Service service, String tokenJsonResource) {
        return new JwtGenerator(service, JwtGenerator::calculateSignature).fromFile(tokenJsonResource);
    }

    static JwtGenerator getInstance(Service service, SignatureCalculator signatureCalculator) {
        JwtGenerator instance = new JwtGenerator(service, signatureCalculator);
        instance.setDefaultsForNewToken("client-id-not-relevant-here");
        return instance;
    }

    private JwtGenerator fromFile(String tokenJsonResource) {
        String tokenJson = this.read(tokenJsonResource);
        JSONObject jsonObject = this.createJsonObject(tokenJson);
        JSONObject header = jsonObject.optJSONObject("header");
        JSONObject payload = jsonObject.optJSONObject("payload");
        this.copyJsonProperties(this.filterPayload(payload), this.jsonPayload);
        this.copyJsonProperties(this.filterHeader(header), this.jsonHeader);
        this.signatureAlgorithm = this.extractAlgorithm(this.jsonHeader).orElse(JwtSignatureAlgorithm.RS256);
        return this;
    }

    private void setDefaultsForNewToken(String azp) {
        this.signatureAlgorithm = JwtSignatureAlgorithm.RS256;
        this.withHeaderParameter("alg", JwtSignatureAlgorithm.RS256.value());
        this.withClaimValue("azp", azp);
        this.withClaimValue("cid", azp);
        if (this.service == Service.IAS) {
            this.jsonPayload.put("aud", (Object)azp);
            this.jsonPayload.put("zone_uuid", (Object)DEFAULT_ZONE_ID);
            this.jsonPayload.put("app_tid", (Object)DEFAULT_APP_TID);
            this.jsonPayload.put("user_uuid", (Object)DEFAULT_USER_ID);
            this.jsonPayload.put("scim_id", (Object)DEFAULT_USER_ID);
        } else {
            this.withClaimValue("cid", azp);
            this.jsonPayload.put("aud", Collections.singletonList(azp));
            this.jsonPayload.put("zid", (Object)DEFAULT_ZONE_ID);
            this.jsonPayload.put("ext_attr", (Object)this.createJsonObject("{\"enhancer\" : \"XSUAA\"} "));
        }
    }

    private void predefineTokenClaims() {
        this.withExpiration(NO_EXPIRE_DATE);
        if (this.service == Service.IAS) {
            this.withHeaderParameter("kid", DEFAULT_KEY_ID_IAS);
        }
        if (this.service == Service.XSUAA) {
            this.withHeaderParameter("kid", DEFAULT_KEY_ID);
            this.withHeaderParameter("jku", DEFAULT_JWKS_URL);
        }
    }

    public JwtGenerator withHeaderParameter(String parameterName, String value) {
        this.jsonHeader.put(parameterName, (Object)value);
        return this;
    }

    public JwtGenerator withClaimValue(String claimName, String value) {
        this.jsonPayload.put(claimName, (Object)value);
        return this;
    }

    public JwtGenerator withClaimValue(String claimName, JsonObject object) {
        try {
            this.jsonPayload.put(claimName, (Object)new JSONObject(object.asJsonString()));
        }
        catch (JSONException e) {
            throw new JsonParsingException(e.getMessage());
        }
        return this;
    }

    public JwtGenerator withClaimValue(String claimName, Map<String, String> map) {
        try {
            this.jsonPayload.put(claimName, map);
        }
        catch (JSONException e) {
            throw new JsonParsingException(e.getMessage());
        }
        return this;
    }

    public JwtGenerator withClaimValues(String claimName, String ... values) {
        this.jsonPayload.put(claimName, (Object)values);
        return this;
    }

    public JwtGenerator withClaimsFromFile(String claimsJsonResource) throws IllegalArgumentException {
        String claimsJson = this.read(claimsJsonResource);
        JSONObject jsonObject = this.createJsonObject(claimsJson);
        this.copyJsonProperties(jsonObject, this.jsonPayload);
        return this;
    }

    public JwtGenerator withExpiration(@Nonnull Instant expiration) {
        this.jsonPayload.put("exp", expiration.getEpochSecond());
        return this;
    }

    public JwtGenerator withSignatureAlgorithm(JwtSignatureAlgorithm signatureAlgorithm) {
        if (signatureAlgorithm != JwtSignatureAlgorithm.RS256) {
            throw new UnsupportedOperationException(signatureAlgorithm + " is not supported yet");
        }
        this.signatureAlgorithm = signatureAlgorithm;
        return this;
    }

    public JwtGenerator withPrivateKey(PrivateKey privateKey) {
        this.privateKey = privateKey;
        return this;
    }

    public JwtGenerator withScopes(String ... scopes) {
        if (this.service != Service.XSUAA) {
            throw new UnsupportedOperationException("Scopes are not supported for service " + this.service);
        }
        this.scopes = Arrays.asList(scopes);
        this.putScopesInJsonPayload();
        return this;
    }

    public JwtGenerator withLocalScopes(String ... scopes) {
        if (this.appId == null) {
            throw new IllegalStateException("Cannot create local scopes because appId has not been set!");
        }
        if (this.service != Service.XSUAA) {
            throw new UnsupportedOperationException("Scopes are not supported for service " + this.service);
        }
        this.localScopes = Stream.of(scopes).map(scope -> this.appId + "." + scope).toList();
        this.putScopesInJsonPayload();
        return this;
    }

    public JwtGenerator withAppId(String appId) {
        this.appId = appId;
        return this;
    }

    public Token createToken() {
        if (this.privateKey == null) {
            throw new IllegalStateException("Private key was not set!");
        }
        return switch (this.service) {
            case Service.IAS -> new SapIdToken(this.createTokenAsString());
            case Service.XSUAA -> new XsuaaToken(this.createTokenAsString());
            default -> throw new UnsupportedOperationException("Identity Service " + this.service + " is not supported.");
        };
    }

    private JSONObject filterPayload(JSONObject payload) {
        if (payload != null) {
            payload.remove("exp");
            payload.remove("iss");
        }
        return payload;
    }

    private JSONObject filterHeader(JSONObject header) {
        if (header != null) {
            header.remove("jku");
            header.remove("kid");
        }
        return header;
    }

    private void copyJsonProperties(JSONObject source, JSONObject target) {
        if (source != null) {
            for (String key : source.keySet()) {
                Object value = source.get(key);
                target.put(key, value);
            }
        }
    }

    private JSONObject createJsonObject(String claimsJson) {
        try {
            return new JSONObject(claimsJson);
        }
        catch (JSONException e) {
            throw new JsonParsingException(e.getMessage());
        }
    }

    private void putScopesInJsonPayload() {
        List resultingScopes = Stream.concat(this.localScopes.stream(), this.scopes.stream()).toList();
        this.jsonPayload.put("scope", resultingScopes);
    }

    private String createTokenAsString() {
        String header = this.base64Encode(this.jsonHeader.toString().getBytes());
        String payload = this.base64Encode(this.jsonPayload.toString().getBytes());
        String headerAndPayload = header + "." + payload;
        String signature = this.calculateSignature(headerAndPayload);
        return headerAndPayload + "." + signature;
    }

    private static byte[] calculateSignature(PrivateKey privateKey, JwtSignatureAlgorithm signatureAlgorithm, byte[] dataToSign) throws SignatureException, InvalidKeyException, NoSuchAlgorithmException {
        Signature signature = Signature.getInstance(signatureAlgorithm.javaSignature());
        signature.initSign(privateKey);
        signature.update(dataToSign);
        return signature.sign();
    }

    private String calculateSignature(String headerAndPayload) {
        try {
            return this.base64Encode(this.signatureCalculator.calculateSignature(this.privateKey, this.signatureAlgorithm, headerAndPayload.getBytes()));
        }
        catch (NoSuchAlgorithmException e) {
            LOGGER.error("Algorithm '{}' not found.", (Object)this.signatureAlgorithm.javaSignature());
            throw new UnsupportedOperationException(e);
        }
        catch (SignatureException e) {
            LOGGER.error("Error creating JWT signature.");
            throw new UnsupportedOperationException(e);
        }
        catch (InvalidKeyException e) {
            LOGGER.error("Invalid private key.");
            throw new UnsupportedOperationException(e);
        }
    }

    private String base64Encode(byte[] bytes) {
        return Base64.getUrlEncoder().withoutPadding().encodeToString(bytes);
    }

    private String read(String tokenJsonResource) {
        try {
            return IOUtils.resourceToString((String)tokenJsonResource, (Charset)StandardCharsets.UTF_8);
        }
        catch (IOException e) {
            throw new IllegalArgumentException("Error reading resource file: " + e.getMessage());
        }
    }

    private Optional<JwtSignatureAlgorithm> extractAlgorithm(JSONObject jsonHeader) {
        if (jsonHeader == null || !jsonHeader.has("alg")) {
            return Optional.empty();
        }
        String alg = jsonHeader.getString("alg");
        JwtSignatureAlgorithm algorithm = JwtSignatureAlgorithm.fromValue((String)alg);
        if (algorithm == null) {
            throw new UnsupportedOperationException(String.format("Algorithm %s of token not supported!", alg));
        }
        return Optional.of(algorithm);
    }

    static interface SignatureCalculator {
        public byte[] calculateSignature(PrivateKey var1, JwtSignatureAlgorithm var2, byte[] var3) throws InvalidKeyException, SignatureException, NoSuchAlgorithmException;
    }
}

