/*
 * Decompiled with CFR 0.152.
 */
package com.sap.cloud.security.token.validation.validators;

import com.sap.cloud.security.config.OAuth2ServiceConfiguration;
import com.sap.cloud.security.config.Service;
import com.sap.cloud.security.token.Token;
import com.sap.cloud.security.token.validation.ValidationResult;
import com.sap.cloud.security.token.validation.ValidationResults;
import com.sap.cloud.security.token.validation.Validator;
import com.sap.cloud.security.token.validation.validators.JsonWebKeyImpl;
import com.sap.cloud.security.token.validation.validators.JwtSignatureAlgorithm;
import com.sap.cloud.security.token.validation.validators.OAuth2TokenKeyServiceWithCache;
import com.sap.cloud.security.token.validation.validators.OidcConfigurationServiceWithCache;
import com.sap.cloud.security.xsuaa.Assertions;
import com.sap.cloud.security.xsuaa.client.DefaultOidcConfigurationService;
import com.sap.cloud.security.xsuaa.client.OAuth2ServiceException;
import java.net.URI;
import java.nio.charset.StandardCharsets;
import java.security.NoSuchAlgorithmException;
import java.security.PublicKey;
import java.security.Signature;
import java.security.spec.InvalidKeySpecException;
import java.util.Base64;
import java.util.regex.Pattern;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;

class JwtSignatureValidator
implements Validator<Token> {
    private final OAuth2TokenKeyServiceWithCache tokenKeyService;
    private final OidcConfigurationServiceWithCache oidcConfigurationService;
    private OAuth2ServiceConfiguration configuration;

    JwtSignatureValidator(OAuth2ServiceConfiguration configuration, OAuth2TokenKeyServiceWithCache tokenKeyService, OidcConfigurationServiceWithCache oidcConfigurationService) {
        Assertions.assertNotNull((Object)configuration, (String)"JwtSignatureValidator requires configuration.");
        Assertions.assertNotNull((Object)tokenKeyService, (String)"JwtSignatureValidator requires a tokenKeyService.");
        Assertions.assertNotNull((Object)oidcConfigurationService, (String)"JwtSignatureValidator requires a oidcConfigurationService.");
        this.configuration = configuration;
        this.tokenKeyService = tokenKeyService;
        this.oidcConfigurationService = oidcConfigurationService;
    }

    @Override
    public ValidationResult validate(Token token) {
        if (Service.IAS == this.configuration.getService() && !token.getIssuer().equals("" + this.configuration.getUrl()) && token.getZoneId() == null) {
            return ValidationResults.createInvalid("Error occurred during signature validation: OIDC token must provide zone_uuid.");
        }
        try {
            String jwksUri = this.getOrRequestJwksUri(token);
            String fallbackPublicKey = null;
            if (this.configuration != null && this.configuration.hasProperty("verificationkey")) {
                fallbackPublicKey = this.configuration.getProperty("verificationkey");
            }
            String keyId = this.getOrDefaultKeyId(token);
            return this.validate(token.getTokenValue(), this.getOrDefaultSignatureAlgorithm(token), keyId, jwksUri, fallbackPublicKey, token.getZoneId());
        }
        catch (OAuth2ServiceException | IllegalArgumentException e) {
            return ValidationResults.createInvalid("Error occurred during jwks uri determination: {}", e.getMessage());
        }
    }

    @Nonnull
    private String getOrDefaultKeyId(Token token) {
        if (this.configuration.isLegacyMode()) {
            return "legacy-token-key";
        }
        if (token.hasHeaderParameter("kid")) {
            return token.getHeaderParameterAsString("kid");
        }
        return "default-kid";
    }

    @Nonnull
    private String getOrDefaultSignatureAlgorithm(Token token) {
        String algHeader = token.getHeaderParameterAsString("alg");
        if (token.hasHeaderParameter("alg") && JwtSignatureAlgorithm.fromValue(algHeader) == null) {
            throw new IllegalArgumentException("Jwt token with signature algorithm '" + algHeader + "' is not supported.");
        }
        return JwtSignatureAlgorithm.RS256.value();
    }

    @Nonnull
    private String getOrRequestJwksUri(Token token) throws OAuth2ServiceException {
        URI discoveryUri;
        URI jkuUri;
        if (this.configuration.isLegacyMode()) {
            return this.configuration.getUrl() + "/token_keys";
        }
        if (this.configuration.getService() == Service.XSUAA && token.hasHeaderParameter("jku")) {
            return token.getHeaderParameterAsString("jku");
        }
        if (this.configuration.getService() != Service.XSUAA && token.getIssuer() != null && (jkuUri = this.oidcConfigurationService.getOrRetrieveEndpoints(discoveryUri = DefaultOidcConfigurationService.getDiscoveryEndpointUri((String)token.getIssuer())).getJwksUri()) != null) {
            return jkuUri.toString();
        }
        throw new IllegalArgumentException("Token signature can not be validated as jwks uri can not be determined: Token does not provide the required 'jku' header or issuer claim.");
    }

    ValidationResult validate(String token, String tokenAlgorithm, String tokenKeyId, String tokenKeysUrl, @Nullable String fallbackPublicKey, @Nullable String zoneId) {
        Assertions.assertHasText((String)token, (String)"token must not be null or empty.");
        Assertions.assertHasText((String)tokenAlgorithm, (String)"tokenAlgorithm must not be null or empty.");
        Assertions.assertHasText((String)tokenKeyId, (String)"tokenKeyId must not be null or empty.");
        Assertions.assertHasText((String)tokenKeysUrl, (String)"tokenKeysUrl must not be null or empty.");
        return Validation.getInstance().validate(this.tokenKeyService, token, tokenAlgorithm, tokenKeyId, URI.create(tokenKeysUrl), fallbackPublicKey, zoneId);
    }

    private static class Validation {
        JwtSignatureAlgorithm jwtSignatureAlgorithm;
        PublicKey publicKey;
        Signature publicSignature;
        private static final Pattern DOT = Pattern.compile("\\.", 0);

        private Validation() {
        }

        static Validation getInstance() {
            return new Validation();
        }

        ValidationResult validate(OAuth2TokenKeyServiceWithCache tokenKeyService, String token, String tokenAlgorithm, String tokenKeyId, URI tokenKeysUrl, @Nullable String fallbackPublicKey, @Nullable String zoneId) {
            ValidationResult validationResult = this.setSupportedJwtAlgorithm(tokenAlgorithm);
            if (validationResult.isErroneous()) {
                return validationResult;
            }
            validationResult = this.setPublicKey(tokenKeyService, tokenKeyId, tokenKeysUrl, zoneId);
            if (validationResult.isErroneous()) {
                if (fallbackPublicKey != null) {
                    try {
                        this.publicKey = JsonWebKeyImpl.createPublicKeyFromPemEncodedPublicKey(JwtSignatureAlgorithm.RS256, fallbackPublicKey);
                    }
                    catch (NoSuchAlgorithmException | InvalidKeySpecException e) {
                        return ValidationResults.createInvalid("Error occurred during signature validation: ({}). Fallback with configured 'verificationkey' was not successful.", e.getMessage());
                    }
                } else {
                    return validationResult;
                }
            }
            if ((validationResult = this.setPublicSignatureForKeyType()).isErroneous()) {
                return validationResult;
            }
            return Validation.validateTokenSignature(token, this.publicKey, this.publicSignature);
        }

        private ValidationResult setSupportedJwtAlgorithm(String tokenAlgorithm) {
            if (tokenAlgorithm != null) {
                this.jwtSignatureAlgorithm = JwtSignatureAlgorithm.fromValue(tokenAlgorithm);
                if (this.jwtSignatureAlgorithm != null) {
                    return ValidationResults.createValid();
                }
                return ValidationResults.createInvalid("Jwt token with signature algorithm '{}' is not supported.", tokenAlgorithm);
            }
            return ValidationResults.createValid();
        }

        private ValidationResult setPublicKey(OAuth2TokenKeyServiceWithCache tokenKeyService, String keyId, URI keyUri, String zoneId) {
            try {
                this.publicKey = tokenKeyService.getPublicKey(this.jwtSignatureAlgorithm, keyId, keyUri, zoneId);
            }
            catch (OAuth2ServiceException e) {
                return ValidationResults.createInvalid("Error retrieving Json Web Keys from Identity Service: {}.", e.getMessage());
            }
            catch (NoSuchAlgorithmException | InvalidKeySpecException e) {
                return ValidationResults.createInvalid("Error creating PublicKey from Json Web Key received from {}: {}.", keyUri, e.getMessage());
            }
            if (this.publicKey == null) {
                return ValidationResults.createInvalid("There is no Json Web Token Key with keyId '{}' and type '{}' found on jwks uri {} for zone '{}' to prove the identity of the Jwt.", keyId, this.jwtSignatureAlgorithm.type(), keyUri, zoneId);
            }
            return ValidationResults.createValid();
        }

        private ValidationResult setPublicSignatureForKeyType() {
            try {
                this.publicSignature = Signature.getInstance(this.jwtSignatureAlgorithm.javaSignature());
                return ValidationResults.createValid();
            }
            catch (NoSuchAlgorithmException noSuchAlgorithmException) {
                return ValidationResults.createInvalid("Jwt token with signature algorithm '{}' can not be verified.", this.jwtSignatureAlgorithm.javaSignature());
            }
        }

        static ValidationResult validateTokenSignature(String token, PublicKey publicKey, Signature publicSignature) {
            String[] tokenHeaderPayloadSignature = DOT.split(token);
            if (tokenHeaderPayloadSignature.length != 3) {
                return ValidationResults.createInvalid("Jwt token does not consist of 'header'.'payload'.'signature'.");
            }
            String headerAndPayload = tokenHeaderPayloadSignature[0] + "." + tokenHeaderPayloadSignature[1];
            try {
                publicSignature.initVerify(publicKey);
                publicSignature.update(headerAndPayload.getBytes(StandardCharsets.UTF_8));
                byte[] decodedSignatureBytes = Base64.getUrlDecoder().decode(tokenHeaderPayloadSignature[2]);
                if (publicSignature.verify(decodedSignatureBytes)) {
                    return ValidationResults.createValid();
                }
                return ValidationResults.createInvalid("Signature of Jwt Token is not valid: the identity provided by the JSON Web Token Key can not be verified.");
            }
            catch (Exception e) {
                return ValidationResults.createInvalid("Error occurred during Json Web Signature Validation: {}.", e.getMessage());
            }
        }
    }
}

