/*
 * Decompiled with CFR 0.152.
 */
package com.sap.cloud.security.token.validation.validators;

import com.sap.cloud.security.config.OAuth2ServiceConfiguration;
import com.sap.cloud.security.token.Token;
import com.sap.cloud.security.token.validation.ValidationResult;
import com.sap.cloud.security.token.validation.ValidationResults;
import com.sap.cloud.security.token.validation.Validator;
import com.sap.cloud.security.token.validation.validators.JsonWebKeyImpl;
import com.sap.cloud.security.token.validation.validators.JwtSignatureAlgorithm;
import com.sap.cloud.security.token.validation.validators.OAuth2TokenKeyServiceWithCache;
import com.sap.cloud.security.token.validation.validators.OidcConfigurationServiceWithCache;
import com.sap.cloud.security.xsuaa.Assertions;
import com.sap.cloud.security.xsuaa.client.DefaultOidcConfigurationService;
import com.sap.cloud.security.xsuaa.client.OAuth2ServiceException;
import java.net.URI;
import java.nio.charset.StandardCharsets;
import java.security.NoSuchAlgorithmException;
import java.security.PublicKey;
import java.security.Signature;
import java.security.spec.InvalidKeySpecException;
import java.util.Base64;
import java.util.regex.Pattern;
import javax.annotation.Nonnull;

public class JwtSignatureValidator
implements Validator<Token> {
    private final OAuth2TokenKeyServiceWithCache tokenKeyService;
    private final OidcConfigurationServiceWithCache oidcConfigurationService;
    private OAuth2ServiceConfiguration configuration;
    private boolean runInLegacyMode;

    JwtSignatureValidator(OAuth2TokenKeyServiceWithCache tokenKeyService, OidcConfigurationServiceWithCache oidcConfigurationService) {
        Assertions.assertNotNull((Object)tokenKeyService, (String)"JwtSignatureValidator requires a tokenKeyService.");
        Assertions.assertNotNull((Object)tokenKeyService, (String)"JwtSignatureValidator requires a oidcConfigurationService.");
        this.tokenKeyService = tokenKeyService;
        this.oidcConfigurationService = oidcConfigurationService;
    }

    JwtSignatureValidator withOAuth2Configuration(OAuth2ServiceConfiguration configuration) {
        this.configuration = configuration;
        return this;
    }

    JwtSignatureValidator runInLegacyMode(boolean isLegacyMode) {
        this.runInLegacyMode = isLegacyMode;
        return this;
    }

    @Override
    public ValidationResult validate(Token token) {
        try {
            String jwksUri = this.getOrRequestJwksUri(token);
            String fallbackPublicKey = null;
            if (this.configuration != null && this.configuration.hasProperty("verificationkey")) {
                fallbackPublicKey = this.configuration.getProperty("verificationkey");
            }
            String keyId = this.getOrDefaultKeyId(token);
            return this.validate(token.getTokenValue(), this.getOrDefaultSignatureAlgorithm(token), keyId, jwksUri, fallbackPublicKey);
        }
        catch (OAuth2ServiceException | IllegalArgumentException e) {
            return ValidationResults.createInvalid("Error occurred during jwks uri determination: {}", e.getMessage());
        }
    }

    @Nonnull
    private String getOrDefaultKeyId(Token token) {
        if (this.runInLegacyMode) {
            return "legacy-token-key";
        }
        if (token.hasHeaderParameter("kid")) {
            return token.getHeaderParameterAsString("kid");
        }
        return "default-kid";
    }

    @Nonnull
    private String getOrDefaultSignatureAlgorithm(Token token) {
        String algHeader = token.getHeaderParameterAsString("alg");
        if (JwtSignatureAlgorithm.fromValue(algHeader) == null) {
            throw new IllegalArgumentException("Jwt token with signature algorithm '" + algHeader + "' is not supported.");
        }
        return JwtSignatureAlgorithm.RS256.value();
    }

    @Nonnull
    private String getOrRequestJwksUri(Token token) throws OAuth2ServiceException, IllegalArgumentException {
        if (this.runInLegacyMode) {
            return this.configuration.getUrl() + "/token_keys";
        }
        if (token.hasHeaderParameter("jku")) {
            return token.getHeaderParameterAsString("jku");
        }
        if (token.hasClaim("iss")) {
            URI discoveryUri = DefaultOidcConfigurationService.getDiscoveryEndpointUri((String)token.getClaimAsString("iss"));
            return this.oidcConfigurationService.getOrRetrieveEndpoints(discoveryUri).getJwksUri().toString();
        }
        throw new IllegalArgumentException("Token signature can not be validated as jwks uri can not be determined: Token does neither provide 'jku' header nor 'issuer' claim.");
    }

    ValidationResult validate(String token, String tokenAlgorithm, String tokenKeyId, String tokenKeysUrl, String fallbackPublicKey) {
        Assertions.assertHasText((String)token, (String)"token must not be null or empty.");
        Assertions.assertHasText((String)tokenKeysUrl, (String)"tokenKeysUrl must not be null or empty.");
        Assertions.assertHasText((String)tokenAlgorithm, (String)"tokenAlgorithm must not be null or empty.");
        return Validation.getInstance().validate(this.tokenKeyService, token, tokenAlgorithm, tokenKeyId, URI.create(tokenKeysUrl), fallbackPublicKey);
    }

    private static class Validation {
        JwtSignatureAlgorithm jwtSignatureAlgorithm;
        PublicKey publicKey;
        Signature publicSignature;
        private static final Pattern DOT = Pattern.compile("\\.", 0);

        private Validation() {
        }

        static Validation getInstance() {
            return new Validation();
        }

        ValidationResult validate(OAuth2TokenKeyServiceWithCache tokenKeyService, String token, String tokenAlgorithm, String tokenKeyId, URI tokenKeysUrl, String fallbackPublicKey) {
            ValidationResult validationResult = this.setSupportedJwtAlgorithm(tokenAlgorithm);
            if (validationResult.isErroneous()) {
                return validationResult;
            }
            validationResult = this.setPublicKey(tokenKeyService, tokenKeyId, tokenKeysUrl);
            if (validationResult.isErroneous()) {
                if (fallbackPublicKey != null) {
                    try {
                        this.publicKey = JsonWebKeyImpl.createPublicKeyFromPemEncodedPublicKey(JwtSignatureAlgorithm.RS256, fallbackPublicKey);
                    }
                    catch (NoSuchAlgorithmException | InvalidKeySpecException e) {
                        return ValidationResults.createInvalid("Error occurred during signature validation: ({}). Fallback with configured verificationkey was not successful.", e.getMessage());
                    }
                } else {
                    return validationResult;
                }
            }
            if ((validationResult = this.setPublicSignatureForKeyType()).isErroneous()) {
                return validationResult;
            }
            return Validation.validateTokenSignature(token, this.publicKey, this.publicSignature);
        }

        private ValidationResult setSupportedJwtAlgorithm(String tokenAlgorithm) {
            if (tokenAlgorithm != null) {
                this.jwtSignatureAlgorithm = JwtSignatureAlgorithm.fromValue(tokenAlgorithm);
                if (this.jwtSignatureAlgorithm != null) {
                    return ValidationResults.createValid();
                }
                return ValidationResults.createInvalid("Jwt token with signature algorithm '{}' is not supported.", tokenAlgorithm);
            }
            this.jwtSignatureAlgorithm = JwtSignatureAlgorithm.RS256;
            return ValidationResults.createValid();
        }

        private ValidationResult setPublicKey(OAuth2TokenKeyServiceWithCache tokenKeyService, String keyId, URI keyUri) {
            try {
                this.publicKey = tokenKeyService.getPublicKey(this.jwtSignatureAlgorithm, keyId, keyUri);
            }
            catch (OAuth2ServiceException e) {
                return ValidationResults.createInvalid("Error retrieving Json Web Keys from Identity Service: {}.", e.getMessage());
            }
            catch (NoSuchAlgorithmException | InvalidKeySpecException e) {
                return ValidationResults.createInvalid("Error creating PublicKey from Json Web Key received from {}: {}.", keyUri, e.getMessage());
            }
            if (this.publicKey == null) {
                return ValidationResults.createInvalid("There is no Json Web Token Key with keyId '{}' and type '{}' to prove the identity of the Jwt.", keyId, this.jwtSignatureAlgorithm.type());
            }
            return ValidationResults.createValid();
        }

        private ValidationResult setPublicSignatureForKeyType() {
            try {
                this.publicSignature = Signature.getInstance(this.jwtSignatureAlgorithm.javaSignature());
                return ValidationResults.createValid();
            }
            catch (NoSuchAlgorithmException noSuchAlgorithmException) {
                return ValidationResults.createInvalid("Jwt token with signature algorithm '{}' can not be verified.", this.jwtSignatureAlgorithm.javaSignature());
            }
        }

        static ValidationResult validateTokenSignature(String token, PublicKey publicKey, Signature publicSignature) {
            String[] tokenHeaderPayloadSignature = DOT.split(token);
            if (tokenHeaderPayloadSignature.length != 3) {
                return ValidationResults.createInvalid("Jwt token does not consist of 'header'.'payload'.'signature'.");
            }
            String headerAndPayload = tokenHeaderPayloadSignature[0] + "." + tokenHeaderPayloadSignature[1];
            try {
                publicSignature.initVerify(publicKey);
                publicSignature.update(headerAndPayload.getBytes(StandardCharsets.UTF_8));
                byte[] decodedSignatureBytes = Base64.getUrlDecoder().decode(tokenHeaderPayloadSignature[2]);
                if (publicSignature.verify(decodedSignatureBytes)) {
                    return ValidationResults.createValid();
                }
                return ValidationResults.createInvalid("Signature of Jwt Token is not valid: the identity provided by the JSON Web Token Key can not be verified.");
            }
            catch (Exception e) {
                return ValidationResults.createInvalid("Error occurred during Json Web Signature Validation: {}.", e.getMessage());
            }
        }
    }
}

