/* 
 * Code generated by Speakeasy (https://speakeasy.com). DO NOT EDIT.
 */
package com.stackone.stackone_client_java.utils.reactive;

import java.util.Iterator;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Flow;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Function;

// Internal API only

/**
 * Utility class for reactive stream operations.
 */
public final class ReactiveUtils {

    private ReactiveUtils() {
        // Utility class
    }

    /**
     * Maps elements from a Flow.Publisher using an asynchronous transformation function.
     * This is useful for transforming HttpResponse<Blob> elements using operations
     * that return CompletableFuture results.
     *
     * @param <T>    the input element type
     * @param <R>    the output element type
     * @param source the source publisher
     * @param mapper the async transformation function (e.g., AsyncOperation::handleResponse)
     * @return a new publisher that emits the mapped elements
     */
    public static <T, R> Flow.Publisher<R> mapAsync(
            Flow.Publisher<T> source,
            Function<T, CompletableFuture<R>> mapper) {

        Objects.requireNonNull(source, "Source publisher cannot be null");
        Objects.requireNonNull(mapper, "Mapper function cannot be null");

        return new AsyncMappingPublisher<>(source, mapper);
    }

    /**
     * Maps elements from a Flow.Publisher using a synchronous transformation function.
     *
     * @param <T>    the input element type
     * @param <R>    the output element type
     * @param source the source publisher
     * @param mapper the transformation function
     * @return a new publisher that emits the mapped elements
     */
    public static <T, R> Flow.Publisher<R> map(
            Flow.Publisher<T> source,
            Function<T, R> mapper) {

        Objects.requireNonNull(source, "Source publisher cannot be null");
        Objects.requireNonNull(mapper, "Mapper function cannot be null");

        return new SyncMappingPublisher<>(source, mapper);
    }

    /**
     * Flattens a stream of collections into a stream of individual items.
     *
     * @param <T>       the input collection type
     * @param <R>       the output element type
     * @param source    the source publisher emitting collections
     * @param flattener the function to extract items from each collection
     * @return a new publisher that emits individual items from the collections
     */
    public static <T, R> Flow.Publisher<R> flatten(
            Flow.Publisher<T> source,
            Function<T, Iterable<R>> flattener) {

        Objects.requireNonNull(source, "Source publisher cannot be null");
        Objects.requireNonNull(flattener, "Flattener function cannot be null");

        return new FlatteningPublisher<>(source, flattener);
    }

    /**
     * Flattens a stream of {@code List<T>} into a stream of individual {@code T} items.
     *
     * @param <T>    the type of elements to emit downstream
     * @param source the source publisher emitting lists
     * @return a new publisher that emits individual items from the lists
     */
    public static <T> Flow.Publisher<T> flatten(Flow.Publisher<List<T>> source) {
        return flatten(source, list -> list);
    }

    /**
     * Wraps a {@code Flow.Publisher<T>} into a {@code Flow.Publisher<List<T>>}.
     *
     * @param source the source publisher
     * @param <T>    the type of elements to emit downstream
     * @return a new publisher that emits lists of items from the source publisher
     */
    public static <T> Flow.Publisher<List<T>> wrapped(Flow.Publisher<T> source) {
        return new SyncMappingPublisher<>(source, List::of);
    }

    /**
     * Concatenates multiple publishers into a single publisher.
     *
     * @param pubs the publishers to concatenate
     * @param <T>  the type of elements to emit downstream
     * @return a new publisher that concatenates the given publishers
     */
    public static <T> Flow.Publisher<T> concat(List<Flow.Publisher<T>> pubs) {
        return new ConcatPublisher<>(pubs);
    }

    /**
     * Internal implementation of async mapping publisher.
     */
    private static class AsyncMappingPublisher<T, R> implements Flow.Publisher<R> {
        private final Flow.Publisher<T> source;
        private final Function<T, CompletableFuture<R>> mapper;

        public AsyncMappingPublisher(Flow.Publisher<T> source, Function<T, CompletableFuture<R>> mapper) {
            this.source = source;
            this.mapper = mapper;
        }

        @Override
        public void subscribe(Flow.Subscriber<? super R> subscriber) {
            source.subscribe(new AsyncMappingSubscriber<>(subscriber, mapper));
        }
    }

    /**
     * Internal implementation of sync mapping publisher.
     */
    private static class SyncMappingPublisher<T, R> implements Flow.Publisher<R> {
        private final Flow.Publisher<T> source;
        private final Function<T, R> mapper;

        public SyncMappingPublisher(Flow.Publisher<T> source, Function<T, R> mapper) {
            this.source = source;
            this.mapper = mapper;
        }

        @Override
        public void subscribe(Flow.Subscriber<? super R> subscriber) {
            source.subscribe(new SyncMappingSubscriber<>(subscriber, mapper));
        }
    }

    /**
     * Internal implementation of flattening publisher.
     */
    private static class FlatteningPublisher<T, R> implements Flow.Publisher<R> {
        private final Flow.Publisher<T> source;
        private final Function<T, Iterable<R>> flattener;

        public FlatteningPublisher(Flow.Publisher<T> source, Function<T, Iterable<R>> flattener) {
            this.source = source;
            this.flattener = flattener;
        }

        @Override
        public void subscribe(Flow.Subscriber<? super R> subscriber) {
            source.subscribe(new FlatteningSubscriber<>(subscriber, flattener));
        }
    }

    /**
     * Internal implementation of concatenating publisher.
     */
    private static final class ConcatPublisher<T> implements Flow.Publisher<T> {
        private final List<Flow.Publisher<T>> pubs;

        ConcatPublisher(List<Flow.Publisher<T>> pubs) {
            this.pubs = List.copyOf(pubs);
        }

        @SuppressWarnings("unchecked")
        @Override
        public void subscribe(Flow.Subscriber<? super T> downstream) {
            downstream.onSubscribe(new ConcatSubscription<>((Flow.Subscriber<T>) downstream, pubs.iterator()));
        }

    }

    /**
     * Subscriber that handles async mapping transformations.
     */
    private static class AsyncMappingSubscriber<T, R> implements Flow.Subscriber<T> {
        private final Flow.Subscriber<? super R> downstream;
        private final Function<T, CompletableFuture<R>> mapper;
        private final AtomicBoolean cancelled = new AtomicBoolean(false);
        private final AtomicLong pendingDemand = new AtomicLong(0);
        private final AtomicLong pendingCompletes = new AtomicLong(0);
        private Flow.Subscription upstream;
        private volatile boolean upstreamCompleted = false;

        public AsyncMappingSubscriber(Flow.Subscriber<? super R> downstream, Function<T, CompletableFuture<R>> mapper) {
            this.downstream = downstream;
            this.mapper = mapper;
        }

        @Override
        public void onSubscribe(Flow.Subscription subscription) {
            this.upstream = subscription;
            downstream.onSubscribe(new Flow.Subscription() {
                @Override
                public void request(long n) {
                    if (n <= 0) {
                        downstream.onError(new IllegalArgumentException("Request count must be positive"));
                        return;
                    }

                    long currentDemand = pendingDemand.addAndGet(n);
                    if (currentDemand < 0) {
                        pendingDemand.set(Long.MAX_VALUE);
                    }

                    upstream.request(n);
                }

                @Override
                public void cancel() {
                    cancelled.set(true);
                    upstream.cancel();
                }
            });
        }

        @Override
        public void onNext(T item) {
            if (cancelled.get()) return;

            try {
                pendingCompletes.incrementAndGet();
                CompletableFuture<R> future = mapper.apply(item);
                future.whenComplete((result, error) -> {
                    if (cancelled.get()) {
                        return;
                    }

                    if (error != null) {
                        cancelled.set(true);
                        downstream.onError(error);
                    } else if (pendingDemand.get() > 0) {
                        pendingDemand.decrementAndGet();
                        downstream.onNext(result);
                    }

                    // Check if we should complete after this async operation
                    if (pendingCompletes.decrementAndGet() == 0 && upstreamCompleted) {
                        downstream.onComplete();
                    }
                });
            } catch (Exception e) {
                if (!cancelled.get()) {
                    cancelled.set(true);
                    downstream.onError(e);
                }
            }
        }

        @Override
        public void onError(Throwable throwable) {
            if (!cancelled.get()) {
                downstream.onError(throwable);
            }
        }

        @Override
        public void onComplete() {
            upstreamCompleted = true;
            if (!cancelled.get() && pendingCompletes.get() == 0) {
                downstream.onComplete();
            }
        }
    }

    /**
     * Subscriber that handles sync mapping transformations.
     */
    private static class SyncMappingSubscriber<T, R> implements Flow.Subscriber<T> {
        private final Flow.Subscriber<? super R> downstream;
        private final Function<T, R> mapper;
        private Flow.Subscription upstream;
        private volatile boolean cancelled = false;

        public SyncMappingSubscriber(Flow.Subscriber<? super R> downstream, Function<T, R> mapper) {
            this.downstream = downstream;
            this.mapper = mapper;
        }

        @Override
        public void onSubscribe(Flow.Subscription subscription) {
            this.upstream = subscription;
            downstream.onSubscribe(new Flow.Subscription() {
                @Override
                public void request(long n) {
                    if (!cancelled) {
                        upstream.request(n);
                    }
                }

                @Override
                public void cancel() {
                    cancelled = true;
                    upstream.cancel();
                }
            });
        }

        @Override
        public void onNext(T item) {
            if (cancelled) return;

            try {
                R result = mapper.apply(item);
                downstream.onNext(result);
            } catch (Exception e) {
                cancelled = true;
                upstream.cancel();
                downstream.onError(e);
            }
        }

        @Override
        public void onError(Throwable throwable) {
            if (!cancelled) {
                downstream.onError(throwable);
            }
        }

        @Override
        public void onComplete() {
            if (!cancelled) {
                downstream.onComplete();
            }
        }
    }

    /**
     * Subscriber that handles flattening transformations.
     */
    private static class FlatteningSubscriber<T, R> implements Flow.Subscriber<T> {
        private final Flow.Subscriber<? super R> downstream;
        private final Function<T, Iterable<R>> flattener;
        private Flow.Subscription upstreamSubscription;
        private volatile boolean cancelled = false;

        public FlatteningSubscriber(Flow.Subscriber<? super R> downstream, Function<T, Iterable<R>> flattener) {
            this.downstream = Objects.requireNonNull(downstream, "Downstream subscriber cannot be null");
            this.flattener = flattener;
        }

        @Override
        public void onSubscribe(Flow.Subscription subscription) {
            this.upstreamSubscription = Objects.requireNonNull(subscription, "Upstream subscription cannot be null");
            downstream.onSubscribe(new Flow.Subscription() {
                @Override
                public void request(long n) {
                    if (n <= 0) {
                        downstream.onError(new IllegalArgumentException("Demand must be positive: " + n));
                        return;
                    }

                    if (!cancelled && upstreamSubscription != null) {
                        upstreamSubscription.request(n);
                    }
                }

                @Override
                public void cancel() {
                    cancelled = true;
                    if (upstreamSubscription != null) {
                        upstreamSubscription.cancel();
                    }
                }
            });
        }

        @Override
        public void onNext(T item) {
            if (cancelled) {
                return;
            }

            try {
                Iterable<R> items = flattener.apply(item);
                for (R flattenedItem : items) {
                    if (cancelled) {
                        break;
                    }
                    downstream.onNext(flattenedItem);
                }
            } catch (Exception e) {
                if (!cancelled) {
                    cancelled = true;
                    upstreamSubscription.cancel();
                    downstream.onError(e);
                }
            }
        }

        @Override
        public void onError(Throwable throwable) {
            if (!cancelled) {
                downstream.onError(throwable);
            }
        }

        @Override
        public void onComplete() {
            if (!cancelled) {
                downstream.onComplete();
            }
        }
    }

    /**
     * Subscriber that handles concatenating publishers.
     */
    private static final class ConcatSubscription<T> implements Flow.Subscription {
        private final Flow.Subscriber<T> downstream;
        private final Iterator<Flow.Publisher<T>> it;

        private Flow.Subscription upstream;
        private long demand = 0L;
        private boolean cancelled = false;
        private boolean completed = false;

        public ConcatSubscription(Flow.Subscriber<T> downstream,
                                  Iterator<Flow.Publisher<T>> it) {
            this.downstream = downstream;
            this.it = it;
        }


        @Override
        public synchronized void request(long n) {
            if (cancelled || completed || n <= 0) return;
            demand = addCap(demand, n);
            if (upstream == null) {
                subscribeNext();
            } else {
                upstream.request(n);
            }
        }

        @Override
        public synchronized void cancel() {
            cancelled = true;
            if (upstream != null) upstream.cancel();
        }

        private void subscribeNext() {
            if (cancelled) return;
            if (!it.hasNext()) {
                completed = true;
                downstream.onComplete();
                return;
            }
            Flow.Publisher<T> next = it.next();
            next.subscribe(new Upstream());
        }

        private final class Upstream implements Flow.Subscriber<T> {
            @Override
            public void onSubscribe(Flow.Subscription s) {
                synchronized (ConcatSubscription.this) {
                    upstream = s;
                    if (demand > 0) s.request(demand);
                }
            }

            @Override
            public void onNext(T item) {
                long afterDecrement;
                synchronized (ConcatSubscription.this) {
                    if (demand == 0) return; // should not happen if upstream respects RS
                    demand--;
                    afterDecrement = demand;
                }
                downstream.onNext(item);
                // no need to request here; downstream will call request() again if needed
            }

            @Override
            public void onError(Throwable t) {
                downstream.onError(t);
            }

            @Override
            public void onComplete() {
                synchronized (ConcatSubscription.this) {
                    upstream = null;
                    if (cancelled) return;
                }
                // carry over remaining demand to next publisher
                subscribeNext();
            }
        }

        private static long addCap(long a, long b) {
            long r = a + b;
            return (r < 0L) ? Long.MAX_VALUE : r;
        }
    }
}
