package com.stackspot.plugin.openapi.template

import com.samskivert.mustache.Mustache
import com.stackspot.plugin.openapi.GenerateOpenApiTaskConfigurator.Framework
import com.stackspot.plugin.util.KotlinLang
import com.stackspot.plugin.util.extensions.mkdirs
import com.stackspot.plugin.util.extensions.toCamelCase
import com.stackspot.plugin.util.extensions.toCamelCaseCapitalized
import com.stackspot.plugin.util.extensions.toKFunctionsWithMicronaut
import com.stackspot.plugin.util.extensions.toKFunctionsWithSpring
import io.swagger.v3.oas.models.OpenAPI
import io.swagger.v3.oas.models.media.ArraySchema
import io.swagger.v3.oas.models.media.ComposedSchema
import io.swagger.v3.oas.models.media.Schema
import io.swagger.v3.parser.util.RefUtils
import java.io.File
import java.io.FileWriter
import java.io.InputStream
import java.io.InputStreamReader
import java.io.Writer
import java.util.StringTokenizer

open class OpenApiTemplate(
    private val outputDir: String,
    private val basePackage: String,
    private val framework: Framework
) {

    open fun createModels(openAPI: OpenAPI) {
        val packageName = "$basePackage.models"
        openAPI.components?.schemas?.forEach { (serviceName, schema) ->

            if (schema?.type.equals("object")) {

                val dirHome = "$outputDir${File.separator}${packageName.replace(".", File.separator)}".apply {
                    mkdirs()
                }

                val modelName = serviceName.toCamelCaseCapitalized()
                val file = File(dirHome + File.separator + "$modelName.kt")

                val constructorArgs = generateConstructorArgs(schema, openAPI.components.schemas)

                val model = when (framework) {
                    Framework.MICRONAUT -> kClassModelMicronaut(modelName, packageName, constructorArgs)
                    Framework.SPRING -> kClassModelSpring(modelName, packageName, constructorArgs)
                }

                KotlinLang.Utils.typeMapping[modelName] =
                    "$packageName.$modelName"

                KotlinLang.Utils.importMap[modelName] =
                    "$packageName.$modelName"

                executeTemplate("model.mustache", FileWriter(file), model)
            }
        }
    }

    fun createOperations(serviceName: String, openAPI: OpenAPI) {
        val packageName = "$basePackage.operations"

        val dirHome = "$outputDir${File.separator}${packageName.replace(".", File.separator)}".apply {
            mkdirs()
        }

        val name = "${serviceName.toCamelCaseCapitalized()}Operations"
        val file = File(dirHome + File.separator + "$name.kt")

        val operation = when (framework) {
            Framework.MICRONAUT -> kClassOperationMicronaut(name, packageName, openAPI)
            Framework.SPRING -> kClassOperationSpring(name, packageName, openAPI)
        }

        executeTemplate("operation.mustache", FileWriter(file), operation)
    }

    private fun kClassOperationSpring(
        name: String,
        packageName: String,
        openAPI: OpenAPI
    ) = KotlinLang.KClass(
        name = name,
        modifier = KotlinLang.KModifier.PUBLIC,
        isInterface = true,
        packageName = packageName,
        functions = openAPI.paths.toKFunctionsWithSpring(),
        extraImports = listOf("httpResponseForSpring"),
        framework = Framework.SPRING
    )

    private fun kClassOperationMicronaut(
        name: String,
        packageName: String,
        openAPI: OpenAPI
    ) = KotlinLang.KClass(
        name = name,
        modifier = KotlinLang.KModifier.PUBLIC,
        isInterface = true,
        packageName = packageName,
        functions = openAPI.paths.toKFunctionsWithMicronaut(),
        extraImports = listOf("httpResponseForMicronaut"),
        framework = Framework.MICRONAUT
    )

    private fun kClassModelMicronaut(
        modelName: String,
        packageName: String,
        constructorArgs: List<KotlinLang.KProperty>?
    ): KotlinLang.KClass {
        return KotlinLang.KClass(
            name = modelName,
            packageName = packageName,
            isData = true,
            constructorArgs = constructorArgs,
            annotations = listOf(KotlinLang.KAnnotation("Introspected")),
            framework = Framework.MICRONAUT
        )
    }

    private fun kClassModelSpring(
        modelName: String,
        packageName: String,
        constructorArgs: List<KotlinLang.KProperty>?
    ): KotlinLang.KClass {
        return KotlinLang.KClass(
            name = modelName,
            packageName = packageName,
            isData = true,
            constructorArgs = constructorArgs,
            framework = Framework.SPRING
        )
    }

    private fun generateConstructorArgs(
        schema: Schema<Any>,
        schemas: MutableMap<String, Schema<Any>>
    ): List<KotlinLang.KProperty> {

        val constructorArgs = mutableListOf<KotlinLang.KProperty>()

        schema.properties?.map { (propKey, propValue) ->

            propValue.`$ref`?.let {
                val schemaRef = findSchemaRef(it, schemas)
                constructorArgs.add(createKProperty(schema, propKey, schemaRef))
            } ?: constructorArgs.add(createKProperty(schema, propKey, propValue))
        } ?: constructorArgs.addAll(extractAllOff(schema, schemas))

        return constructorArgs
    }

    private fun createKProperty(
        schema: Schema<Any>,
        propKey: String,
        propValue: Schema<Any>
    ): KotlinLang.KProperty {
        val isRequired = schema.required?.contains(propKey) ?: false
        val (isFormatted, type) = getSchemaTypes(propValue)
        val (outType, kType) = getKotlinType(propValue)

        return KotlinLang.KProperty(
            name = propKey.toCamelCase(),
            type = kType,
            hasOutType = outType.first,
            outType = outType.second,
            isNullable = !isRequired,
            annotations = mutableListOf<KotlinLang.KAnnotation>().apply {
                if (isRequired) {
                    when (type) {
                        "string" -> {
                            if (propValue.minLength?.let { it > 0 } == true) {
                                add(KotlinLang.KAnnotation("NotBlank"))
                            }
                        }
                        else -> add(KotlinLang.KAnnotation("NotNull"))
                    }
                }
                if (isFormatted) {
                    add(
                        KotlinLang.KAnnotation(
                            type = "JsonFormat",
                            hasExtraProperties = true,
                            extraProperties = listOf(
                                KotlinLang.Properties(
                                    key = "shape",
                                    value = "JsonFormat.Shape.STRING",
                                    hasComma = false
                                ),
                                KotlinLang.Properties(
                                    key = "pattern",
                                    value = propValue.pattern ?: "",
                                    hasComma = true
                                )
                            )
                        )
                    )
                }
            }
        )
    }

    private fun findSchemaRef(
        ref: String,
        schemas: MutableMap<String, Schema<Any>>
    ): Schema<Any> {
        val stringTokenizer = StringTokenizer(ref.reversed())
        val refSchemaName = stringTokenizer.nextToken("/").reversed()
        val findSchema = schemas.filter { it.key == refSchemaName }
        return findSchema[refSchemaName]!!
    }

    private fun extractAllOff(
        schema: Schema<Any>,
        schemas: MutableMap<String, Schema<Any>>
    ): List<KotlinLang.KProperty> {

        val args = mutableListOf<KotlinLang.KProperty>()

        val schemaAllOf = (schema as ComposedSchema).allOf

        schemaAllOf.forEach { schemaAllOf ->
            schemaAllOf.properties?.let {
                args.addAll(generateConstructorArgs(schemaAllOf, schemas))
            } ?: args.addAll(generateConstructorArgs(findSchemaRef(schemaAllOf.`$ref`, schemas), schemas))
        }
        return args
    }

    private fun getSchemaTypes(propValue: Schema<Any>): Pair<Boolean, String> {
        return when (propValue.type) {
            "string" -> {
                return when (propValue.format) {
                    null -> false to propValue.type
                    "date-time", "date", "dateTime" -> true to propValue.format
                    else -> false to propValue.format
                }
            }
            else -> false to propValue.type
        }
    }

    private fun getKotlinType(propValue: Schema<Any>): Pair<Pair<Boolean, String?>, String> {
        return when (propValue.type) {
            "array" -> {
                val ouType = RefUtils.computeDefinitionName((propValue as ArraySchema).items.`$ref`)
                (true to ouType) to "List<$ouType>"
            }
            else -> {
                val (_, type) = getSchemaTypes(propValue)
                (false to null) to KotlinLang.Utils.typeMapping[type]!!
            }
        }
    }

    private fun executeTemplate(template: String, out: Writer?, data: Any?) {
        Mustache.compiler().escapeHTML(false).withLoader { name ->
            InputStreamReader(
                readFileAsLinesUsingGetResourceAsStream("/$name.mustache")
            )
        }.compile(InputStreamReader(readFileAsLinesUsingGetResourceAsStream("/$template"))).execute(data, out)
        out?.flush()
    }

    private fun readFileAsLinesUsingGetResourceAsStream(fileName: String): InputStream =
        this::class.java.getResourceAsStream(fileName)
}
