/*
 * SPDX-FileCopyrightText: none
 * SPDX-License-Identifier: CC0-1.0
 */

package dev.metaschema.cli.processor.completion;

import org.apache.commons.cli.Option;

import java.util.Collection;
import java.util.List;
import java.util.stream.Collectors;

import dev.metaschema.cli.processor.command.ExtraArgument;
import dev.metaschema.cli.processor.command.ICommand;
import dev.metaschema.core.util.ObjectUtils;
import edu.umd.cs.findbugs.annotations.NonNull;

/**
 * Generates shell completion scripts for Bash and Zsh.
 * <p>
 * This generator introspects registered commands and their options to produce
 * completion scripts that provide intelligent tab-completion for command-line
 * tools built on the cli-processor framework.
 */
public class CompletionScriptGenerator {

  /**
   * Supported shell types.
   */
  public enum Shell {
    /** Bash shell. */
    BASH,
    /** Zsh shell. */
    ZSH
  }

  @NonNull
  private final String programName;
  @NonNull
  private final List<ICommand> commands;

  /**
   * Construct a new generator.
   *
   * @param programName
   *          the name of the CLI program
   * @param commands
   *          the top-level commands to include in completion
   */
  public CompletionScriptGenerator(
      @NonNull String programName,
      @NonNull List<ICommand> commands) {
    this.programName = programName;
    this.commands = commands;
  }

  /**
   * Get the program name.
   *
   * @return the program name
   */
  @NonNull
  public String getProgramName() {
    return programName;
  }

  /**
   * Get the commands.
   *
   * @return the commands
   */
  @NonNull
  public List<ICommand> getCommands() {
    return commands;
  }

  /**
   * Generate a Bash completion script.
   *
   * @return the bash completion script
   */
  @NonNull
  public String generateBashCompletion() {
    StringBuilder sb = new StringBuilder();
    String funcName = "_" + sanitizeFunctionName(programName);

    // Header
    sb.append("# Bash completion for ").append(programName).append("\n");
    sb.append("# Generated by metaschema cli-processor\n\n");

    // Fallback for systems without bash-completion
    sb.append("# Provide fallback if _init_completion is not available\n");
    sb.append("if ! type _init_completion &>/dev/null; then\n");
    sb.append("    _init_completion() {\n");
    sb.append("        COMPREPLY=()\n");
    sb.append("        cur=\"${COMP_WORDS[COMP_CWORD]}\"\n");
    sb.append("        prev=\"${COMP_WORDS[COMP_CWORD-1]}\"\n");
    sb.append("        words=(\"${COMP_WORDS[@]}\")\n");
    sb.append("        cword=$COMP_CWORD\n");
    sb.append("    }\n");
    sb.append("fi\n\n");

    // Fallback for _filedir if not available
    sb.append("# Provide fallback if _filedir is not available\n");
    sb.append("if ! type _filedir &>/dev/null; then\n");
    sb.append("    _filedir() {\n");
    sb.append("        COMPREPLY=($(compgen -f -- \"$cur\"))\n");
    sb.append("    }\n");
    sb.append("fi\n\n");

    // Main function
    sb.append(funcName).append("() {\n");
    sb.append("    local cur prev words cword\n");
    sb.append("    _init_completion || return\n\n");

    // Top-level commands
    String commandNames = commands.stream()
        .map(ICommand::getName)
        .collect(Collectors.joining(" "));
    sb.append("    local commands=\"").append(commandNames).append("\"\n\n");

    // Case statement for command-specific completions
    sb.append("    case \"${words[1]}\" in\n");
    for (ICommand cmd : commands) {
      generateBashCommandCase(sb, cmd, 2);
    }
    sb.append("        *)\n");
    sb.append("            COMPREPLY=($(compgen -W \"$commands\" -- \"$cur\"))\n");
    sb.append("            ;;\n");
    sb.append("    esac\n");
    sb.append("}\n\n");

    // Register completion
    sb.append("complete -F ").append(funcName).append(" ").append(programName).append("\n");

    return ObjectUtils.notNull(sb.toString());
  }

  /**
   * Generate a Zsh completion script.
   *
   * @return the zsh completion script
   */
  @NonNull
  public String generateZshCompletion() {
    StringBuilder sb = new StringBuilder();
    String funcName = "_" + sanitizeFunctionName(programName);

    // Header
    sb.append("#compdef ").append(programName).append("\n\n");
    sb.append("# Zsh completion for ").append(programName).append("\n");
    sb.append("# Generated by metaschema cli-processor\n\n");

    // Main function
    sb.append(funcName).append("() {\n");
    sb.append("    local -a commands\n");
    sb.append("    commands=(\n");
    for (ICommand cmd : commands) {
      String escaped = cmd.getDescription().replace("'", "'\\''");
      sb.append("        '").append(cmd.getName()).append(":").append(escaped).append("'\n");
    }
    sb.append("    )\n\n");

    sb.append("    _arguments -C \\\n");
    sb.append("        '1:command:->command' \\\n");
    sb.append("        '*::arg:->args'\n\n");

    sb.append("    case $state in\n");
    sb.append("        command)\n");
    sb.append("            _describe 'command' commands\n");
    sb.append("            ;;\n");
    sb.append("        args)\n");
    sb.append("            case $words[1] in\n");
    for (ICommand cmd : commands) {
      generateZshCommandCase(sb, cmd);
    }
    sb.append("            esac\n");
    sb.append("            ;;\n");
    sb.append("    esac\n");
    sb.append("}\n\n");

    sb.append(funcName).append(" \"$@\"\n");

    return ObjectUtils.notNull(sb.toString());
  }

  private void generateBashCommandCase(StringBuilder sb, ICommand cmd, int depth) {
    String indent = "        ".repeat(Math.max(1, depth - 1));
    sb.append(indent).append(cmd.getName()).append(")\n");

    Collection<? extends Option> options = cmd.gatherOptions();
    List<ExtraArgument> extraArgs = cmd.getExtraArguments();
    Collection<ICommand> subCommands = cmd.getSubCommands();

    // Handle sub-commands
    if (!subCommands.isEmpty()) {
      String subCmdNames = subCommands.stream()
          .map(ICommand::getName)
          .collect(Collectors.joining(" "));

      sb.append(indent).append("    if [[ ${#words[@]} -eq ").append(depth).append(" ]]; then\n");
      sb.append(indent).append("        COMPREPLY=($(compgen -W \"").append(subCmdNames).append("\" -- \"$cur\"))\n");
      sb.append(indent).append("    else\n");
      sb.append(indent).append("        case \"${words[").append(depth).append("]}\" in\n");
      for (ICommand subCmd : subCommands) {
        generateBashCommandCase(sb, subCmd, depth + 1);
      }
      sb.append(indent).append("        esac\n");
      sb.append(indent).append("    fi\n");
    } else {
      // Generate option completions
      String optionNames = options.stream()
          .map(CompletionScriptGenerator::getBashOptionName)
          .collect(Collectors.joining(" "));

      sb.append(indent).append("    if [[ \"$cur\" == -* ]]; then\n");
      sb.append(indent).append("        COMPREPLY=($(compgen -W \"").append(optionNames).append("\" -- \"$cur\"))\n");
      sb.append(indent).append("    else\n");

      // Check previous word for option-specific completion
      boolean hasOptionCompletion = false;
      for (Option opt : options) {
        if (opt.hasArg()) {
          String completion = getCompletionForOption(opt, Shell.BASH);
          if (!completion.isEmpty()) {
            if (!hasOptionCompletion) {
              sb.append(indent).append("        case \"$prev\" in\n");
              hasOptionCompletion = true;
            }
            String optNames = getBashOptionName(opt);
            sb.append(indent).append("            ").append(optNames.replace(" ", "|")).append(")\n");
            sb.append(indent).append("                COMPREPLY=($(").append(completion).append(" -- \"$cur\"))\n");
            sb.append(indent).append("                ;;\n");
          }
        }
      }
      if (hasOptionCompletion) {
        sb.append(indent).append("            *)\n");
      }

      // Default: file completion for extra arguments
      String extraArgCompletion = getDefaultExtraArgumentCompletion(extraArgs, Shell.BASH);
      if (!extraArgCompletion.isEmpty()) {
        sb.append(indent).append("                ").append(extraArgCompletion).append("\n");
      }

      if (hasOptionCompletion) {
        sb.append(indent).append("                ;;\n");
        sb.append(indent).append("        esac\n");
      }
      sb.append(indent).append("    fi\n");
    }
    sb.append(indent).append("    ;;\n");
  }

  private static void generateZshCommandCase(StringBuilder sb, ICommand cmd) {
    sb.append("                ").append(cmd.getName()).append(")\n");

    Collection<? extends Option> options = cmd.gatherOptions();
    List<ExtraArgument> extraArgs = cmd.getExtraArguments();

    sb.append("                    _arguments \\\n");

    // Add options
    for (Option opt : options) {
      String optSpec = getZshOptionSpec(opt);
      sb.append("                        ").append(optSpec).append(" \\\n");
    }

    // Add extra arguments
    int argNum = 1;
    for (ExtraArgument arg : extraArgs) {
      String completion = getCompletionForExtraArgument(arg, Shell.ZSH);
      String required = arg.isRequired() ? "" : "::";
      if (completion.isEmpty()) {
        completion = " ";
      }
      sb.append("                        '").append(required.isEmpty() ? argNum : "").append(required)
          .append(arg.getName()).append(":").append(completion).append("' \\\n");
      argNum++;
    }

    // Remove trailing backslash from last line
    int lastBackslash = sb.lastIndexOf(" \\");
    if (lastBackslash > 0) {
      sb.delete(lastBackslash, lastBackslash + 2);
    }
    sb.append("\n");

    sb.append("                    ;;\n");
  }

  @NonNull
  private static String getBashOptionName(Option opt) {
    StringBuilder sb = new StringBuilder();
    if (opt.getOpt() != null) {
      sb.append("-").append(opt.getOpt());
    }
    if (opt.getLongOpt() != null) {
      if (sb.length() > 0) {
        sb.append(" ");
      }
      sb.append("--").append(opt.getLongOpt());
    }
    return ObjectUtils.notNull(sb.toString());
  }

  @NonNull
  private static String getZshOptionSpec(Option opt) {
    StringBuilder sb = new StringBuilder("'");
    if (opt.getLongOpt() != null) {
      sb.append("--").append(opt.getLongOpt());
    } else if (opt.getOpt() != null) {
      sb.append("-").append(opt.getOpt());
    }

    String desc = opt.getDescription();
    if (desc != null) {
      desc = desc.replace("'", "'\\''");
      sb.append("[").append(desc).append("]");
    }

    if (opt.hasArg()) {
      String argName = opt.getArgName();
      if (argName == null) {
        argName = "arg";
      }
      String completion = getCompletionForOption(opt, Shell.ZSH);
      sb.append(":").append(argName).append(":").append(completion);
    }

    sb.append("'");
    return ObjectUtils.notNull(sb.toString());
  }

  @NonNull
  private static String getCompletionForOption(Option opt, Shell shell) {
    Object typeObj = opt.getType();
    Class<?> type = typeObj instanceof Class ? (Class<?>) typeObj : null;
    ICompletionType completion = CompletionTypeRegistry.lookup(type);
    if (completion != null) {
      return shell == Shell.BASH ? completion.getBashCompletion() : completion.getZshCompletion();
    }
    return "";
  }

  @NonNull
  private static String getCompletionForExtraArgument(ExtraArgument arg, Shell shell) {
    Class<?> type = arg.getType();
    ICompletionType completion = CompletionTypeRegistry.lookup(type);
    if (completion != null) {
      return shell == Shell.BASH ? completion.getBashCompletion() : completion.getZshCompletion();
    }
    return "";
  }

  @NonNull
  private static String getDefaultExtraArgumentCompletion(List<ExtraArgument> args, Shell shell) {
    // Use the first argument's type, or default to file completion
    if (!args.isEmpty()) {
      String completion = getCompletionForExtraArgument(args.get(0), shell);
      if (!completion.isEmpty()) {
        return completion;
      }
    }
    // Default to file completion
    if (shell == Shell.BASH) {
      return "_filedir";
    }
    return "_files";
  }

  @NonNull
  private static String sanitizeFunctionName(String name) {
    return ObjectUtils.notNull(name.replaceAll("[^a-zA-Z0-9_]", "_"));
  }
}
