package fi.evolver.script.app;

import java.nio.file.Files;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import com.google.gson.Gson;

import fi.evolver.script.Dialog;
import fi.evolver.script.Dialog.DialogEntry;
import fi.evolver.script.Shell.Command;
import fi.evolver.script.Step;
import fi.evolver.script.app.Nvm.Node;

public class Bitwarden {
	private static final Pattern REGEX_BW_SESSION = Pattern.compile("\\$ export BW_SESSION=\"(?<sessionId>[^\"]+)\"\n");
	private static final Gson GSON = new Gson();

	private static String email = System.getenv("BITWARDEN_EMAIL");
	private static String sessionId;

	private static Node node;

	private static final DialogEntry DIALOG_EMAIL = Dialog.textField("Email address");
	private static final DialogEntry DIALOG_PASSWORD = Dialog.passwordField("Password");
	private static final DialogEntry DIALOG_MFA = Dialog.passwordField("MFA");


	public static void install() {
		try (Step step = Step.start("Bitwarden: install")) {
			node = Nvm.use("20");
			if (Files.isExecutable(node.bin().resolve("bw"))) {
				step.skip("Already installed");
				return;
			}
			node.npm("install", "-g", "@bitwarden/cli");
		}
	}


	private static void startSession(Step step) {
		if (sessionId != null)
			return;

		install();

		boolean needLogin = command("status").contains("\"status\":\"unauthenticated\"");

		Dialog dialog = new Dialog("Bitwarden login");
		if (needLogin && email == null)
			dialog.add(DIALOG_EMAIL);

		dialog.add(DIALOG_PASSWORD);
		if (needLogin)
			dialog.add(DIALOG_MFA);

		Map<DialogEntry, String> values = dialog.show();

		String result;
		if (needLogin)
			result = command("login", values.getOrDefault(DIALOG_EMAIL, email), values.get(DIALOG_PASSWORD), "--method", "0", "--code", values.get(DIALOG_MFA));
		else
			result = command("unlock", values.get(DIALOG_PASSWORD));

		Matcher matcher = REGEX_BW_SESSION.matcher(result);
		if (!matcher.find())
			step.fail("Failed opening the Bitwarden vault");

		sessionId = matcher.group("sessionId");
	}


	private static String command(String... args) {
		List<String> fullCommand = new ArrayList<>();
		fullCommand.add(node.bin().resolve("bw").toString());
		fullCommand.addAll(Arrays.asList(args));
		Command command = node.command(fullCommand);
		if (sessionId != null)
			command.env("BW_SESSION", sessionId);
		return command.run().stdout();
	}


	public static void sync() {
		try (Step step = Step.start("Bitwarden: sync")) {
			startSession(step);
			command("sync");
		}
	}


	public static BitwardenItem getItem(String id) {
		try (Step step = Step.start("Bitwarden: get item %s".formatted(id))) {
			startSession(step);
			return GSON.fromJson(command("get", "item", id), BitwardenItem.class);
		}
	}


	public record BitwardenItem(
			String id,
			String name,
			String notes,
			List<Field> fields,
			Login login) {

		public String getField(String name) {
			return fields.stream()
					.filter(f -> name.equals(f.name()))
					.map(Field::value)
					.findFirst()
					.orElse(null);
		}
	}


	public record Login(
			String username,
			String password) {

	}

	public record Field(
			String name,
			String value) {

	}

}
