package fi.evolver.script.app;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import fi.evolver.script.Shell;
import fi.evolver.script.Step;

public class NetworkManager {

    public enum VpnType {
        OPENVPN,
        WIREGUARD,
        PPTP,
        L2TP,
        SSTP,
        OPENCONNECT,
    }


    /**
     * Install NetworkManager and the listed VPN plugins.
     *
     * @param vpnTypes
     */
    public static void installNetworkManager(VpnType... vpnTypes) {
        try (Step step = Step.start("NetworkManager: install")) {
            List<String> packages = new ArrayList<>();
            packages.add("network-manager");

            for (VpnType vpnType : vpnTypes) {
                switch (vpnType) {
                    case OPENVPN -> packages.add("network-manager-openvpn-gnome");
                    case WIREGUARD -> packages.add("wireguard");
                    case PPTP -> packages.add("network-manager-pptp-gnome");
                    case L2TP -> packages.add("network-manager-l2tp-gnome");
                    case SSTP -> packages.add("network-manager-sstp-gnome");
                    case OPENCONNECT -> packages.add("network-manager-openconnect-gnome");
                }
            }

            Apt.install(packages);
        }
    }


    /**
     * Create a new Cisco AnyConnect / OpenConnect VPN connection.
     *
     * @param connectionName    The name for the VPN connection
     * @param gateway           The VPN gateway URL (e.g., "vpn.company.com")
     * @param vpnDataOverrides  Overrides for vpn.data properties
     */
    public static void addOpenConnectConnection(String connectionName, String gateway, Map<String, Object> vpnDataOverrides) {
        try (Step step = Step.start("NetworkManager: create OpenConnect VPN connection: " + connectionName)) {
            if (hasConnection(connectionName)) {
                step.skip("already exists");
                return;
            }

            List<String> command = new ArrayList<>();
            command.add("nmcli");
            command.add("connection");
            command.add("add");
            command.add("type");
            command.add("vpn");
            command.add("con-name");
            command.add(connectionName);
            command.add("vpn-type");
            command.add("openconnect");

            Map<String, Object> vpnProperties = new LinkedHashMap<>();
            vpnProperties.put("authtype", "password");
            vpnProperties.put("autoconnect-flags", "0");
            vpnProperties.put("certsigs-flags", "0");
            vpnProperties.put("cookie-flags", "2");
            vpnProperties.put("disable_udp", "no");
            vpnProperties.put("enable_csd_trojan", "no");
            vpnProperties.put("gateway", gateway);
            vpnProperties.put("gateway-flags", "2");
            vpnProperties.put("gwcert-flags", "2");
            vpnProperties.put("lasthost-flags", "0");
            vpnProperties.put("pem_passphrase_fsid", "no");
            vpnProperties.put("prevent_invalid_cert", "no");
            vpnProperties.put("protocol", "anyconnect");
            vpnProperties.put("resolve-flags", "2");
            vpnProperties.put("stoken_source", "totp");
            vpnProperties.put("useragent", "AnyConnect");
            vpnProperties.put("xmlconfig-flags", "0");
            vpnProperties.putAll(vpnDataOverrides);

            String vpnData = vpnProperties.entrySet().stream()
                    .map(e -> "%s = %s".formatted(e.getKey(), e.getValue()))
                    .collect(Collectors.joining(", "));

            command.add("vpn.data");
            command.add(vpnData);

            Shell.user(command);
        }
    }


    /**
     * Create a new Cisco AnyConnect / OpenConnect VPN connection.
     *
     * @param connectionName The name for the VPN connection
     * @param gateway        The VPN gateway URL (e.g., "vpn.company.com")
     */
    public static void addOpenConnectConnection(String connectionName, String gateway) {
        addOpenConnectConnection(connectionName, gateway, Map.of());
    }


    private static boolean hasConnection(String name) {
        return Arrays.stream(Shell.user("nmcli", "-t", "connection").split("\n"))
                .filter(r -> r.endsWith(":vpn:"))
                .anyMatch(r -> r.startsWith(name + ':'));
    }

}
