/*
 * Decompiled with CFR 0.152.
 */
package io.dimeformat.crypto;

import io.dimeformat.Key;
import io.dimeformat.Utility;
import io.dimeformat.crypto.ICryptoSuite;
import io.dimeformat.crypto.StandardSuite;
import io.dimeformat.enums.Claim;
import io.dimeformat.enums.KeyCapability;
import io.dimeformat.exceptions.CryptographyException;
import java.nio.charset.StandardCharsets;
import java.util.HashMap;
import java.util.List;
import java.util.Set;

public final class Crypto {
    private HashMap<String, ICryptoSuite> _suiteMap;
    private String _defaultSuiteName;

    public Crypto() {
        this.registerCryptoSuite(new StandardSuite("DSC"));
        this.registerCryptoSuite(new StandardSuite("STN"));
        this._defaultSuiteName = "DSC";
    }

    public synchronized void setDefaultSuiteName(String name) {
        if (this._suiteMap == null) {
            throw new IllegalStateException("Unable to set default cryptographic suite name, no suites registered.");
        }
        if (!this._suiteMap.containsKey(name)) {
            throw new IllegalArgumentException("No cryptographic suite registered for name: " + name);
        }
        this._defaultSuiteName = name;
    }

    public synchronized String getDefaultSuiteName() {
        return this._defaultSuiteName;
    }

    public String generateKeyName(Key key) {
        if (key == null) {
            throw new IllegalArgumentException("Unable to generate key identifier, key must not be null.");
        }
        ICryptoSuite impl = this.getCryptoSuite(key.getCryptoSuiteName());
        byte[] id = impl.generateKeyName(new byte[][]{key.getKeyBytes(Claim.KEY), key.getKeyBytes(Claim.PUB)});
        if (id != null) {
            return Utility.toHex(id);
        }
        return null;
    }

    public byte[] generateSignature(String data, Key key) throws CryptographyException {
        if (data == null || data.length() == 0) {
            throw new IllegalArgumentException("Unable to sign, data must not be null or of length zero.");
        }
        if (key == null || key.getSecret() == null) {
            throw new IllegalArgumentException("Unable to sign, secret key in key must not be null.");
        }
        if (!key.hasCapability(KeyCapability.SIGN)) {
            throw new IllegalArgumentException("Provided key does not specify SIGN usage.");
        }
        ICryptoSuite impl = this.getCryptoSuite(key.getCryptoSuiteName());
        return impl.generateSignature(data.getBytes(StandardCharsets.UTF_8), key.getKeyBytes(Claim.KEY));
    }

    public boolean verifySignature(String data, byte[] signature, Key key) throws CryptographyException {
        if (key == null) {
            throw new IllegalArgumentException("Unable to verify signature, key must not be null.");
        }
        if (data == null || data.length() == 0) {
            throw new IllegalArgumentException("Data must not be null, or of length zero.");
        }
        if (signature == null || signature.length == 0) {
            throw new IllegalArgumentException("Signature must not be null, or of length zero.");
        }
        if (key.getPublic() == null) {
            throw new IllegalArgumentException("Unable to verify, public key in key must not be null.");
        }
        if (!key.hasCapability(KeyCapability.SIGN)) {
            throw new IllegalArgumentException("Provided key does not specify SIGN usage.");
        }
        ICryptoSuite impl = this.getCryptoSuite(key.getCryptoSuiteName());
        return impl.verifySignature(data.getBytes(StandardCharsets.UTF_8), signature, key.getKeyBytes(Claim.PUB));
    }

    public byte[][] generateKey(List<KeyCapability> capabilities) throws CryptographyException {
        return this.generateKey(capabilities, this.getDefaultSuiteName());
    }

    public byte[][] generateKey(List<KeyCapability> capabilities, String suiteName) throws CryptographyException {
        if (capabilities == null || capabilities.size() == 0) {
            throw new CryptographyException("Key usage must not be null or empty.");
        }
        ICryptoSuite impl = this.getCryptoSuite(suiteName);
        return impl.generateKey(capabilities);
    }

    public byte[] generateSharedSecret(Key clientKey, Key serverKey, List<KeyCapability> capabilities) throws CryptographyException {
        if (!clientKey.hasCapability(KeyCapability.EXCHANGE) || !serverKey.hasCapability(KeyCapability.EXCHANGE)) {
            throw new IllegalArgumentException("Provided keys do not specify EXCHANGE usage.");
        }
        if (!clientKey.getCryptoSuiteName().equals(serverKey.getCryptoSuiteName())) {
            throw new IllegalArgumentException("Client key and server key are not generated using the same cryptographic suite");
        }
        ICryptoSuite impl = this.getCryptoSuite(clientKey.getCryptoSuiteName());
        byte[][] rawClientKeys = new byte[][]{clientKey.getKeyBytes(Claim.KEY), clientKey.getKeyBytes(Claim.PUB)};
        byte[][] rawServerKeys = new byte[][]{serverKey.getKeyBytes(Claim.KEY), serverKey.getKeyBytes(Claim.PUB)};
        return impl.generateSharedSecret(rawClientKeys, rawServerKeys, capabilities);
    }

    public byte[] encrypt(byte[] plainText, Key key) throws CryptographyException {
        if (plainText == null || plainText.length == 0) {
            throw new IllegalArgumentException("Plain text to encrypt must not be null and not have a length of 0.");
        }
        if (key == null) {
            throw new IllegalArgumentException("Key must not be null.");
        }
        if (!key.hasCapability(KeyCapability.ENCRYPT)) {
            throw new CryptographyException("Provided key does not specify ENCRYPT usage.");
        }
        ICryptoSuite impl = this.getCryptoSuite(key.getCryptoSuiteName());
        return impl.encrypt(plainText, key.getKeyBytes(Claim.KEY));
    }

    public byte[] decrypt(byte[] cipherText, Key key) throws CryptographyException {
        if (cipherText == null || cipherText.length == 0) {
            throw new IllegalArgumentException("Cipher text to decrypt must not be null and not have a length of 0.");
        }
        if (key == null) {
            throw new IllegalArgumentException("Key must not be null.");
        }
        if (!key.hasCapability(KeyCapability.ENCRYPT)) {
            throw new CryptographyException("Provided key does not specify ENCRYPT usage.");
        }
        ICryptoSuite impl = this.getCryptoSuite(key.getCryptoSuiteName());
        return impl.decrypt(cipherText, key.getKeyBytes(Claim.KEY));
    }

    public String generateHash(byte[] data) throws CryptographyException {
        return this.generateHash(data, this.getDefaultSuiteName());
    }

    public String generateHash(byte[] data, String suiteName) throws CryptographyException {
        ICryptoSuite crypto = this.getCryptoSuite(suiteName);
        return crypto.generateHash(data);
    }

    public String encodeKey(byte[] key, String suiteName) {
        ICryptoSuite crypto = this.getCryptoSuite(suiteName);
        return crypto.encodeKey(key);
    }

    public byte[] decodeKey(String encodedKey, String suiteName) {
        ICryptoSuite crypto = this.getCryptoSuite(suiteName);
        return crypto.decodeKey(encodedKey);
    }

    public void registerCryptoSuite(ICryptoSuite impl) {
        if (impl == null) {
            throw new IllegalArgumentException("Instance of ICrypto implementation must not be null.");
        }
        if (this._suiteMap == null) {
            this._suiteMap = new HashMap();
        } else if (this._suiteMap.containsKey(impl.getName())) {
            throw new IllegalArgumentException("Cryptographic suite already exists with name: " + impl.getName());
        }
        this._suiteMap.put(impl.getName(), impl);
    }

    public boolean hasCryptoSuite(String name) {
        if (this._suiteMap == null) {
            return false;
        }
        return this._suiteMap.containsKey(name);
    }

    public Set<String> allCryptoSuites() {
        if (this._suiteMap == null) {
            return null;
        }
        return this._suiteMap.keySet();
    }

    private ICryptoSuite getCryptoSuite(String name) {
        if (this._suiteMap == null || this._suiteMap.isEmpty()) {
            throw new IllegalStateException("Unable to perform cryptographic operation, no suites registered.");
        }
        ICryptoSuite impl = this._suiteMap.get(name);
        if (impl == null) {
            throw new IllegalArgumentException("Unable to find cryptographic suite with name: " + name);
        }
        return impl;
    }
}

