/*
 * Decompiled with CFR 0.152.
 */
package io.dimeformat.crypto;

import io.dimeformat.Item;
import io.dimeformat.Key;
import io.dimeformat.Signature;
import io.dimeformat.crypto.ICryptoSuite;
import io.dimeformat.crypto.LegacySuite;
import io.dimeformat.crypto.NaClSuite;
import io.dimeformat.enums.Claim;
import io.dimeformat.enums.KeyCapability;
import io.dimeformat.exceptions.CryptographyException;
import java.util.HashMap;
import java.util.List;
import java.util.Set;

public final class Crypto {
    private HashMap<String, ICryptoSuite> _suiteMap;
    private String _defaultSuiteName;

    public Crypto() {
        this.registerCryptoSuite(new NaClSuite("NaCl"));
        this.registerCryptoSuite(new LegacySuite("DSC"));
        this.registerCryptoSuite(new LegacySuite("STN"));
        this._defaultSuiteName = "NaCl";
    }

    public synchronized void setDefaultSuiteName(String name) {
        if (this._suiteMap == null) {
            throw new IllegalStateException("Unable to set default cryptographic suite name, no suites registered.");
        }
        if (!this._suiteMap.containsKey(name)) {
            throw new IllegalArgumentException("No cryptographic suite registered for name: " + name);
        }
        this._defaultSuiteName = name;
    }

    public synchronized String getDefaultSuiteName() {
        return this._defaultSuiteName;
    }

    public String generateKeyName(Key key) {
        if (key == null) {
            throw new IllegalArgumentException("Unable to generate key identifier, key must not be null.");
        }
        ICryptoSuite impl = this.getCryptoSuite(key.getCryptoSuiteName());
        return impl.generateKeyName(key);
    }

    public Signature generateSignature(Item item, Key key) throws CryptographyException {
        if (item == null) {
            throw new IllegalArgumentException("Unable to generate signature, item to sign must not be null.");
        }
        if (key == null || key.getSecret() == null) {
            throw new IllegalArgumentException("Unable to generate signature, key or secret key must not be null.");
        }
        if (!key.hasCapability(KeyCapability.SIGN)) {
            throw new IllegalArgumentException("Unable to generate signature, provided key does not specify SIGN usage.");
        }
        ICryptoSuite impl = this.getCryptoSuite(key.getCryptoSuiteName());
        byte[] bytes = impl.generateSignature(item, key);
        String name = item.isLegacy() ? null : this.generateKeyName(key);
        return new Signature(bytes, name);
    }

    public boolean verifySignature(Item item, Signature signature, Key key) throws CryptographyException {
        if (item == null) {
            throw new IllegalArgumentException("Unable to verify signature, item to sign must not be null.");
        }
        if (signature == null) {
            throw new IllegalArgumentException("Unable to verify signature, item to sign must not be null.");
        }
        if (key == null || key.getPublic() == null) {
            throw new IllegalArgumentException("Unable to verify signature, key or public key must not be null.");
        }
        if (!key.hasCapability(KeyCapability.SIGN)) {
            throw new IllegalArgumentException("Unable to verify signature, provided key does not specify SIGN usage.");
        }
        ICryptoSuite impl = this.getCryptoSuite(key.getCryptoSuiteName());
        return impl.verifySignature(item, signature.getBytes(), key);
    }

    public Key generateKey(List<KeyCapability> capabilities) throws CryptographyException {
        return this.generateKey(capabilities, this.getDefaultSuiteName());
    }

    public Key generateKey(List<KeyCapability> capabilities, String suiteName) throws CryptographyException {
        if (capabilities == null || capabilities.size() == 0) {
            throw new CryptographyException("Key usage must not be null or empty.");
        }
        ICryptoSuite impl = this.getCryptoSuite(suiteName);
        return impl.generateKey(capabilities);
    }

    public Key generateSharedSecret(Key clientKey, Key serverKey, List<KeyCapability> capabilities) throws CryptographyException {
        if (!clientKey.hasCapability(KeyCapability.EXCHANGE) || !serverKey.hasCapability(KeyCapability.EXCHANGE)) {
            throw new IllegalArgumentException("Provided keys do not specify EXCHANGE usage.");
        }
        if (!clientKey.getCryptoSuiteName().equals(serverKey.getCryptoSuiteName())) {
            throw new IllegalArgumentException("Client key and server key are not generated using the same cryptographic suite");
        }
        ICryptoSuite impl = this.getCryptoSuite(clientKey.getCryptoSuiteName());
        return impl.generateSharedSecret(clientKey, serverKey, capabilities);
    }

    public byte[] encrypt(byte[] plainText, Key key) throws CryptographyException {
        if (plainText == null || plainText.length == 0) {
            throw new IllegalArgumentException("Plain text to encrypt must not be null and not have a length of 0.");
        }
        if (key == null) {
            throw new IllegalArgumentException("Key must not be null.");
        }
        if (!key.hasCapability(KeyCapability.ENCRYPT)) {
            throw new CryptographyException("Provided key does not specify ENCRYPT usage.");
        }
        ICryptoSuite impl = this.getCryptoSuite(key.getCryptoSuiteName());
        return impl.encrypt(plainText, key);
    }

    public byte[] decrypt(byte[] cipherText, Key key) throws CryptographyException {
        if (cipherText == null || cipherText.length == 0) {
            throw new IllegalArgumentException("Cipher text to decrypt must not be null and not have a length of 0.");
        }
        if (key == null) {
            throw new IllegalArgumentException("Key must not be null.");
        }
        if (!key.hasCapability(KeyCapability.ENCRYPT)) {
            throw new CryptographyException("Provided key does not specify ENCRYPT usage.");
        }
        ICryptoSuite impl = this.getCryptoSuite(key.getCryptoSuiteName());
        return impl.decrypt(cipherText, key);
    }

    public String generateHash(byte[] data) throws CryptographyException {
        return this.generateHash(data, this.getDefaultSuiteName());
    }

    public String generateHash(byte[] data, String suiteName) throws CryptographyException {
        ICryptoSuite crypto = this.getCryptoSuite(suiteName);
        return crypto.generateHash(data);
    }

    public String encodeKeyBytes(byte[] rawKey, Claim claim, String suiteName) {
        ICryptoSuite crypto = this.getCryptoSuite(suiteName);
        return crypto.encodeKeyBytes(rawKey, claim);
    }

    public byte[] decodeKeyBytes(String encodedKey, Claim claim, String suiteName) {
        ICryptoSuite crypto = this.getCryptoSuite(suiteName);
        return crypto.decodeKeyBytes(encodedKey, claim);
    }

    public void registerCryptoSuite(ICryptoSuite impl) {
        if (impl == null) {
            throw new IllegalArgumentException("Instance of ICrypto implementation must not be null.");
        }
        if (this._suiteMap == null) {
            this._suiteMap = new HashMap();
        } else if (this._suiteMap.containsKey(impl.getName())) {
            throw new IllegalArgumentException("Cryptographic suite already exists with name: " + impl.getName());
        }
        this._suiteMap.put(impl.getName(), impl);
    }

    public boolean hasCryptoSuite(String name) {
        if (this._suiteMap == null) {
            return false;
        }
        return this._suiteMap.containsKey(name);
    }

    public Set<String> allCryptoSuites() {
        if (this._suiteMap == null) {
            return null;
        }
        return this._suiteMap.keySet();
    }

    private ICryptoSuite getCryptoSuite(String name) {
        if (this._suiteMap == null || this._suiteMap.isEmpty()) {
            throw new IllegalStateException("Unable to perform cryptographic operation, no suites registered.");
        }
        ICryptoSuite impl = this._suiteMap.get(name);
        if (impl == null) {
            throw new IllegalArgumentException("Unable to find cryptographic suite with name: " + name);
        }
        return impl;
    }
}

