/*
 * Decompiled with CFR 0.152.
 */
package io.dimeformat.crypto;

import com.goterl.lazysodium.SodiumJava;
import io.dimeformat.Item;
import io.dimeformat.Key;
import io.dimeformat.Utility;
import io.dimeformat.crypto.ICryptoSuite;
import io.dimeformat.enums.Claim;
import io.dimeformat.enums.KeyCapability;
import io.dimeformat.exceptions.CryptographyException;
import java.nio.charset.StandardCharsets;
import java.util.List;

class NaClSuite
implements ICryptoSuite {
    static final String SUITE_NAME = "NaCl";
    protected static final int NBR_SIGNATURE_BYTES = 64;
    protected static final int NBR_A_KEY_BYTES = 32;
    protected static final int NBR_S_KEY_BYTES = 32;
    protected static final int NBR_X_KEY_BYTES = 32;
    protected static final int NBR_NONCE_BYTES = 24;
    protected static final int NBR_MAC_BYTES = 16;
    protected static final int NBR_HASH_BYTES = 32;
    protected final SodiumJava _sodium = new SodiumJava();
    protected final String _suiteName;

    @Override
    public String getName() {
        return this._suiteName;
    }

    public NaClSuite(String name) {
        this._suiteName = name;
    }

    @Override
    public String generateKeyName(Key key) {
        byte[] bytes = key.getKeyBytes(Claim.PUB);
        if (bytes != null && bytes.length > 0) {
            try {
                byte[] hash = this.hash(bytes);
                byte[] name = Utility.subArray(hash, 0, 8);
                return Utility.toHex(name);
            }
            catch (CryptographyException cryptographyException) {
                // empty catch block
            }
        }
        return null;
    }

    @Override
    public byte[] generateSignature(Item item, Key key) throws CryptographyException {
        this.generateHash(item.rawEncoded(false));
        String thumbprint = item.generateThumbprint(false, this._suiteName);
        if (thumbprint != null && !thumbprint.isEmpty()) {
            byte[] signature = new byte[64];
            byte[] data = thumbprint.getBytes(StandardCharsets.UTF_8);
            int result = this._sodium.crypto_sign_detached(signature, null, data, (long)data.length, key.getKeyBytes(Claim.KEY));
            if (result != 0) {
                throw new CryptographyException("Failed to generate signature, error code returned: " + result);
            }
            return signature;
        }
        throw new IllegalArgumentException("Failed to generate signature, item thumbprint was null or empty.");
    }

    @Override
    public boolean verifySignature(Item item, byte[] signature, Key key) throws CryptographyException {
        String thumbprint = item.generateThumbprint(false, this._suiteName);
        if (thumbprint != null && !thumbprint.isEmpty()) {
            byte[] data = thumbprint.getBytes(StandardCharsets.UTF_8);
            return this._sodium.crypto_sign_verify_detached(signature, data, (long)data.length, key.getKeyBytes(Claim.PUB)) == 0;
        }
        throw new IllegalArgumentException("Failed to generate signature, item thumbprint was null or empty.");
    }

    @Override
    public Key generateKey(List<KeyCapability> capabilities) throws CryptographyException {
        byte[] secretKey;
        if (capabilities == null || capabilities.size() != 1) {
            throw new IllegalArgumentException("Unable to generate, invalid key capabilities requested.");
        }
        KeyCapability firstUse = capabilities.get(0);
        if (firstUse == KeyCapability.ENCRYPT) {
            byte[] secretKey2 = new byte[32];
            this._sodium.crypto_secretbox_keygen(secretKey2);
            return new Key(capabilities, secretKey2, null, this._suiteName);
        }
        byte[] publicKey = new byte[32];
        switch (capabilities.get(0)) {
            case SIGN: {
                secretKey = new byte[64];
                this._sodium.crypto_sign_keypair(publicKey, secretKey);
                break;
            }
            case EXCHANGE: {
                secretKey = new byte[32];
                this._sodium.crypto_kx_keypair(publicKey, secretKey);
                break;
            }
            default: {
                throw new CryptographyException("Unable to generate keypair for key type " + capabilities + ".");
            }
        }
        return new Key(capabilities, secretKey, publicKey, this._suiteName);
    }

    @Override
    public Key generateSharedSecret(Key clientKey, Key serverKey, List<KeyCapability> capabilities) throws CryptographyException {
        if (!capabilities.contains((Object)KeyCapability.ENCRYPT)) {
            throw new IllegalArgumentException("Unable to generate, key capability for shared secret must be ENCRYPT.");
        }
        if (capabilities.size() > 1) {
            throw new IllegalArgumentException("Unable to generate, key capability for shared secret may only be ENCRYPT.");
        }
        byte[][] rawClientKeys = new byte[][]{clientKey.getKeyBytes(Claim.KEY), clientKey.getKeyBytes(Claim.PUB)};
        byte[][] rawServerKeys = new byte[][]{serverKey.getKeyBytes(Claim.KEY), serverKey.getKeyBytes(Claim.PUB)};
        byte[] shared = new byte[32];
        if (rawClientKeys[0] != null && rawClientKeys.length == 2) {
            byte[] secret = Utility.combine(rawClientKeys[0], rawClientKeys[1]);
            if (this._sodium.crypto_kx_client_session_keys(shared, null, rawClientKeys[1], secret, rawServerKeys[1]) != 0) {
                throw new CryptographyException("Unable to generate, cryptographic operation failed.");
            }
        } else if (rawServerKeys[0] != null && rawServerKeys.length == 2) {
            if (this._sodium.crypto_kx_server_session_keys(null, shared, rawServerKeys[1], rawServerKeys[0], rawClientKeys[1]) != 0) {
                throw new CryptographyException("Unable to generate, cryptographic operation failed.");
            }
        } else {
            throw new CryptographyException("Unable to generate, invalid keys provided.");
        }
        return new Key(capabilities, shared, null, this._suiteName);
    }

    @Override
    public byte[] encrypt(byte[] data, Key key) throws CryptographyException {
        byte[] nonce = Utility.randomBytes(24);
        if (nonce.length > 0) {
            byte[] cipherText = new byte[16 + data.length];
            if (this._sodium.crypto_secretbox_easy(cipherText, data, (long)data.length, nonce, key.getKeyBytes(Claim.KEY)) != 0) {
                throw new CryptographyException("Cryptographic operation failed.");
            }
            return Utility.combine(nonce, cipherText);
        }
        throw new CryptographyException("Unable to generate sufficient nonce.");
    }

    @Override
    public byte[] decrypt(byte[] data, Key key) throws CryptographyException {
        byte[] nonce = Utility.subArray(data, 0, 24);
        byte[] bytes = Utility.subArray(data, 24);
        byte[] plain = new byte[bytes.length - 16];
        int result = this._sodium.crypto_secretbox_open_easy(plain, bytes, (long)bytes.length, nonce, key.getKeyBytes(Claim.KEY));
        if (result != 0) {
            throw new CryptographyException("Cryptographic operation failed (" + result + ").");
        }
        return plain;
    }

    @Override
    public String generateHash(byte[] data) throws CryptographyException {
        return Utility.toHex(this.hash(data));
    }

    @Override
    public String encodeKeyBytes(byte[] rawKey, Claim claim) {
        return Utility.toBase64(rawKey);
    }

    @Override
    public byte[] decodeKeyBytes(String encodedKey, Claim claim) {
        return Utility.fromBase64(encodedKey);
    }

    private byte[] hash(byte[] data) throws CryptographyException {
        byte[] hash = new byte[32];
        if (this._sodium.crypto_generichash(hash, hash.length, data, (long)data.length, null, 0) != 0) {
            throw new CryptographyException("Cryptographic operation failed.");
        }
        return hash;
    }
}

