/*
 * Decompiled with CFR 0.152.
 */
package io.featureflow.client.core;

import com.google.gson.Gson;
import com.google.gson.reflect.TypeToken;
import io.featureflow.client.FeatureControlCallbackHandler;
import io.featureflow.client.FeatureflowConfig;
import io.featureflow.client.core.CallbackEvent;
import io.featureflow.client.core.FeatureControlCache;
import io.featureflow.client.model.FeatureControl;
import java.io.Closeable;
import java.io.IOException;
import java.io.InterruptedIOException;
import java.lang.reflect.Type;
import java.net.URI;
import java.net.UnknownHostException;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import javax.net.ssl.SSLException;
import org.apache.hc.client5.http.ConnectTimeoutException;
import org.apache.hc.client5.http.HttpRequestRetryStrategy;
import org.apache.hc.client5.http.classic.methods.HttpGet;
import org.apache.hc.client5.http.config.RequestConfig;
import org.apache.hc.client5.http.impl.cache.CacheConfig;
import org.apache.hc.client5.http.impl.cache.CachingHttpClients;
import org.apache.hc.client5.http.impl.classic.CloseableHttpClient;
import org.apache.hc.client5.http.impl.classic.CloseableHttpResponse;
import org.apache.hc.client5.http.impl.io.PoolingHttpClientConnectionManager;
import org.apache.hc.client5.http.io.HttpClientConnectionManager;
import org.apache.hc.client5.http.protocol.HttpClientContext;
import org.apache.hc.core5.http.ClassicHttpRequest;
import org.apache.hc.core5.http.Header;
import org.apache.hc.core5.http.HttpRequest;
import org.apache.hc.core5.http.HttpResponse;
import org.apache.hc.core5.http.message.BasicClassicHttpRequest;
import org.apache.hc.core5.http.protocol.HttpContext;
import org.apache.hc.core5.util.TimeValue;
import org.apache.hc.core5.util.Timeout;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FeatureflowPollingClient
implements Closeable {
    private static final Logger logger = LoggerFactory.getLogger(FeatureflowPollingClient.class);
    private static final int DEFAULT_INTERVAL = 60000;
    private static final String CLIENT_VERSION = "JavaClient/1.2.0";
    private final String apiKey;
    private final FeatureflowConfig config;
    private final FeatureControlCache repository;
    private final Map<CallbackEvent, List<FeatureControlCallbackHandler>> callbacks;
    private final CloseableHttpClient httpClient;
    private final ScheduledExecutorService scheduler;
    private final Gson gson;
    private final AtomicBoolean initialized = new AtomicBoolean(false);
    private final AtomicBoolean closed = new AtomicBoolean(false);
    private String etag = "";
    private final int pollingInterval;
    private ScheduledFuture<?> pollingTask;
    private final Type mapOfFeatureControlsType = new TypeToken<Map<String, FeatureControl>>(){}.getType();

    public FeatureflowPollingClient(String apiKey, FeatureflowConfig config, FeatureControlCache repository, Map<CallbackEvent, List<FeatureControlCallbackHandler>> callbacks) {
        this.apiKey = apiKey;
        this.config = config;
        this.repository = repository;
        this.callbacks = callbacks;
        this.httpClient = this.createHttpClient();
        this.scheduler = Executors.newSingleThreadScheduledExecutor(r -> {
            Thread t = new Thread(r, "featureflow-polling-client");
            t.setDaemon(true);
            return t;
        });
        this.gson = new Gson();
        this.pollingInterval = this.getPollingInterval(config);
        if (this.isValidApiKey(apiKey)) {
            this.startPolling();
        } else {
            logger.warn("API key is missing or too short. Features will not be fetched.");
        }
    }

    private void startPolling() {
        this.getFeatures();
        if (this.pollingInterval > 0) {
            this.pollingTask = this.scheduler.scheduleWithFixedDelay(this::getFeatures, this.pollingInterval, this.pollingInterval, TimeUnit.MILLISECONDS);
            logger.info("Started polling for feature updates every {} ms", (Object)this.pollingInterval);
        } else {
            logger.info("Polling interval set to 0. Featureflow will NOT poll for feature changes.");
        }
    }

    private void getFeatures() {
        if (this.closed.get()) {
            return;
        }
        try {
            URI uri = URI.create(this.getPollingUri());
            HttpGet request = this.createGetRequest(uri);
            try (CloseableHttpResponse response = this.httpClient.execute((ClassicHttpRequest)request);){
                int statusCode = response.getCode();
                if (statusCode == 200) {
                    String responseBody;
                    Map controls;
                    Header etagHeader = response.getFirstHeader("ETag");
                    if (etagHeader != null) {
                        this.etag = etagHeader.getValue();
                    }
                    if ((controls = (Map)this.gson.fromJson(responseBody = new String(response.getEntity().getContent().readAllBytes()), this.mapOfFeatureControlsType)) != null) {
                        logger.debug("Updating features from polling response");
                        this.updateFeatures(controls);
                    }
                    if (!this.initialized.getAndSet(true)) {
                        logger.info("Featureflow polling client initialized.");
                    }
                } else if (statusCode == 304) {
                    logger.debug("No feature changes detected (304 Not Modified)");
                } else if (statusCode >= 400) {
                    logger.warn("Request for features failed with response status {}", (Object)statusCode);
                }
            }
            catch (Exception e) {
                logger.error("Error fetching features", (Throwable)e);
            }
        }
        catch (Exception e) {
            logger.error("Failed to fetch features", (Throwable)e);
        }
    }

    private void updateFeatures(Map<String, FeatureControl> controls) {
        for (Map.Entry<String, FeatureControl> entry : controls.entrySet()) {
            String featureKey = entry.getKey();
            FeatureControl featureControl = entry.getValue();
            if (logger.isDebugEnabled()) {
                logger.debug("Updating feature {} enabled: {}", (Object)featureKey, (Object)featureControl.enabled);
            }
            this.repository.update(featureKey, featureControl);
            if (this.callbacks == null || this.callbacks.get((Object)CallbackEvent.UPDATED_FEATURE) == null) continue;
            for (FeatureControlCallbackHandler callback : this.callbacks.get((Object)CallbackEvent.UPDATED_FEATURE)) {
                try {
                    callback.onUpdate(featureControl);
                }
                catch (Exception e) {
                    logger.error("Error in feature update callback", (Throwable)e);
                }
            }
        }
    }

    private HttpGet createGetRequest(URI uri) {
        HttpGet request = new HttpGet(uri);
        request.setHeader("Authorization", (Object)("Bearer " + this.apiKey));
        request.setHeader("X-Featureflow-Client", (Object)CLIENT_VERSION);
        request.setHeader("Accept", (Object)"application/json");
        if (!this.etag.isEmpty()) {
            request.setHeader("If-None-Match", (Object)this.etag);
        }
        return request;
    }

    private CloseableHttpClient createHttpClient() {
        PoolingHttpClientConnectionManager manager = new PoolingHttpClientConnectionManager();
        manager.setMaxTotal(100);
        manager.setDefaultMaxPerRoute(20);
        CacheConfig cacheConfig = CacheConfig.custom().setMaxCacheEntries(1000).setMaxObjectSize(131072L).setSharedCache(false).build();
        HttpRequestRetryStrategy serviceUnavailableRetryStrategy = new HttpRequestRetryStrategy(){

            public boolean retryRequest(HttpRequest request, IOException exception, int executionCount, HttpContext context) {
                return false;
            }

            public boolean retryRequest(HttpResponse response, int executionCount, HttpContext context) {
                int statusCode = response.getCode();
                return statusCode == 502 && executionCount < 1000;
            }

            public TimeValue getRetryInterval(HttpRequest request, IOException exception, int executionCount, HttpContext context) {
                return TimeValue.of((long)5L, (TimeUnit)TimeUnit.SECONDS);
            }

            public TimeValue getRetryInterval(HttpResponse response, int executionCount, HttpContext context) {
                return TimeValue.of((long)5L, (TimeUnit)TimeUnit.SECONDS);
            }
        };
        HttpRequestRetryStrategy myRetryHandler = new HttpRequestRetryStrategy(){

            public boolean retryRequest(HttpRequest request, IOException exception, int executionCount, HttpContext context) {
                if (executionCount >= 5) {
                    return false;
                }
                if (exception instanceof ConnectTimeoutException) {
                    return false;
                }
                if (exception instanceof InterruptedIOException) {
                    return false;
                }
                if (exception instanceof UnknownHostException) {
                    return false;
                }
                if (exception instanceof SSLException) {
                    return false;
                }
                HttpClientContext clientContext = HttpClientContext.adapt((HttpContext)context);
                HttpRequest originalRequest = clientContext.getRequest();
                boolean idempotent = !(originalRequest instanceof BasicClassicHttpRequest);
                return idempotent;
            }

            public boolean retryRequest(HttpResponse response, int executionCount, HttpContext context) {
                return false;
            }

            public TimeValue getRetryInterval(HttpRequest request, IOException exception, int executionCount, HttpContext context) {
                return TimeValue.of((long)1L, (TimeUnit)TimeUnit.SECONDS);
            }

            public TimeValue getRetryInterval(HttpResponse response, int executionCount, HttpContext context) {
                return TimeValue.of((long)1L, (TimeUnit)TimeUnit.SECONDS);
            }
        };
        RequestConfig requestConfig = RequestConfig.custom().setConnectionRequestTimeout(Timeout.ofMilliseconds((long)this.config.getConnectTimeout())).setResponseTimeout(Timeout.ofMilliseconds((long)this.config.getSocketTimeout())).setProxy(this.config.getHttpProxyHost()).build();
        return CachingHttpClients.custom().setCacheConfig(cacheConfig).setConnectionManager((HttpClientConnectionManager)manager).setDefaultRequestConfig(requestConfig).setRetryStrategy(myRetryHandler).setRetryStrategy(serviceUnavailableRetryStrategy).build();
    }

    private int getPollingInterval(FeatureflowConfig config) {
        int interval = config.getPollingInterval();
        if (interval > 0 && interval < 20000) {
            logger.warn("Polling interval {}ms is too short, using minimum of 20 seconds", (Object)interval);
            return 20000;
        }
        return interval > 0 ? interval : 60000;
    }

    private boolean isValidApiKey(String apiKey) {
        return apiKey != null && apiKey.length() > 10;
    }

    private String getPollingUri() {
        return this.config.getPollingUri();
    }

    public boolean initialized() {
        return this.initialized.get();
    }

    @Override
    public void close() throws IOException {
        if (this.closed.getAndSet(true)) {
            return;
        }
        logger.info("Closing Featureflow polling client");
        if (this.pollingTask != null) {
            this.pollingTask.cancel(true);
        }
        this.scheduler.shutdown();
        try {
            if (!this.scheduler.awaitTermination(5L, TimeUnit.SECONDS)) {
                this.scheduler.shutdownNow();
            }
        }
        catch (InterruptedException e) {
            this.scheduler.shutdownNow();
            Thread.currentThread().interrupt();
        }
        if (this.httpClient != null) {
            this.httpClient.close();
        }
        if (this.repository != null) {
            this.repository.close();
        }
    }
}

