package io.github.atkawa7.codegen.annotations.processing;
import io.github.atkawa7.codegen.annotations.DomainModel;
import io.github.atkawa7.codegen.annotations.Queryable;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.MutablePair;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.commons.text.StringSubstitutor;

import javax.annotation.processing.AbstractProcessor;
import javax.annotation.processing.Messager;
import javax.annotation.processing.ProcessingEnvironment;
import javax.annotation.processing.RoundEnvironment;
import javax.lang.model.SourceVersion;
import javax.lang.model.element.*;
import javax.lang.model.type.DeclaredType;
import javax.lang.model.util.ElementFilter;
import javax.tools.Diagnostic;
import javax.tools.JavaFileObject;
import java.io.IOException;
import java.io.Writer;
import java.util.*;

import static java.lang.String.format;
import static java.util.Collections.singleton;
import static javax.lang.model.SourceVersion.latestSupported;
import static javax.lang.model.element.ElementKind.FIELD;

public class RepositoryCodegen extends AbstractProcessor {
    private Messager messager;
    private Map<String,String>  options;

    @Override
    public synchronized void init(ProcessingEnvironment processingEnv) {
        super.init(processingEnv);
        this.messager = processingEnv.getMessager();
        this.options  = processingEnv.getOptions();
    }

    @Override
    public SourceVersion getSupportedSourceVersion() {
        return latestSupported();
    }

    @Override
    public Set<String> getSupportedAnnotationTypes() {
        return singleton(DomainModel.class.getName());
    }

    @Override
    public boolean process(Set<? extends TypeElement> annotations, RoundEnvironment roundEnv) {
        messager.printMessage(Diagnostic.Kind.WARNING, "Processing data codegen");
        Set<? extends Element> annotatedWith = roundEnv.getElementsAnnotatedWith(DomainModel.class);
        Set<TypeElement> elements = ElementFilter.typesIn(annotatedWith);
        messager.printMessage(Diagnostic.Kind.WARNING, format("Size of elements is %d", elements.size()));

        for (TypeElement typeElement : elements) {
            messager.printMessage(Diagnostic.Kind.WARNING, format("Processing data codegen class <%s>", typeElement.getQualifiedName()));
            Map<String, String> context = new HashMap<>(5);

            context.put("qualifiedName", typeElement.getQualifiedName().toString());
            context.put("primaryKeyClass", Long.class.getName());

            List<Pair<List<String>, Map<String, String>>> queriablesContexts = new ArrayList<>();


            for (AnnotationMirror annotationMirror : typeElement.getAnnotationMirrors()) {
                for (Map.Entry<? extends ExecutableElement, ? extends AnnotationValue> entry : annotationMirror.getElementValues().entrySet()) {
                    ExecutableElement key = entry.getKey();
                    Object value = entry.getValue().getValue();

                    if (value instanceof List<?>) {
                        Object first = CollectionUtils.first((List<?>) value);
                        if (first instanceof AnnotationMirror) {

                            List<AnnotationMirror> childAnnotations = (List<AnnotationMirror>) value;
                            for (AnnotationMirror mirror : childAnnotations) {
                                List<String> fieldNames = new ArrayList<>(0);
                                Map<String, String> methodContext = new HashMap<>();

                                for (Map.Entry<? extends ExecutableElement, ? extends AnnotationValue> e : mirror.getElementValues().entrySet()) {
                                    ExecutableElement k = e.getKey();
                                    AnnotationValue v = e.getValue();
                                    messager.printMessage(Diagnostic.Kind.WARNING, format("Child Return Type (%s)", k.getReturnType()));
                                    if (k.getReturnType().toString().equalsIgnoreCase("java.lang.String[]")) {
                                        List<?> values = (List<?>) v.getValue();
                                        if (values != null && values.size() > 1) {
                                            for (Object attr : values) {
                                                messager.printMessage(Diagnostic.Kind.WARNING, format("Attr Key (%s) (%s)", attr, attr.getClass()));
                                                fieldNames.add(((AnnotationValue) attr).getValue().toString());
                                            }
                                        }
                                    } else {
                                        methodContext.put(k.getSimpleName().toString(), v.getValue().toString());
                                    }
                                }
                                queriablesContexts.add(new MutablePair<>(fieldNames, methodContext));
                            }
                        }
                    }


                    messager.printMessage(Diagnostic.Kind.WARNING, format("Class Annotation Key (%s) (%s)", key.getReturnType(), value.getClass().getSimpleName()));
                    context.put(key.getSimpleName().toString(), value.toString());
                }
            }


            Set<String> methods = new HashSet<>();

            Map<String, String> nameTypeCache = new HashMap<>();

            for (Element fieldElement : typeElement.getEnclosedElements()) {
                if (fieldElement.getKind() == FIELD) {
                    Map<String, String> methodContext = new HashMap<>(5);
                    nameTypeCache.put(fieldElement.getSimpleName().toString(), fieldElement.asType().toString());
                    for (AnnotationMirror annotationMirror : fieldElement.getAnnotationMirrors()) {
                        DeclaredType declaredType = annotationMirror.getAnnotationType();
                        if (declaredType.toString().equals(Queryable.class.getName())) {
                            messager.printMessage(Diagnostic.Kind.WARNING, format("Field Annotation(%s)(%s)", annotationMirror.getAnnotationType(), fieldElement.asType()));
                            methodContext.put("qualifiedName", fieldElement.asType().toString());
                            methodContext.put("name", fieldElement.getSimpleName().toString());
                            methodContext.put("capitalized", StringUtils.capitalize(fieldElement.getSimpleName().toString()));
                            methodContext.put("fieldNames", fieldElement.asType().toString() + " " +fieldElement.getSimpleName().toString() );

                            for (Map.Entry<? extends ExecutableElement, ? extends AnnotationValue> entry : annotationMirror.getElementValues().entrySet()) {
                                ExecutableElement key = entry.getKey();
                                Object value = entry.getValue().getValue();
                                methodContext.put(key.getSimpleName().toString(), value.toString());
                            }
                            methods.add( getMethod(typeElement, context, methodContext));
                        }
                    }
                }
            }


            for (Pair<List<String>, Map<String, String>> mapPair : queriablesContexts) {
                Map<String, String> methodContext = mapPair.getValue();
                List<String> fieldNames = mapPair.getKey();
                List<String> capitalized = new ArrayList<>(fieldNames.size());
                List<String> parameters = new ArrayList<>(fieldNames.size());


                for (String fieldName : fieldNames) {
                    String type = nameTypeCache.get(fieldName);
                    if (type == null) {
                        throw new IllegalArgumentException(format("FieldName(%s) doesn't exists on (%s)", fieldName, typeElement.getQualifiedName()));
                    }
                    capitalized.add(StringUtils.capitalize(fieldName));
                    parameters.add(type + "  " + fieldName);
                }

                methodContext.put("capitalized", String.join("And", capitalized));
                methodContext.put("fieldNames", String.join(", ", parameters));
                methods.add(getMethod(typeElement, context, methodContext));
            }


            String name = typeElement.getSimpleName().toString();
            String qualifiedName = typeElement.getQualifiedName().toString();

            String packageName = options.getOrDefault("codegen.packageName", qualifiedName.substring(0, qualifiedName.lastIndexOf(".")) + ".repositories");
            String repositoryName = name + "Repository";

            StringJoiner imports = new StringJoiner(";\nimport ", "import ", ";\n\n");

            imports.add("org.springframework.data.domain.Page");
            imports.add("org.springframework.data.domain.Pageable");
            imports.add("org.springframework.data.jpa.repository.JpaRepository");
            imports.add("org.springframework.data.jpa.repository.JpaSpecificationExecutor");
            imports.add("org.springframework.data.repository.NoRepositoryBean");
            imports.add("org.springframework.stereotype.Repository");

            imports.add("java.util.List");
            imports.add("java.util.Optional");


            StringBuilder source = new StringBuilder();
            source.append("package ");
            source.append(packageName);
            source.append(";\n\n");

            source.append(imports.toString());


            boolean noRepositoryBean = Boolean.parseBoolean(context.getOrDefault("noRepositoryBean", "false"));

            if (noRepositoryBean) {
                source.append("@NoRepositoryBean\n");
            } else {
                source.append("@Repository\n");
            }

            source.append("public interface ");
            source.append(repositoryName);

            source.append(new StringSubstitutor(context).replace(" extends JpaRepository<${qualifiedName}, ${primaryKeyClass}>, JpaSpecificationExecutor<${qualifiedName}>"));


            source.append(" {\n\n");
            source.append(String.join(";\n\n", methods));
            if (methods.size() > 0) {
                source.append(";");
            }
            source.append("\n\n}\n");


            try {
                JavaFileObject javaFileObject = processingEnv.getFiler().createSourceFile(packageName + '.' + repositoryName);
                try (Writer writer = javaFileObject.openWriter()) {
                    writer.write(source.toString());
                }
            } catch (IOException e) {
                e.printStackTrace();
            }
        }
        return false;
    }

    private String getMethod(TypeElement typeElement, Map<String, String> context, Map<String, String> methodContext) {
        boolean many = Boolean.parseBoolean(methodContext.getOrDefault("many", "false"));
        boolean pageable = Boolean.parseBoolean(methodContext.getOrDefault("pageable", "false"));
        boolean optional = Boolean.parseBoolean(methodContext.getOrDefault("optional", "true"));
        messager.printMessage(Diagnostic.Kind.WARNING, "Optional " + optional + methodContext.getOrDefault("optional", "true"));


        String methodTemplate;
        String returnType;

        if (many) {
            if (pageable) {
                returnType = new StringSubstitutor(context).replace("Page<${qualifiedName}> ");
                methodTemplate = new StringSubstitutor(methodContext).replace("findAllBy${capitalized}(${fieldNames}, Pageable pageable)");
            } else {
                returnType = new StringSubstitutor(context).replace("List<${qualifiedName}> ");
                methodTemplate = new StringSubstitutor(methodContext).replace("findAllBy${capitalized}(${fieldNames})");
            }
        } else {
            if (optional) {
                returnType = new StringSubstitutor(context).replace("Optional<${qualifiedName}> ");
            } else {
                returnType = typeElement.getQualifiedName() + " ";
            }
            methodTemplate = new StringSubstitutor(methodContext).replace("findBy${capitalized}(${fieldNames})");
        }


        return "    " + returnType + methodTemplate;
    }




}
