package io.github.atkawa7.codegen.annotations.processing;
import io.github.atkawa7.codegen.annotations.DomainModel;
import io.github.atkawa7.codegen.annotations.Queryable;
import io.github.atkawa7.codegen.annotations.processing.models.DomainModelMeta;
import io.github.atkawa7.codegen.annotations.processing.models.FieldMeta;
import io.github.atkawa7.codegen.annotations.processing.models.QueryableMeta;
import io.github.atkawa7.codegen.annotations.processing.models.QueryablesMeta;
import org.apache.commons.lang3.StringUtils;

import javax.annotation.processing.AbstractProcessor;
import javax.annotation.processing.Messager;
import javax.annotation.processing.ProcessingEnvironment;
import javax.annotation.processing.RoundEnvironment;
import javax.lang.model.SourceVersion;
import javax.lang.model.element.*;
import javax.lang.model.type.DeclaredType;
import javax.lang.model.type.TypeMirror;
import javax.lang.model.util.ElementFilter;
import javax.lang.model.util.Types;
import javax.persistence.Id;
import javax.tools.Diagnostic;
import javax.tools.JavaFileObject;
import java.io.IOException;
import java.io.Writer;
import java.lang.annotation.Annotation;
import java.sql.Time;
import java.sql.Timestamp;
import java.time.*;
import java.util.*;
import java.util.stream.Collectors;

import static io.github.atkawa7.codegen.annotations.processing.CodegenUtils.convertToTypeElement;
import static io.github.atkawa7.codegen.annotations.processing.CodegenUtils.isInternalClass;
import static java.lang.String.format;
import static java.util.Collections.emptyList;
import static java.util.Collections.singleton;
import static javax.lang.model.SourceVersion.latestSupported;
import static javax.lang.model.element.ElementKind.FIELD;


public class RepositoryCodegen extends AbstractProcessor {
    private Messager messager;
    public static Map<String, List<FieldMeta>> cache = new HashMap<>();
    private Map<String, String> options;
    private Types typesUtils;

    public static List<String> getClassNames(Class<?>... args) {
        if (args != null) {
            List<String> names = new ArrayList<>(args.length);
            for (Class<?> classType : args) {
                names.add(classType.getCanonicalName());
            }
            return names;
        }
        return emptyList();
    }

    public static boolean isDate(TypeMirror typeMirror) {
        String fullQualified = typeMirror.toString();
        for (String className : getClassNames(
                Date.class, java.sql.Date.class, Time.class, Timestamp.class,
                LocalDate.class, LocalDateTime.class, ZonedDateTime.class, Instant.class,
                OffsetDateTime.class, OffsetTime.class, Year.class, YearMonth.class,
                ZonedDateTime.class, LocalTime.class
        )) {
            if (className.equalsIgnoreCase(fullQualified)) {
                return true;
            }
        }
        return false;
    }

    public static boolean isString(TypeMirror typeMirror) {
        String fullQualified = typeMirror.toString();
        for (String className : getClassNames(String.class)) {
            if (className.equalsIgnoreCase(fullQualified)) {
                return true;
            }
        }
        return false;
    }

    public static boolean isArrayOfType(TypeMirror typeMirror, Class<?> type) {
        return typeMirror.toString().equalsIgnoreCase(format("%s[]", type.getCanonicalName()));
    }

    public static Map<ExecutableElement, AnnotationValue> getAnnotationValuesWithDefaults(
            AnnotationMirror annotation) {
        Map<ExecutableElement, AnnotationValue> values = new HashMap<>();
        Map<? extends ExecutableElement, ? extends AnnotationValue> declaredValues =
                annotation.getElementValues();
        for (ExecutableElement method :
                ElementFilter.methodsIn(annotation.getAnnotationType().asElement().getEnclosedElements())) {
            // Must iterate and put in this order, to ensure consistency in generated code.

            if (declaredValues.containsKey(method)) {
                values.put(method, declaredValues.get(method));
            } else if (method.getDefaultValue() != null) {
                values.put(method, method.getDefaultValue());
            } else {
                throw new IllegalStateException(
                        "Unset annotation value without default should never happen: ");
            }
        }
        return values;
    }

    @Override
    public synchronized void init(ProcessingEnvironment processingEnv) {
        super.init(processingEnv);
        this.messager = processingEnv.getMessager();
        this.options = processingEnv.getOptions();
        this.typesUtils = processingEnv.getTypeUtils();

    }

    @Override
    public Set<String> getSupportedAnnotationTypes() {
        return singleton(DomainModel.class.getName());
    }

    @Override
    public Set<String> getSupportedOptions() {
        return singleton("codegen.packageName");
    }

    @Override
    public SourceVersion getSupportedSourceVersion() {
        return latestSupported();
    }

    public Set<TypeElement> getTypeElementsAnnotatedWith(Class<? extends Annotation> classType, RoundEnvironment roundEnv) {
        Set<? extends Element> annotatedWith = roundEnv.getElementsAnnotatedWith(classType);
        return ElementFilter.typesIn(annotatedWith);
    }

    public List<FieldMeta> getFieldsForType(TypeElement typeElement) {
        List<FieldMeta> cached = cache.get(typeElement.getQualifiedName().toString());
        if (cached != null) {
            return cached;
        }
        List<FieldMeta> metas = new ArrayList<>(5);
        for (Element element : typeElement.getEnclosedElements()) {
            if (element.getKind() == FIELD) {
                FieldMeta meta = new FieldMeta();
                meta.setName(element.getSimpleName().toString());
                meta.setFullyQualifiedName(element.asType().toString());
                meta.setElement(element);
                metas.add(meta);
            }

        }
        return metas;
    }

    public List<FieldMeta> getAllFieldsForType(TypeElement typeElement) {
        List<FieldMeta> metas = new ArrayList<>(0);
        for (TypeMirror supertype : typesUtils.directSupertypes(typeElement.asType())) {
            metas.addAll(getFieldsForType(convertToTypeElement(supertype)));
        }
        metas.addAll(getFieldsForType(typeElement));
        return metas;
    }

    public Map<String, Object> getValues(AnnotationMirror annotationMirror) {
        Map<ExecutableElement, AnnotationValue> values = getAnnotationValuesWithDefaults(annotationMirror);
        Map<String, Object> out = new HashMap<>(values.size());
        for (Map.Entry<ExecutableElement, AnnotationValue> entry : values.entrySet()) {
            String key = entry.getKey().getSimpleName().toString();
            Object value = entry.getValue().getValue();
            out.put(key, value);
        }
        return out;
    }

    public String json(Map<String, Object> map) {
        return "{" + map.entrySet().stream()
                .map(e -> "\"" + e.getKey() + "\":\"" + e.getValue() + "\"")
                .collect(Collectors.joining(", ")) + "}";
    }

    @Override
    public boolean process(Set<? extends TypeElement> annotations, RoundEnvironment roundEnv) {
        Set<TypeElement> elements = getTypeElementsAnnotatedWith(DomainModel.class, roundEnv);
        if (elements.isEmpty()) {
            return false;
        }

        StringBuilder builder = new StringBuilder();
        builder.append("---------------------Processing----------------\n\n");
        builder.append(format("Codegen processing started %s\n", new Date()));
        builder.append(format("Processing %d classes \n", elements.size()));

        int classesProcessed = 0;
        List<DomainModelMeta> classMetaList = new ArrayList<>();

        for (TypeElement typeElement : elements) {
            classesProcessed++;

            DomainModelMeta domainModelMeta = new DomainModelMeta();
            domainModelMeta.setName(typeElement.getSimpleName().toString());
            domainModelMeta.setFullyQualifiedName(typeElement.getQualifiedName().toString());
            domainModelMeta.setTypeElement(typeElement);

            builder.append(format("%-2d. Processing Class <%s>\n", classesProcessed, domainModelMeta.getFullyQualifiedName()));
            List<FieldMeta> metaList = getAllFieldsForType(typeElement);
            for (int i = 0; i < metaList.size(); i++) {
                FieldMeta meta = metaList.get(i);
                Optional<AnnotationMirror> hasQueryable = meta.findAnnotationMirrorForField(Queryable.class);
                builder.append(format("\t\t%-2d. name=%s type=%s hasQueryable=%s\n", i + 1, meta.getName(), meta.getFullyQualifiedName(), hasQueryable.isPresent()));

                domainModelMeta.getAllFields().put(meta.getName(), meta);
                if (hasQueryable.isPresent()) {
                    meta.setAnnotationMirror(hasQueryable.get());
                    domainModelMeta.getFieldMetas().add(meta);
                    Map<String, Object> values = getValues(hasQueryable.get());

                    QueryableMeta queryableMeta = new QueryableMeta();
                    queryableMeta.setMany((Boolean) values.get("many"));
                    queryableMeta.setPageable((Boolean) values.get("pageable"));
                    queryableMeta.setOptional((Boolean) values.get("optional"));
                    meta.setQueryableMeta(queryableMeta);
                }


                int nestedCounter = 0;
                if (!isInternalClass(meta.getFullyQualifiedName())) {
                    TypeMirror typeMirror = meta.getElement().asType();
                    if(typeMirror instanceof DeclaredType){
                        for (FieldMeta nested : getAllFieldsForType(convertToTypeElement(meta.getElement().asType()))) {
                            nested.setSource(meta);
                            Optional<AnnotationMirror> hasId = nested.findAnnotationMirrorForField(Id.class);

                            String key = meta.getName() + StringUtils.capitalize(nested.getName());

                            if (hasQueryable.isPresent() && hasId.isPresent()) {
                                domainModelMeta.getFieldMetas().add(nested);
                                nested.setQueryableMeta(meta.getQueryableMeta());
                            }

                            domainModelMeta.getAllFields().put(key, nested);
                            nestedCounter++;
                            builder.append(format("\t\t\t\t%-2d. name=%s type=%s hasId=%s\n", nestedCounter, nested.getName(), nested.getFullyQualifiedName(), hasId.isPresent()));
                        }
                    }

                }


            }
            classMetaList.add(domainModelMeta);
            Optional<AnnotationMirror> hasDomainModel = domainModelMeta.findAnnotationMirrorForField(DomainModel.class);
            if (hasDomainModel.isPresent()) {
                builder.append("\n\nDomainModel Annotation Values\n\n");
                Map<ExecutableElement, AnnotationValue> values = getAnnotationValuesWithDefaults(hasDomainModel.get());
                builder.append(format("Found %d Annotation Values\n", values.size()));
                int counter = 0;
                Map<String, Object> context = new HashMap<>(values.size());

                for (Map.Entry<ExecutableElement, AnnotationValue> entry : values.entrySet()) {
                    counter++;
                    ExecutableElement key = entry.getKey();
                    String name = key.getSimpleName().toString();
                    Object value = entry.getValue().getValue();
                    if (value instanceof List<?>) {
                        List<Object> contexts = new ArrayList<>(((List<?>) value).size());
                        for (Object v1 : (List<?>) value) {
                            if (v1 instanceof AnnotationMirror) {
                                int[] nestedCounter = {0};
                                Map<String, Object> nestedContext = new HashMap<>(values.size());
                                Map<ExecutableElement, AnnotationValue> nested = getAnnotationValuesWithDefaults((AnnotationMirror) v1);
                                nested.forEach((k, v) -> {
                                    nestedCounter[0]++;
                                    Object v2 = v.getValue();
                                    if (v2 instanceof List<?>) {
                                        List<String> attributes = new ArrayList<>(((List<?>) v2).size());
                                        for (Object f : (List<?>) v2) {
                                            if (f instanceof AnnotationValue) {
                                                attributes.add(Objects.toString(((AnnotationValue) f).getValue()));
                                            }
                                        }

                                        nestedContext.put(k.getSimpleName().toString(), attributes);
                                    } else {
                                        nestedContext.put(k.getSimpleName().toString(), v2);
                                    }

                                    builder.append(format("\t\t\t\t%-2d. key=%s value=%s\n", nestedCounter[0], k.getSimpleName(), v.getValue()));
                                });
                                contexts.add(nestedContext);

                            } else {
                                contexts.add(v1);
                            }
                        }
                        context.put(name, contexts);
                    } else {

                        context.put(name, value);
                        builder.append(format("\t\t%-2d. key=%s value=%s\n", counter, name, value));
                    }

                }

                domainModelMeta.setPrimaryKeyClass(Objects.toString(context.get("primaryKeyClass")));
                domainModelMeta.setNoRepositoryBean((Boolean) context.get("noRepositoryBean"));

                Object nestedData = context.get("queryables");
                if (nestedData != null) {
                    List<QueryablesMeta> queryablesMetas = new ArrayList<>(((List<Map<String, Object>>) nestedData).size());
                    for (Map<String, Object> data : (List<Map<String, Object>>) nestedData) {
                        messager.printMessage(Diagnostic.Kind.WARNING, json(data));
                        QueryablesMeta queryablesMeta = new QueryablesMeta();

                        queryablesMeta.setMany((Boolean) data.get("many"));
                        queryablesMeta.setOptional((Boolean) data.get("optional"));
                        queryablesMeta.setPageable((Boolean) data.get("pageable"));
                        queryablesMeta.setFieldNames((List<String>) data.get("fieldNames"));
                        queryablesMetas.add(queryablesMeta);
                    }
                    domainModelMeta.setMetas(queryablesMetas);
                }

            }
            builder.append("\n\n");

        }

        generate(classMetaList);


        messager.printMessage(Diagnostic.Kind.WARNING, builder.toString());

        return false;
    }

    public void generate(List<DomainModelMeta> classMetaList) {

        for (DomainModelMeta domainModelMeta : classMetaList) {
            final String repositoryName = domainModelMeta.getName() + "Repository";
            final String qualifiedName = domainModelMeta.getFullyQualifiedName();
            final String packageName = options.getOrDefault("codegen.packageName", qualifiedName.substring(0, qualifiedName.lastIndexOf(".")) + ".repositories");

            //generate methods

            List<String> methods = getMethods(domainModelMeta);


            StringJoiner imports = new StringJoiner(";\nimport ", "import ", ";\n\n");
            imports.add("org.springframework.data.domain.Page");
            imports.add("org.springframework.data.domain.Pageable");
            imports.add("org.springframework.data.jpa.repository.JpaRepository");
            imports.add("org.springframework.data.jpa.repository.JpaSpecificationExecutor");
            imports.add("org.springframework.data.repository.NoRepositoryBean");
            imports.add("org.springframework.stereotype.Repository");

            imports.add("java.util.List");
            imports.add("java.util.Optional");


            StringBuilder source = new StringBuilder();
            source.append("package ");
            source.append(packageName);
            source.append(";\n\n");

            source.append(imports);


            if (domainModelMeta.isNoRepositoryBean()) {
                source.append("@NoRepositoryBean\n");
            } else {
                source.append("@Repository\n");
            }

            source.append("public interface ");
            source.append(repositoryName);

            source.append(format(" extends JpaRepository<%s, %s>, JpaSpecificationExecutor<%s>", domainModelMeta.getFullyQualifiedName(),
                    domainModelMeta.getPrimaryKeyClass(), domainModelMeta.getFullyQualifiedName()));


            source.append(" {\n\n");
            source.append(String.join(";\n\n", methods));
            if (methods.size() > 0) {
                source.append(";");
            }
            source.append("\n\n}\n");


            try {
                JavaFileObject javaFileObject = processingEnv.getFiler().createSourceFile(packageName + '.' + repositoryName);
                try (Writer writer = javaFileObject.openWriter()) {
                    writer.write(source.toString());
                }
            } catch (IOException e) {
                e.printStackTrace();
            }
        }
    }

    private List<String> getMethods(DomainModelMeta domainModelMeta) {
        List<String> methods = new ArrayList<>(0);
        for (QueryablesMeta queryablesMeta : domainModelMeta.getMetas()) {
            List<String> fieldNames = queryablesMeta.getFieldNames();

            if (fieldNames != null && fieldNames.size() > 0) {
                List<String> capitalized = new ArrayList<>(fieldNames.size());
                List<String> parameters = new ArrayList<>(fieldNames.size());
                for (String fieldName : fieldNames) {
                    FieldMeta fieldMeta = domainModelMeta.getAllFields().get(fieldName);
                    if (fieldMeta == null) {
                        throw new IllegalArgumentException(format("FieldName(%s) doesn't exists on (%s)", fieldName, domainModelMeta.getFullyQualifiedName()));
                    }
                    capitalized.add(StringUtils.capitalize(fieldName));
                    parameters.add(fieldMeta.getFullyQualifiedName() + "  " + fieldName);
                }

                String methodTemplate;
                String returnType;


                if (queryablesMeta.isMany() || queryablesMeta.isPageable()) {
                    if (queryablesMeta.isPageable()) {
                        returnType = format("Page<%s>", domainModelMeta.getFullyQualifiedName());
                        methodTemplate = format("findAllBy%s(%s, Pageable pageable)", StringUtils.join(capitalized, "And"), StringUtils.join(parameters, ", "));
                    } else {
                        returnType = format("List<%s>", domainModelMeta.getFullyQualifiedName());
                        methodTemplate = format("findAllBy%s(%s)", StringUtils.join(capitalized, "And"), StringUtils.join(parameters, ", "));
                    }
                } else {
                    if (queryablesMeta.isOptional()) {
                        returnType = format("Optional<%s>", domainModelMeta.getFullyQualifiedName());
                    } else {
                        returnType = domainModelMeta.getFullyQualifiedName();
                    }
                    methodTemplate = format("findBy%s(%s)", StringUtils.join(capitalized, "And"), StringUtils.join(parameters, ", "));
                }
                methods.add("    " + returnType + " " + methodTemplate);


            }
        }

        for(FieldMeta fieldMeta: domainModelMeta.getFieldMetas()){
            QueryableMeta queryableMeta = fieldMeta.getQueryableMeta();
            String methodTemplate;
            String returnType;

            String key = fieldMeta.getName();
            if (fieldMeta.getSource() != null) {
                key = fieldMeta.getSource().getName() + StringUtils.capitalize(fieldMeta.getName());
            }

            if (queryableMeta.isMany() || queryableMeta.isPageable()) {
                if (queryableMeta.isPageable()) {
                    returnType = format("Page<%s>", domainModelMeta.getFullyQualifiedName());
                    methodTemplate = format("findAllBy%s(%s %s, Pageable pageable)", StringUtils.capitalize(key), fieldMeta.getFullyQualifiedName(), key);
                } else {
                    returnType = format("List<%s>", domainModelMeta.getFullyQualifiedName());
                    methodTemplate = format("findAllBy%s(%s %s)", StringUtils.capitalize(key), fieldMeta.getFullyQualifiedName(), key);
                }
            } else {
                if (queryableMeta.isOptional()) {
                    returnType = format("Optional<%s>", domainModelMeta.getFullyQualifiedName());
                } else {
                    returnType = domainModelMeta.getFullyQualifiedName();
                }
                methodTemplate = format("findBy%s(%s %s)", StringUtils.capitalize(key), fieldMeta.getFullyQualifiedName(), key);
            }
            methods.add( "    " + returnType + " " + methodTemplate);

        }
        return methods;
    }


}
