package io.github.gajendragusain.httpserver.embeddedserver;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import io.github.gajendragusain.httpserver.annotations.*;
import io.netty.buffer.ByteBuf;
import io.netty.buffer.Unpooled;
import io.netty.channel.ChannelFutureListener;
import io.netty.channel.ChannelHandlerContext;
import io.netty.channel.SimpleChannelInboundHandler;
import io.netty.handler.codec.DecoderResult;
import io.netty.handler.codec.http.*;
import io.netty.handler.codec.http.cookie.Cookie;
import io.netty.handler.codec.http.cookie.ServerCookieDecoder;
import io.netty.handler.codec.http.cookie.ServerCookieEncoder;
import io.netty.util.CharsetUtil;
import io.netty.util.concurrent.EventExecutorGroup;
import io.netty.util.concurrent.Future;
import io.netty.util.concurrent.GenericFutureListener;

import java.lang.annotation.Annotation;
import java.lang.reflect.Method;
import java.util.*;
import java.util.concurrent.Callable;

import static io.netty.handler.codec.http.HttpHeaderNames.*;
import static io.netty.handler.codec.http.HttpResponseStatus.BAD_REQUEST;
import static io.netty.handler.codec.http.HttpResponseStatus.OK;
import static io.netty.handler.codec.http.HttpVersion.HTTP_1_1;

public class DefaultHttpRequestHandler extends SimpleChannelInboundHandler<FullHttpRequest> {
    private final String TAG = "HttpRequestHandler";
    private FullHttpRequest request;
    private ChannelHandlerContext ctx;
    private Gson gson = new GsonBuilder().setPrettyPrinting().create();

    private Configuration configuration;
    private final EventExecutorGroup eventExecutors;

    public DefaultHttpRequestHandler(Configuration configuration, EventExecutorGroup eventExecutors) {
        this.configuration = configuration;
        this.eventExecutors = eventExecutors;
    }

    private Map<String, String> getPathVariables(String requestUri, String urlPattern) {
        String[] uriTokens = requestUri.split("/");
        String[] mappingToken = urlPattern.split("/");
        Map<String, String> pathVariableMap = new HashMap<>();
        for (int i = 0; i < mappingToken.length; i++) {
            if (mappingToken[i].matches("\\{.+?\\}")) {
                String rplace = mappingToken[i].replaceAll("\\{|\\}", "");
                pathVariableMap.put(rplace, uriTokens[i]);
            }
        }
        return pathVariableMap;
    }

    private Map<String, List<String>> getQueryParams(FullHttpRequest request) {
        Map<String, List<String>> paramMap = new HashMap<>();
        QueryStringDecoder queryStringDecoder = new QueryStringDecoder(request.uri());
        Map<String, List<String>> params = queryStringDecoder.parameters();
        if (!params.isEmpty()) {
            for (Map.Entry<String, List<String>> p : params.entrySet()) {
                String key = p.getKey();
                List<String> vals = p.getValue();
                paramMap.put(key, vals);
            }
        }
        return paramMap;
    }

    private Map<String, List<String>> getFormUrlEncodedParams(String data) {
        System.out.println("----------form data: " + data.trim());
        System.out.println("----------form data length: " + data.trim().length());
        Map<String, List<String>> formUrlEncodedParams = new HashMap<>();
        String[] tokens = data.trim().split("&");
        if (tokens.length > 0) {
            for (String keyVal : tokens) {
                String[] keyValtokens = keyVal.split("=");
                if (keyValtokens.length > 1) {
                    List<String> values = Arrays.asList(keyValtokens[1].split(";"));
                    formUrlEncodedParams.put(keyValtokens[0], values);
                }
            }
        }
        return formUrlEncodedParams;
    }

    private Map<String, Configuration.RequestConfig> getUriRequestConfigMap(String method) {
        return configuration.getMethodRequestMap().get(method);
    }

    public void invokeHandler(FullHttpRequest request) throws Exception {
        boolean handlerFound = false;
        System.out.println("----------------request uri: " + request.uri());
        String uri = request.uri().split("\\?")[0].trim();
        String contentType = request.headers().get(CONTENT_TYPE) == null ? "" : request.headers().get(CONTENT_TYPE);
        String method = request.method().name();
        String requestkey = method + "|" + uri + "|" + contentType;
        Map<String, Configuration.RequestConfig> stringRequestConfigMap = getUriRequestConfigMap(method);
        if (stringRequestConfigMap == null) {
            if (!writeDefaultResponse(request, HttpResponseStatus.METHOD_NOT_ALLOWED, ctx)) {
                ctx.writeAndFlush(Unpooled.EMPTY_BUFFER).addListener(ChannelFutureListener.CLOSE);
            }
            return;
        }

        Configuration.RequestConfig requestConfig = stringRequestConfigMap.get(requestkey);
        if (requestConfig != null) { // if key matches
            Controller a = requestConfig.controller;
            Method handlerMethod = requestConfig.handlerMethod;
            String urlPattern = requestConfig.urlPattern;
            String rawBody;
            Map<String, String> pathVariableMap = getPathVariables(uri, urlPattern);

            if (request.method().equals(HttpMethod.GET)) {
                Map<String, List<String>> paramMap = getQueryParams(request);
                processRequest(a, handlerMethod, requestConfig.responseType, pathVariableMap, paramMap);
            } else if (request.method().equals(HttpMethod.POST)) {
                ByteBuf byteBuf = request.content();
                String data = byteBuf.toString(CharsetUtil.UTF_8);
                if (requestConfig.contentType.equals(HttpHeaderValues.APPLICATION_JSON.toString()) || requestConfig.contentType.equals(HttpHeaderValues.TEXT_PLAIN.toString())) {
                    rawBody = data;
                    processRequest(a, handlerMethod, requestConfig.responseType, pathVariableMap, rawBody);
                } else if (requestConfig.contentType.equals(HttpHeaderValues.APPLICATION_X_WWW_FORM_URLENCODED.toString())) {
                    // name=gajendra&age=30&place=dehradun
                    Map<String, List<String>> paramMap = getFormUrlEncodedParams(data);
                    processRequest(a, handlerMethod, requestConfig.responseType, pathVariableMap, paramMap);
                }
            }
            handlerFound = true;
        } else {
            // match url pattern
            for (Configuration.RequestConfig u : stringRequestConfigMap.values()) {
                System.out.println("---------regex: " + u.urlRegex);
                if (uri.matches(u.urlRegex) && request.method().name().equals(u.httpMethod) && contentType.equals(u.contentType)) {
                    Controller a = u.controller;
                    Method handlerMethod = u.handlerMethod;
                    String rawBody;
                    Map<String, String> pathVariableMap = getPathVariables(uri, u.urlPattern);
                    if (request.method().equals(HttpMethod.GET)) {
                        Map<String, List<String>> paramMap = getQueryParams(request);
                        processRequest(a, handlerMethod, u.responseType, pathVariableMap, paramMap);
                    } else if (request.method().equals(HttpMethod.POST)) {
                        ByteBuf byteBuf = request.content();
                        String data = byteBuf.toString(CharsetUtil.UTF_8);
                        if (u.contentType.equals(HttpHeaderValues.APPLICATION_JSON.toString()) || u.contentType.equals(HttpHeaderValues.TEXT_PLAIN.toString())) {
                            rawBody = data;
                            processRequest(a, handlerMethod, u.responseType, pathVariableMap, rawBody);
                        } else if (u.contentType.equals(HttpHeaderValues.APPLICATION_X_WWW_FORM_URLENCODED.toString())) {
                            // name=gajendra&age=30&place=dehradun
                            Map<String, List<String>> paramMap = getFormUrlEncodedParams(data);
                            processRequest(a, handlerMethod, u.responseType, pathVariableMap, paramMap);
                        }


                            /*QueryStringDecoder queryStringDecoder = new QueryStringDecoder(
                                    request.uri());
                            Map<String, List<String>> requestParameters;

                            requestParameters = queryStringDecoder.parameters();
                            System.out.println("--------------requestParams: "+requestParameters);
                            HttpPostRequestDecoder decoder = new HttpPostRequestDecoder(
                                    new DefaultHttpDataFactory(false), request);
                            try {
                                while (decoder.hasNext()) {
                                    InterfaceHttpData httpData = decoder.next();
                                    if (httpData.getHttpDataType() == InterfaceHttpData.HttpDataType.Attribute) {
                                        Attribute attribute = (Attribute) httpData;
                                        if (!requestParameters.containsKey(attribute.getName())) {
                                            requestParameters.put(attribute.getName(),
                                                    new LinkedList<>());
                                        }
                                        requestParameters.get(attribute.getName()).add(
                                                attribute.getValue());
                                        attribute.release();
                                    }
                                }
                                System.out.println("--------------updated requestParams: "+requestParameters);
                            } catch (HttpPostRequestDecoder.EndOfDataDecoderException ex) {
                                // Exception when the body is fully decoded, even if there
                                // is still data
                            }

                            decoder.destroy();*/

                    }
                    handlerFound = true;
                    break;
                }
            }
        }
        if (!handlerFound) {
            resourceNotFound(request, ctx);
        }
    }

    protected static void resourceNotFound(FullHttpRequest request, ChannelHandlerContext ctx, StringBuilder buf, CharSequence contentType) {
        HttpResponseStatus status = request.decoderResult().isSuccess() ? HttpResponseStatus.OK : HttpResponseStatus.BAD_REQUEST;
        if (!writeResponse(buf, request, status, ctx, contentType)) {
            // If keep-alive is off, close the connection once the content is fully written.
            ctx.writeAndFlush(Unpooled.EMPTY_BUFFER).addListener(ChannelFutureListener.CLOSE);
        }
    }

    protected static void resourceNotFound(FullHttpRequest request, ChannelHandlerContext ctx) {
        if (!writeDefaultResponse(request, HttpResponseStatus.BAD_REQUEST, ctx)) {
            // If keep-alive is off, close the connection once the content is fully written.
            ctx.writeAndFlush(Unpooled.EMPTY_BUFFER).addListener(ChannelFutureListener.CLOSE);
        }
    }

    protected static boolean writeDefaultResponse(FullHttpRequest request, HttpResponseStatus status, ChannelHandlerContext ctx) {
        // Decide whether to close the connection or not.
//        boolean keepAlive = HttpUtil.isKeepAlive(request);
        boolean keepAlive = false;

        // Build the response object.
        HttpResponse fullhttpResponse = new DefaultHttpResponse(
                HTTP_1_1, status);

        if (keepAlive) {
            // Add 'Content-Length' header only for a keep-alive connection.
//            fullhttpResponse.headers().set(CONTENT_LENGTH, fullhttpResponse.content().readableBytes());
            // Add keep alive header as per:
            // - http://www.w3.org/Protocols/HTTP/1.1/draft-ietf-http-v11-spec-01.html#Connection
            fullhttpResponse.headers().set(CONNECTION, HttpHeaderValues.KEEP_ALIVE);
        }

        // Encode the cookie.
        String cookieString = request.headers().get(COOKIE);
        if (cookieString != null) {
            Set<Cookie> cookies = ServerCookieDecoder.STRICT.decode(cookieString);
            if (!cookies.isEmpty()) {
                // Reset the cookies if necessary.
                for (Cookie cookie : cookies) {
                    fullhttpResponse.headers().add(SET_COOKIE, ServerCookieEncoder.STRICT.encode(cookie));
                }
            }
        } else {
            // Browser sent no cookie.  Add some.
            fullhttpResponse.headers().add(SET_COOKIE, ServerCookieEncoder.STRICT.encode("key1", "value1"));
            fullhttpResponse.headers().add(SET_COOKIE, ServerCookieEncoder.STRICT.encode("key2", "value2"));
        }

        // Write the response.
        ctx.write(fullhttpResponse);

        return keepAlive;
    }

    protected static boolean writeResponse(StringBuilder buffer, FullHttpRequest request, HttpResponseStatus status, ChannelHandlerContext ctx, CharSequence contentType) {
        // Decide whether to close the connection or not.
//        boolean keepAlive = HttpUtil.isKeepAlive(request);
        boolean keepAlive = false;

        buffer.append("\r\n");
        // Build the response object.
        FullHttpResponse fullhttpResponse = new DefaultFullHttpResponse(
                HTTP_1_1, status,
                Unpooled.copiedBuffer(buffer.toString(), CharsetUtil.UTF_8));

        fullhttpResponse.headers().set(CONTENT_TYPE, contentType);

        if (keepAlive) {
            // Add 'Content-Length' header only for a keep-alive connection.
            fullhttpResponse.headers().set(CONTENT_LENGTH, fullhttpResponse.content().readableBytes());
            // Add keep alive header as per:
            // - http://www.w3.org/Protocols/HTTP/1.1/draft-ietf-http-v11-spec-01.html#Connection
            fullhttpResponse.headers().set(CONNECTION, HttpHeaderValues.KEEP_ALIVE);
        }

        // Encode the cookie.
        String cookieString = request.headers().get(COOKIE);
        if (cookieString != null) {
            Set<Cookie> cookies = ServerCookieDecoder.STRICT.decode(cookieString);
            if (!cookies.isEmpty()) {
                // Reset the cookies if necessary.
                for (Cookie cookie : cookies) {
                    fullhttpResponse.headers().add(SET_COOKIE, ServerCookieEncoder.STRICT.encode(cookie));
                }
            }
        } else {
            // Browser sent no cookie.  Add some.
            fullhttpResponse.headers().add(SET_COOKIE, ServerCookieEncoder.STRICT.encode("key1", "value1"));
            fullhttpResponse.headers().add(SET_COOKIE, ServerCookieEncoder.STRICT.encode("key2", "value2"));
        }

        // Write the response.
        ctx.write(fullhttpResponse);

        return keepAlive;
    }

    private void processRequest(Controller urlHandler, Method method, String responseType, Map<String, String> pathVariableMap, Map<String, List<String>> params) throws Exception {
        Annotation[][] annotationArr = method.getParameterAnnotations();
        Class<?>[] types = method.getParameterTypes();
        List<Object> args = new ArrayList<>();
        args.add(request);
//        args.add(ctx);

        List<Annotation[]> annotation = new ArrayList<>();
        for (Annotation[] annotations1 : annotationArr) {
            if (annotations1.length > 0)
                annotation.add(annotations1);
        }

        for (int i = 0; i < annotationArr.length; i++) {
            if (annotationArr[i].length == 0) {
                continue;
            }
            Annotation a = annotationArr[i][0];
            if (a instanceof PathVariable) {
                PathVariable pathVariable = (PathVariable) a;
                String val = pathVariableMap.get(pathVariable.value());
                processString(types[i], val, args);
            } else if (a instanceof Field) {
                Field field = (Field) a;
                for (String val : params.get(field.value())) {
                    processString(types[i], val, args);
                }
            } else if (a instanceof Query) {
                Query query = (Query) a;
                for (String val : params.get(query.value())) {
                    processString(types[i], val, args);
                }
            } else if (a instanceof QueryParams) {
                args.add(params);
            }
        }
//        method.invoke(urlHandler, args.toArray());
        sendResponse(urlHandler, method, responseType, args.toArray());
    }

    private void processRequest(Controller urlHandler, Method method, String responseType, Map<String, String> pathVariableMap, String raw) throws Exception {
        Annotation[][] annotationArr = method.getParameterAnnotations();
        Class<?>[] types = method.getParameterTypes();
        List<Object> args = new ArrayList<>();
        args.add(request);
//        args.add(ctx);

        List<Annotation[]> annotation = new ArrayList<>();
        for (Annotation[] annotations1 : annotationArr) {
            if (annotations1.length > 0)
                annotation.add(annotations1);
        }

        for (int i = 0; i < annotationArr.length; i++) {
            if (annotationArr[i].length == 0) {
                continue;
            }
            Annotation a = annotationArr[i][0];
            if (a instanceof PathVariable) {
                PathVariable pathVariable = (PathVariable) a;
                String val = pathVariableMap.get(pathVariable.value());
                processString(types[i], val, args);
            } else if (a instanceof Body) {
                String contentType = request.headers().get(CONTENT_TYPE);
                if (contentType.equals(HttpHeaderValues.APPLICATION_JSON.toString())) {
                    Object o = gson.fromJson(raw, types[i]);
                    args.add(o);
                } else if(contentType.equals(HttpHeaderValues.TEXT_PLAIN.toString())) {
                    args.add(raw);
                } else {
                    throw new RuntimeException(contentType + " is currently not supported for raw body type");
                }
            }
        }
        sendResponse(urlHandler, method, responseType, args.toArray());
//        method.invoke(urlHandler, args.toArray());
    }

    private void sendResponse(Controller controller, Method method, String responseType, Object[] args) {
        Callable<Object> callable = new Callable<Object>() {
            @Override
            public Object call() throws Exception {
                // TODO: perform request processing
                Object result = method.invoke(controller, args);
                return encodeResult(result, responseType);
            }
        };

        Future<Object> future = eventExecutors.submit(callable);
        future.addListener(new GenericFutureListener<Future<Object>>() {
            @Override
            public void operationComplete(Future<Object> objectFuture) throws Exception {
                if (objectFuture.isSuccess()) {
                    HttpResponseStatus status = request.decoderResult().isSuccess() ? OK : BAD_REQUEST;
                    String result = objectFuture.get() == null ? "NULL" : objectFuture.get().toString();
                    StringBuilder buf = new StringBuilder(result);
                    if (!writeResponse(buf, request, status, ctx, responseType)) {
                        // If keep-alive is off, close the connection once the content is fully written.
                        ctx.writeAndFlush(Unpooled.EMPTY_BUFFER).addListener(ChannelFutureListener.CLOSE);
                    }
                } else {
                    StringBuilder buf = new StringBuilder(objectFuture.cause().getMessage());
                    if (!writeResponse(buf, request, HttpResponseStatus.INTERNAL_SERVER_ERROR, ctx, responseType)) {
                        ctx.writeAndFlush(Unpooled.EMPTY_BUFFER).addListener(ChannelFutureListener.CLOSE);
                    }
                }
            }
        });
    }

    private String encodeResult(Object result, String responseType) throws Exception {
        String response = "";
        switch (responseType) {
            case "application/json":
                response = gson.toJson(result);
                break;
            case "text/plain":
                response = result.toString();
                break;
            default:
                throw new RuntimeException("Unsupported response type");
        }
        return response;
    }

    private void processString(Class<?> type, String val, List<Object> args) {
        switch (type.getName()) {
            case "int":
                args.add(Integer.parseInt(val));
                break;
            case "long":
                args.add(Long.parseLong(val));
                break;
            case "boolean":
                args.add(Boolean.parseBoolean(val));
                break;
            case "java.lang.String":
                args.add(val);
                break;
            default:
                throw new RuntimeException(type.getName() + " is not supported");
        }
    }


    @Override
    protected void channelRead0(ChannelHandlerContext ctx, FullHttpRequest msg) throws Exception {
        this.request = msg;
        this.ctx = ctx;
        // TODO: handle request type /{id}/
        invokeHandler(request);
    }


    private static void appendDecoderResult(StringBuilder buf, HttpObject o) {
        DecoderResult result = o.decoderResult();
        if (result.isSuccess()) {
            return;
        }

        buf.append(".. WITH DECODER FAILURE: ");
        buf.append(result.cause());
        buf.append("\r\n");
    }

    @Override
    public void exceptionCaught(ChannelHandlerContext ctx, Throwable cause) throws Exception {
        cause.printStackTrace();
        ctx.close();
    }
}
