package io.github.gajendragusain.httpserver.embeddedserver;

import io.netty.bootstrap.ServerBootstrap;
import io.netty.channel.Channel;
import io.netty.channel.ChannelOption;
import io.netty.channel.EventLoopGroup;
import io.netty.channel.nio.NioEventLoopGroup;
import io.netty.channel.socket.nio.NioServerSocketChannel;
import io.netty.handler.logging.LogLevel;
import io.netty.handler.logging.LoggingHandler;
import io.netty.handler.ssl.SslContext;
import io.netty.handler.ssl.SslContextBuilder;
import io.netty.handler.ssl.util.SelfSignedCertificate;
import io.netty.util.concurrent.Future;

import javax.net.ssl.SSLException;
import java.security.cert.CertificateException;

public final class EmbeddedHttpServer {
    private static final String TAG = "EposServer";
    static final boolean SSL = System.getProperty("ssl") != null;
    static final int PORT = Integer.parseInt(System.getProperty("port", SSL ? "8443" : "8080"));

    private final int MAX_THREADS = 2;
    private int maxThreads = MAX_THREADS;
    private Callback callback;

    public void setCallback(Callback callback) {
        this.callback = callback;
    }

    public int getMaxThread() {
        return maxThreads;
    }

    public void setMaxThread(int maxThreads) {
        this.maxThreads = maxThreads;
    }

    //    public void start(Context context, Handler handler) throws CertificateException, SSLException, InterruptedException {
    public void start(Configuration configuration) throws CertificateException, SSLException, InterruptedException {
        final SslContext sslCtx;
        if (SSL) {
            SelfSignedCertificate ssc = new SelfSignedCertificate();
            sslCtx = SslContextBuilder.forServer(ssc.certificate(), ssc.privateKey()).build();
        } else {
            sslCtx = null;
        }

        // Configure the server.
        EventLoopGroup bossGroup = new NioEventLoopGroup(maxThreads);
        EventLoopGroup workerGroup = new NioEventLoopGroup();
        try {
            ServerBootstrap b = new ServerBootstrap();
            b.group(bossGroup, workerGroup)
                    .channel(NioServerSocketChannel.class)
                    .handler(new LoggingHandler(LogLevel.INFO))
                    .childHandler(new EmbeddedHttpServerInitializer(sslCtx, configuration))
                    .option(ChannelOption.SO_BACKLOG, 128)
                    .childOption(ChannelOption.SO_KEEPALIVE, true);

            Channel ch = b.bind(PORT).sync().channel();
            if (callback != null) {
                callback.onStart();
            }

            System.err.println("Open your web browser and navigate to " +
                    (SSL ? "https" : "http") + "://127.0.0.1:" + PORT + '/');

            ch.closeFuture().sync();
            if (callback != null) {
                callback.onShutDown();
            }
        } finally {
            Future bossFuture = bossGroup.shutdownGracefully();
            Future workerFuture = workerGroup.shutdownGracefully();

            bossFuture.addListener(future -> {
                if (future.isSuccess()) {
                    System.out.println("Boss group was shutdown successfully");
                } else {
                    System.out.println("Boss group did not shutdown successfully");
                    future.cause().printStackTrace();
                }
            });

            workerFuture.addListener(future -> {
                if (future.isSuccess()) {
                    System.out.println("Worker group was shutdown successfully");
                } else {
                    System.out.println("Worker group did not shutdown successfully");
                    future.cause().printStackTrace();
                }
            });
        }
    }

    public interface Callback {
        void onShutDown();

        void onStart();
    }
}