/*
 * Decompiled with CFR 0.152.
 */
package io.gorules.zen.loader;

import com.github.benmanes.caffeine.cache.Cache;
import com.github.benmanes.caffeine.cache.Caffeine;
import io.gorules.zen.loader.ApiLoaderConfig;
import io.gorules.zen.loader.ApiLoaderException;
import io.gorules.zen.loader.DecisionLoader;
import io.gorules.zen_engine.JsonBuffer;
import java.net.URI;
import java.net.URLEncoder;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.nio.charset.StandardCharsets;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.concurrent.TimeUnit;

public class ApiDecisionLoader
implements DecisionLoader {
    private final HttpClient httpClient;
    private final ApiLoaderConfig config;
    private final Cache<String, JsonBuffer> cache;

    public ApiDecisionLoader(ApiLoaderConfig config) {
        this.config = config;
        this.httpClient = HttpClient.newBuilder().connectTimeout(config.getTimeout()).build();
        this.cache = config.isEnableCaching() ? this.buildCache(config) : null;
    }

    private Cache<String, JsonBuffer> buildCache(ApiLoaderConfig config) {
        Caffeine builder = Caffeine.newBuilder().expireAfterWrite(config.getCacheTtl());
        switch (config.getCacheEvictionPolicy()) {
            case LRU: {
                builder.maximumSize(config.getCacheMaxSize());
                break;
            }
            case LFU: {
                builder.maximumSize(config.getCacheMaxSize());
                break;
            }
            case SIZE_BASED: {
                long maxBytes = config.getCacheMaxMemoryMb() * 1024L * 1024L;
                builder.maximumWeight(maxBytes).weigher((key, value) -> {
                    int keySize = Math.min(0x3FFFFFFF, key.length() * 2);
                    int valueSize = value == null || value.value() == null ? 0 : value.value().length;
                    long size = (long)keySize + (long)valueSize + 64L;
                    return (int)Math.min(Integer.MAX_VALUE, size);
                });
            }
        }
        return builder.build();
    }

    @Override
    public CompletableFuture<JsonBuffer> load(String key) {
        JsonBuffer cached;
        if (this.cache != null && (cached = (JsonBuffer)this.cache.getIfPresent((Object)key)) != null) {
            return CompletableFuture.completedFuture(cached);
        }
        return this.loadFromApi(key, 0);
    }

    private CompletableFuture<JsonBuffer> loadFromApi(String key, int attempt) {
        try {
            String encodedKey = URLEncoder.encode(key, StandardCharsets.UTF_8);
            String url = this.config.getBaseUrl() + "/" + encodedKey;
            HttpRequest.Builder requestBuilder = HttpRequest.newBuilder().uri(URI.create(url)).timeout(this.config.getTimeout()).GET();
            Map<String, String> headers = this.config.getAllHeaders();
            headers.forEach(requestBuilder::header);
            HttpRequest request = requestBuilder.build();
            return ((CompletableFuture)this.httpClient.sendAsync(request, HttpResponse.BodyHandlers.ofByteArray()).handle((response, ex) -> {
                if (ex != null) {
                    return this.handleException((Throwable)ex, key, attempt);
                }
                return this.handleResponse((HttpResponse<byte[]>)response, key, attempt);
            })).thenCompose(future -> future);
        }
        catch (Exception e) {
            return CompletableFuture.failedFuture(new ApiLoaderException("Failed to build request for key: " + key, e));
        }
    }

    private CompletableFuture<JsonBuffer> handleResponse(HttpResponse<byte[]> response, String key, int attempt) {
        int statusCode = response.statusCode();
        if (statusCode == 200) {
            JsonBuffer buffer = new JsonBuffer(response.body());
            if (this.cache != null) {
                this.cache.put((Object)key, (Object)buffer);
            }
            return CompletableFuture.completedFuture(buffer);
        }
        if (statusCode >= 500 && statusCode < 600 && attempt < this.config.getMaxRetries()) {
            return this.retryWithBackoff(key, attempt);
        }
        if (statusCode == 404) {
            return CompletableFuture.failedFuture(new ApiLoaderException("Decision not found: " + key + " (HTTP 404)"));
        }
        if (statusCode == 401 || statusCode == 403 || statusCode == 400) {
            return CompletableFuture.failedFuture(new ApiLoaderException("Authentication failed for key: " + key + " (HTTP " + statusCode + ")"));
        }
        String body = new String(response.body(), StandardCharsets.UTF_8);
        return CompletableFuture.failedFuture(new ApiLoaderException("Failed to load decision: " + key + " (HTTP " + statusCode + "): " + body));
    }

    private CompletableFuture<JsonBuffer> handleException(Throwable ex, String key, int attempt) {
        if (attempt < this.config.getMaxRetries()) {
            return this.retryWithBackoff(key, attempt);
        }
        Throwable cause = ex instanceof CompletionException && ex.getCause() != null ? ex.getCause() : ex;
        return CompletableFuture.failedFuture(new ApiLoaderException("Failed to load decision: " + key, cause));
    }

    private CompletableFuture<JsonBuffer> retryWithBackoff(String key, int attempt) {
        long baseDelayMs = Math.max(1L, this.config.getRetryDelay().toMillis());
        long delayMs = baseDelayMs * (1L << Math.min(attempt, 30));
        return CompletableFuture.supplyAsync(() -> null, CompletableFuture.delayedExecutor(delayMs, TimeUnit.MILLISECONDS)).thenCompose(v -> this.loadFromApi(key, attempt + 1));
    }

    public HttpClient getHttpClient() {
        return this.httpClient;
    }

    public ApiLoaderConfig getConfig() {
        return this.config;
    }

    public Cache<String, JsonBuffer> getCache() {
        return this.cache;
    }
}

