package com.liecoder.framework.base

import android.view.LayoutInflater
import androidx.viewbinding.ViewBinding
import java.lang.reflect.ParameterizedType
import java.lang.reflect.Type

/**
 * 基础视图绑定活动类，提供 ViewBinding 支持。
 *
 * 此类继承自 BaseActivity，并实现了 IViewBindingService 接口，用于简化使用 ViewBinding 的过程。
 * 它使用泛型 VB 来指定具体的 ViewBinding 类型，以确保类型安全和编译时检查。
 *
 * @author liecoder
 * @date 2024/6/21 周五
 * @version 1.0
 */
@Suppress("UNCHECKED_CAST")
abstract class BaseViewBindingActivity<VB : ViewBinding> : BaseActivity(), IViewBindingService {


    /**
     * 延迟初始化的 ViewBinding 实例。
     *
     * 使用 lateinit 修饰，确保在 setContentView 方法中初始化。
     */
    override lateinit var mBinding: VB


    /**
     * 设置内容视图的方法，使用 ViewBinding 绑定布局。
     */
    override fun setContentView() {
        mBinding = viewBinding()
        setContentView(mBinding.root)
    }


    /**
     * 创建 ViewBinding 实例的抽象方法。
     *
     * 此方法通过反射获取当前 Activity 的 ViewBinding 类型，并调用其 inflate 方法来创建实例。
     * 它使用 Java 类的 genericSuperclass 属性来获取泛型超类的类型信息，然后通过反射调用 inflate 方法。
     *
     * @return 创建的 ViewBinding 实例。
     */
    override fun viewBinding(): VB {
        val binding: VB
        val type: Type = javaClass.genericSuperclass!!
        val clazz: Class<VB> = ((type as ParameterizedType).actualTypeArguments[0] as Class<VB>)
        val inflateMethod = clazz.getDeclaredMethod(
            "inflate",
            LayoutInflater::class.java
        )
        binding = inflateMethod.invoke(this, layoutInflater) as VB
        return binding
    }

}