package com.liecoder.framework.base

import android.content.Context
import android.view.LayoutInflater
import android.view.View
import android.view.animation.Animation
import androidx.viewbinding.ViewBinding
import com.liecoder.framework.ktx.saveAs
import com.liecoder.framework.ktx.saveAsUnchecked
import razerdp.basepopup.BasePopupWindow
import java.lang.reflect.ParameterizedType

/**
 * 基础PopupWindow类，提供通用的PopupWindow视图和行为实现。
 *
 * 此类继承自 BasePopupWindow 并实现了 IBaseWindowService 接口，用于创建和管理弹出PopupWindow。
 * 它使用泛型 VB 来指定具体的 ViewBinding 类型，确保类型安全和编译时检查。
 *
 * @param context 用于创建PopupWindow的上下文。
 * @author liecoder
 * @date 2024/7/3 周三
 * @version 1.0
 */
abstract class BaseWindow<VB : ViewBinding>(context: Context) : BasePopupWindow(context),
    IBaseWindowService {

    /**
     * 延迟初始化的 ViewBinding 实例。
     *
     * <p>使用初始化块来通过反射创建 ViewBinding 实例。
     */
    protected var mBinding: VB


    /**
     * 构造函数的初始化块。
     *
     * 在初始化块中，使用 Java 反射来动态创建 ViewBinding 实例，并设置PopupWindow的属性。
     */
    init {
        // 获取当前类的泛型超类类型。
        val type = javaClass.genericSuperclass
        // 获取 ViewBinding 类型参数。
        val vbClass: Class<VB> =
            type!!.saveAs<ParameterizedType>().actualTypeArguments[0].saveAsUnchecked()
        // 获取 ViewBinding 类的 inflate 方法。
        val method = vbClass.getDeclaredMethod(
            "inflate",
            LayoutInflater::class.java
        )
        // 确保私有方法可以被访问。
        method.isAccessible = true
        // 调用 inflate 方法并创建 ViewBinding 实例。
        mBinding = method.invoke(null, LayoutInflater.from(context))!!.saveAsUnchecked()
        // 设置PopupWindow属性。
        setOverlayNavigationBar(true)
        popupGravity = gravity()
        initView()
        // 设置PopupWindow内容视图为 ViewBinding 的根视图。
        contentView = mBinding.root
    }


    /**
     * 显示PopupWindow的方法。
     *
     * 在显示PopupWindow之前，更新视图内容，并调用父类的 showPopupWindow 方法来显示PopupWindow。
     *
     * @param anchorView PopupWindow附着的视图，用于定位PopupWindow。
     */
    override fun show(anchorView: View?) {
        updateView()
        showPopupWindow(anchorView)
    }


    /**
     * 创建显示PopupWindow时的动画。
     *
     * 调用 IBaseWindowService 接口的 showAnimation 方法来获取显示动画。
     */
    override fun onCreateShowAnimation(): Animation? = showAnimation()


    /**
     * 创建隐藏PopupWindow时的动画。
     *
     * 调用 IBaseWindowService 接口的 dismissAnimation 方法来获取隐藏动画。
     */
    override fun onCreateDismissAnimation(): Animation? = dismissAnimation()

}