package com.liecoder.framework.ktx


/**
 * 扩展属性，用于检查 CharSequence 对象是否包含至少一个 emoji 表情符号。
 *
 * @receiver CharSequence 被检查的对象。
 * @return Boolean 返回 true 如果 CharSequence 包含至少一个 emoji，否则返回 false。
 */
val CharSequence.isEmoji: Boolean
    get() {
        for (i in indices) {
            if (get(i).isEmoji()) {
                return true
            }
        }
        return false
    }


/**
 * 判断当前字符是否是 Emoji 表情符号。
 *
 * Emoji 符号包括多种字符，可能单独存在或成对出现（称为代理对）。此函数检查字符是否属于 Unicode 标准中定义的 Emoji 范围。
 *
 * @return 如果字符是 Emoji 符号，则返回 true，否则返回 false。
 */
fun Char.isEmoji(): Boolean {
    // 检查字符的类型是否为代理对（surrogate pair），这是 Unicode 中用于表示某些复杂表情符号的一对字符。
    // 如果是代理对，函数直接返回 true，表示这是一个表情符号
    if (Character.getType(this) == Character.SURROGATE.toInt()) {
        return true
    }
    // 检查字符的 Unicode 编码是否落在表情符号的基本多语言平面（Basic Multilingual Plane）范围内。
    if (this.code in 0x1F600..0x1F64F // 表情符号和情感符号
        || this.code in 0x1F300..0x1F5FF // 交通工具和地图符号
        || this.code in 0x1F680..0x1F6FF // 运输和地图标志
        || this.code in 0x2600..0x26FF // 天气、星座和宗教符号
        || this.code in 0x2700..0x27BF // 箭头和其他符号
        || this.code in 0xFE00..0xFE0F // 字符的变体选择器，用于调整表情符号的样式
        || this.code in 0x1F900..0x1F9FF // 其他补充符号和图像
    ) {
        return true
    }
    // 检查它是否落在辅助多语言平面（Supplementary Multilingual Plane）的 0x1F1E6..0x1F1FF 范围内，这也是表情符号的一个区间
    return this.code in 0x1F1E6..0x1F1FF
}


/**
 * 判断给定的字符是否为汉字。
 *
 * @return 如果字符是汉字，则返回 true，否则返回 false。
 */
val Char.isChineseCharacter: Boolean
    get() = this.code in 0x4E00..0x9FFF // 基本汉字
            || this.code in 0x3400..0x4DBF // 扩展A区
            || this.code in 0x20000..0x2A6DF // 扩展B区
            || this.code in 0x2A700..0x2B73F // 扩展C区
            || this.code in 0x2B740..0x2B81F // 扩展D区
            || this.code in 0x2B820..0x2CEAF // 扩展E区
            || this.code in 0x2CEB0..0x2EBEF // 扩展F区
            || this.code in 0x30000..0x3134F // 扩展G区