@file:JvmName("DateTime")

package com.liecoder.framework.ktx

import android.annotation.SuppressLint
import com.liecoder.framework.model.DateFormatPattern
import java.text.SimpleDateFormat
import java.util.Calendar
import java.util.Date


/**
 * 使用指定的日期格式模式将 [Date] 对象格式化为字符串。
 * @param pattern 用于格式化的日期格式模式。
 * @return 格式化后的日期字符串。
 */
@SuppressLint("SimpleDateFormat")
fun Date.format(pattern: DateFormatPattern): String? =
    SimpleDateFormat(pattern.pattern).format(this)


/**
 * 使用指定的日期格式模式将时间戳（Long 类型）格式化为字符串。
 * @param pattern 用于格式化的日期格式模式。
 * @return 格式化后的日期字符串。
 */
@SuppressLint("SimpleDateFormat")
fun Long.format(pattern: DateFormatPattern): String? = SimpleDateFormat(pattern.pattern).format(
    Date(if ("$this".length == 10) this.times(1000) else this)
)


/**
 * 将字符串表示的日期按当前模式解析，并使用新模式格式化。
 * @param currentPattern 当前日期字符串的格式模式。
 * @param nextPattern 要格式化到的日期格式模式。
 * @return 格式化后的日期字符串。
 */
fun String.format(currentPattern: DateFormatPattern, nextPattern: DateFormatPattern): String? =
    date(currentPattern)?.format(nextPattern)


/**
 * 尝试使用指定的日期格式模式解析字符串为 [Date] 对象。
 * @param pattern 用于解析的日期格式模式。
 * @return 解析后的 [Date] 对象，如果解析失败返回 null。
 */
@SuppressLint("SimpleDateFormat")
fun String.date(pattern: DateFormatPattern): Date? = try {
    if (isEmpty()) null else SimpleDateFormat(pattern.pattern).parse(this)
} catch (e: Exception) {
    e.printStackTrace()
    null
}

/**
 * 检查 [Date] 对象是否表示今天的日期。
 * @return 如果日期是今天的日期，则返回 true。
 */
fun Date.withinToday(): Boolean {
    val calendar = Calendar.getInstance()
    val today = Calendar.getInstance()
    calendar.timeInMillis = time
    return calendar.get(Calendar.YEAR) == today.get(Calendar.YEAR) && calendar.get(Calendar.DAY_OF_YEAR) == today.get(
        Calendar.DAY_OF_YEAR
    )
}



/**
 * 检查 Long 类型的时间戳是否表示今天的日期。
 * @return 如果时间戳表示的是今天的日期，则返回 true。
 */
fun Long.withinToday(): Boolean {
    val calendar = Calendar.getInstance()
    val today = Calendar.getInstance()
    calendar.timeInMillis = this
    return calendar.get(Calendar.YEAR) == today.get(Calendar.YEAR) && calendar.get(Calendar.DAY_OF_YEAR) == today.get(
        Calendar.DAY_OF_YEAR
    )
}


/**
 * 检查 [Date] 对象是否在未来7天内。
 * @return 如果日期在未来7天内，则返回 true。
 */
fun Date.within7Days(): Boolean {
    val currentDate = Date()
    val calendar = Calendar.getInstance()
    calendar.time = currentDate
    val currentYear = calendar.get(Calendar.YEAR)
    val currentDayOfYear = calendar.get(Calendar.DAY_OF_YEAR)
    calendar.time = this
    val dateYear = calendar.get(Calendar.YEAR)
    val dateDayOfYear = calendar.get(Calendar.DAY_OF_YEAR)
    return (dateYear == currentYear && dateDayOfYear in currentDayOfYear until (currentDayOfYear + 7))
}


/**
 * 检查 Long 类型的时间戳是否表示未来7天内的日期。
 * @return 如果时间戳表示的日期在未来7天内，则返回 true。
 */
fun Long.within7Days(): Boolean =
    Date(if ("$this".length == 10) this.times(1000) else this).within7Days()


/**
 * 检查 [Date] 对象是否在未来3天内。
 * @return 如果日期在未来3天内，则返回 true。
 */
fun Date.within3Days(): Boolean {
    val currentDate = Date()
    val calendar = Calendar.getInstance()
    calendar.time = currentDate
    val currentYear = calendar.get(Calendar.YEAR)
    val currentDayOfYear = calendar.get(Calendar.DAY_OF_YEAR)
    calendar.time = this
    val dateYear = calendar.get(Calendar.YEAR)
    val dateDayOfYear = calendar.get(Calendar.DAY_OF_YEAR)
    return (dateYear == currentYear && dateDayOfYear in currentDayOfYear until (currentDayOfYear + 3))
}

/**
 * 检查 Long 类型的时间戳表示的日期是否已过期，即是否在当前日期之前。
 * @return 如果时间戳表示的日期已过期，则返回 true。
 */
fun Long.isExpired(): Boolean =
    Date(if ("$this".length == 10) this.times(1000) else this).before(Date())


/**
 * 获取 [Date] 对象在指定天数后的日期。
 * @param days 要添加的天数。
 * @return 新的 [Date] 对象，表示原始日期加上指定天数后的日期。
 */
fun Date.after(days: Int): Date {
    val calendar = Calendar.getInstance()
    calendar.time = this
    calendar.set(Calendar.DATE, calendar.get(Calendar.DATE) + days)
    return calendar.time
}

