package com.liecoder.framework.ktx

import android.annotation.SuppressLint
import android.content.Intent
import android.graphics.Bitmap
import android.net.Uri
import android.os.Environment
import com.liecoder.framework.global.AppHelper.application
import com.liecoder.framework.model.DateFormatPattern
import java.io.File
import java.io.FileInputStream
import java.io.FileOutputStream
import java.io.IOException
import java.io.InputStream
import java.text.DecimalFormat
import java.util.Date
import kotlin.math.log10
import kotlin.math.pow

/**
 * 将当前输入流（InputStream）中的数据复制到指定的文件中，并返回复制操作是否成功。
 *
 * @param destinationFile 要复制数据到的目标文件。
 * @param bufferSize 用于复制操作的缓冲区大小，默认为4096字节。
 * @return 如果复制成功返回true，否则返回false。
 */
fun InputStream.copy(destinationFile: File, bufferSize: Int = 4096): Boolean {
    val buffer = ByteArray(bufferSize)
    try {
        use { fileInputStream ->
            FileOutputStream(destinationFile).use { fileOutputStream ->
                var length: Int
                while (fileInputStream.read(buffer).also { length = it } != -1) {
                    fileOutputStream.write(buffer, 0, length)
                }
            }
        }
        return true
    } catch (e: IOException) {
        e.printStackTrace()
        return false
    }
}

/**
 * 将当前输入流（InputStream）中的数据复制到由路径指定的新文件中，并返回复制操作是否成功。
 *
 * @param destinationPath 目标文件的路径。
 * @param bufferSize 用于复制操作的缓冲区大小，默认为4096字节。
 * @return 如果复制成功返回true，否则返回false。
 */
fun InputStream.copy(destinationPath: String, bufferSize: Int = 4096): Boolean {
    val destinationFile = File(destinationPath)
    if (destinationFile.exists()) {
        destinationFile.delete()
    }
    destinationFile.createNewFile()
    return copy(destinationFile, bufferSize)
}

/**
 * 将由当前字符串表示的文件路径的文件内容复制到指定的新文件路径，并返回复制操作是否成功。
 *
 * @param destinationPath 要复制数据到的目标文件路径。
 * @param bufferSize 用于复制操作的缓冲区大小，默认为4096字节。
 * @return 如果复制成功返回true，否则返回false。
 */
fun String.copy(destinationPath: String, bufferSize: Int = 4096): Boolean {
    val sourceFile = File(this)
    if (!sourceFile.exists()) {
        return false
    }
    return FileInputStream(sourceFile).copy(destinationPath, bufferSize)
}

/**
 * 将 Bitmap 对象保存到系统相册。
 * 请确保在调用此函数之前 [application] 已经被初始化，因为它依赖于应用程序的上下文来发送广播通知系统图片已更新。
 * @param imageType 压缩格式，如 PNG 或 JPEG。
 * @param compressQuality 压缩质量，从0（最差）到100（无损）。
 * @param album 相册名称，用于在公共图片目录下创建子目录。
 * @param imageFileName 要保存的图片文件名，如果为 null，则使用当前日期和时间。
 * @return 保存的图片文件的绝对路径，如果保存失败返回 null。
 */
fun Bitmap.saveAs(
    imageType: Bitmap.CompressFormat = Bitmap.CompressFormat.PNG,
    compressQuality: Int = 100,
    album: String = "Default",
    imageFileName: String? = null
): String? {
    // 检查外部存储是否可用
    if (Environment.getExternalStorageState() != Environment.MEDIA_MOUNTED) return null
    // 获取或创建相册目录
    val albumFile = File(
        Environment.getExternalStoragePublicDirectory(Environment.DIRECTORY_PICTURES), album
    )
    // 如果无法创建目录，返回 null
    if (!albumFile.exists() && !albumFile.mkdirs()) return null
    // 生成文件名，如果未提供自定义文件名，则使用当前日期和时间
    val fileName =
        imageFileName ?: Date().format(DateFormatPattern.FULL_DATE_TIME_DASH) ?: return null
    val imageFile = File(albumFile, fileName)
    try {
        // 使用 FileOutputStream 将图片写入文件
        val outputStream = FileOutputStream(imageFile)
        compress(imageType, compressQuality, outputStream)
        outputStream.flush() // 确保所有数据都写入到文件
        outputStream.close() // 关闭流
        // 媒体扫描器，通知系统图片已更新
        val mediaScannerIntent = Intent(Intent.ACTION_MEDIA_SCANNER_SCAN_FILE)
        val contentUri = Uri.fromFile(imageFile)
        mediaScannerIntent.data = contentUri
        application.sendBroadcast(mediaScannerIntent)
        return imageFile.absolutePath // 返回图片的完整路径
    } catch (e: Exception) {
        e.printStackTrace()
        return null
    }

}


/**
 * 返回文件大小的字符串表示形式。
 * @receiver File 文件对象。
 * @return 文件大小的字符串表示，单位为KB或MB。
 */
val File.fileSize: String
    @SuppressLint("DefaultLocale") get() {
        val fileSizeInBytes = length()
        val fileSizeInKB = fileSizeInBytes / 1024.0
        val fileSizeInMB = fileSizeInKB / 1024.0

        return when {
            fileSizeInMB >= 1 -> String.format("%.1f MB", fileSizeInMB)
            else              -> String.format("%.1f KB", fileSizeInKB)
        }
    }


/**
 * 计算一组文件大小字符串的总和，并返回格式化的总大小字符串。
 * @param fileSizes 包含文件大小的字符串列表，格式如"123 KB"或"456.78 MB"。
 * @return 格式化后的总大小字符串，单位可能是B, KB, MB, GB或TB。
 */
fun fileSizeSummation(fileSizes: List<String?>): String {
    val fileSizePattern = "(\\d+(\\.\\d+)?)\\s*(K|M|G)B?".toRegex()
    var totalSizeInBytes = 0.0
    fileSizes.forEach { fileSizeStr ->
        fileSizeStr?.let { str ->
            val matchResult = fileSizePattern.find(str) ?: return@let
            if (matchResult.groups.size != 4) return@let
            val sizeStr = matchResult.groupValues[1]
            val size = sizeStr.toDouble()
            val sizeInBytes: Double = when (matchResult.groupValues[3]) {
                "K"  -> size * 1024
                "M"  -> size * 1024 * 1024
                "G"  -> size * 1024 * 1024 * 1024
                else -> size
            }
            totalSizeInBytes += sizeInBytes
        }
    }
    val units = arrayOf("B", "KB", "MB", "GB", "TB")
    val decimalFormat = DecimalFormat("#,##0.#")
    val digitGroups = (log10(totalSizeInBytes) / log10(1024.0)).toInt()
    return decimalFormat.format(totalSizeInBytes / 1024.0.pow(digitGroups)) + " " + units[digitGroups]
}
