package com.liecoder.framework.ktx

import android.content.res.ColorStateList
import android.content.res.Resources
import android.graphics.Bitmap
import android.graphics.BlendMode
import android.graphics.BlendModeColorFilter
import android.graphics.Canvas
import android.graphics.Color
import android.graphics.PorterDuff
import android.graphics.drawable.Drawable
import android.os.Build
import android.view.View
import androidx.annotation.ColorRes
import androidx.annotation.DrawableRes
import androidx.annotation.StringRes
import androidx.core.content.ContextCompat
import androidx.core.content.res.ResourcesCompat
import androidx.recyclerview.widget.RecyclerView
import com.liecoder.framework.global.AppHelper.application
import com.liecoder.framework.manager.DeviceInfoManager
import kotlin.math.ceil

/**
 * 从系统资源中获取字符串。
 * 通常用于获取Android系统定义的标准字符串。
 *
 * @param stringRes 字符串资源的ID。
 * @return 返回与给定资源ID对应的字符串。
 * @see android.content.res.Resources.getSystem()
 */
fun getSystemString(@StringRes stringRes: Int): String = Resources.getSystem().getString(stringRes)


/**
 * 从系统资源中获取格式化后的字符串。
 * 通常用于获取Android系统定义的标准字符串，并进行参数格式化。
 *
 * @param stringRes 字符串资源的ID。
 * @param format 用于格式化字符串的参数列表。
 * @return 返回格式化后的字符串。
 * @see android.content.res.Resources.getSystem()
 */
fun getSystemString(@StringRes stringRes: Int, vararg format: Any): String =
    Resources.getSystem().getString(stringRes, format)


/**
 * 从应用程序资源中获取字符串。
 * 用于获取当前应用程序定义的字符串资源。
 *
 * @param stringRes 字符串资源的ID。
 * @return 返回与给定资源ID对应的字符串。
 * @see android.content.Context.resources
 */
fun getResourcesString(@StringRes stringRes: Int): String =
    application.resources.getString(stringRes)


/**
 * 从应用程序资源中获取格式化后的字符串。
 * 用于获取当前应用程序定义的字符串资源，并进行参数格式化。
 *
 * @param stringRes 字符串资源的ID。
 * @param format 用于格式化字符串的参数列表。
 * @return 返回格式化后的字符串。
 * @see android.content.Context.resources
 */
fun getResourcesString(@StringRes stringRes: Int, vararg format: Any): String =
    application.resources.getString(stringRes, *format)


/**
 * 从应用程序资源中获取 Drawable 对象，考虑主题兼容性。
 * 此函数使用全局的 application resources 和 theme 来获取 Drawable，以确保与主题的兼容性。
 *
 * @param drawableRes Drawable 资源的 ID。
 * @return 返回与给定资源 ID 对应的 Drawable 对象，如果没有找到，则返回 null。
 * @see ResourcesCompat.getDrawable()
 */
fun getResourcesDrawable(@DrawableRes drawableRes: Int): Drawable? =
    ResourcesCompat.getDrawable(application.resources, drawableRes, application.theme)


/**
 * 从应用程序上下文中获取 Drawable 对象。
 * 此函数使用全局的 application context 来获取 Drawable，适用于不需要考虑主题兼容性的情况。
 *
 * @param drawableRes Drawable 资源的 ID。
 * @return 返回与给定资源 ID 对应的 Drawable 对象，如果没有找到，则返回 null。
 * @see ContextCompat.getDrawable()
 */
fun getContextDrawable(@DrawableRes drawableRes: Int): Drawable? =
    ContextCompat.getDrawable(application, drawableRes)


/**
 * 根据资源ID从应用程序资源中获取颜色值，考虑主题属性。
 * 此函数使用全局的application资源和主题来获取颜色，以确保与主题的兼容性。
 *
 * @param colorRes 颜色资源的ID。
 * @return 返回与给定资源ID对应的颜色值。
 * @see ResourcesCompat.getColor()
 */
fun getResourcesColor(@ColorRes colorRes: Int): Int =
    ResourcesCompat.getColor(application.resources, colorRes, application.theme)


/**
 * 从应用程序上下文中获取颜色值。
 * 此函数使用全局的application上下文来获取颜色，适用于不需要考虑主题兼容性的情况。
 *
 * @param colorRes 颜色资源的ID。
 * @return 返回与给定资源ID对应的颜色值。
 * @see ContextCompat.getColor()
 */
fun getContextColor(@ColorRes colorRes: Int): Int = ContextCompat.getColor(application, colorRes)


/**
 * 根据资源ID从应用程序资源中获取颜色状态列表，考虑主题属性。
 * 此函数使用全局的application资源和主题来获取颜色状态列表，以确保与主题的兼容性。
 *
 * @param colorRes 颜色状态列表资源的ID。
 * @return 返回与给定资源ID对应的颜色状态列表，如果没有找到，则返回null。
 * @see ResourcesCompat.getColorStateList()
 */
fun getResourcesColorStateList(@ColorRes colorRes: Int): ColorStateList? =
    ResourcesCompat.getColorStateList(application.resources, colorRes, application.theme)


/**
 * 从应用程序上下文中获取颜色状态列表。
 * 此函数使用全局的application上下文来获取颜色状态列表，适用于不需要考虑主题兼容性的情况。
 *
 * @param colorRes 颜色状态列表资源的ID。
 * @return 返回与给定资源ID对应的颜色状态列表，如果没有找到，则返回null。
 * @see ContextCompat.getColorStateList()
 */
fun getContextColorStateList(@ColorRes colorRes: Int): ColorStateList? =
    ContextCompat.getColorStateList(application, colorRes)


/**
 * 根据条件为 Drawable 应用或移除颜色滤镜（tint）。
 *
 * 此函数允许开发者根据 [shouldApplyTint] 参数决定是否为 Drawable 应用颜色滤镜。如果 [shouldApplyTint] 为 true，
 * 则使用指定的 [tint] 颜色值应用颜色滤镜；如果为 false，则移除任何已应用的颜色滤镜。
 * 颜色滤镜的应用方式根据 Android 系统版本决定，Android 10（API 级别 29，Q）及以上版本使用 [BlendModeColorFilter]，
 * 低版本使用 [setColorFilter] 方法结合 [PorterDuff.Mode.SRC_IN] 模式。
 *
 * @param shouldApplyTint 是否应用颜色滤镜的布尔值，默认为 true。
 * @param tint 要应用的颜色值，默认为透明（[Color.TRANSPARENT]）。
 * @return 返回 Drawable 实例自身，允许链式调用。
 */
fun Drawable.applyTint(shouldApplyTint: Boolean = true, tint: Int = Color.TRANSPARENT): Drawable {
    // 创建 Drawable 的可修改副本
    if (tint == Color.TRANSPARENT) return this
    mutate()
    if (shouldApplyTint) {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.Q) {
            colorFilter = BlendModeColorFilter(tint, BlendMode.SRC_IN)
        } else {
            setColorFilter(tint, PorterDuff.Mode.SRC_IN)
        }
    } else {
        clearColorFilter()
    }
    return this
}

/**
 * 为 Drawable 设置尺寸，并保持原始宽高比。
 *
 * 此函数允许开发者为 Drawable 设置新的宽度和高度。如果只指定了一个维度（宽度或高度），函数会根据 Drawable 的原始宽高比计算缺失的维度。
 * 如果没有指定有效的宽度或高度（即两者都小于或等于0），则重置为 Drawable 的固有尺寸。
 *
 * @param width 要设置的宽度，或 null 如果不更改宽度。
 * @param height 要设置的高度，或 null 如果不更改高度。
 * @return 返回 Drawable 实例自身，允许链式调用。
 */
fun Drawable.applySize(width: Int?, height: Int?): Drawable {

    val drawableWidth = width?.takeIf { it > 0 } ?: 0
    val drawableHeight = height?.takeIf { it > 0 } ?: 0

    if (drawableWidth <= 0 && drawableHeight <= 0) {
        setBounds(0, 0, intrinsicWidth, intrinsicHeight)
        return this
    }
    if (drawableWidth > 0 && drawableHeight > 0) {
        setBounds(0, 0, drawableWidth, drawableHeight)
        return this
    }
    val drawableWidthIntrinsic = if (intrinsicWidth > 0) intrinsicWidth else drawableWidth
    val drawableHeightIntrinsic = if (intrinsicHeight > 0) intrinsicHeight else drawableHeight

    if (drawableWidth > 0) {
        val ratio = drawableHeightIntrinsic.toFloat() / drawableWidthIntrinsic
        setBounds(0, 0, drawableWidth, (drawableWidth * ratio).toInt())
    } else {
        val ratio = drawableWidthIntrinsic.toFloat() / drawableHeightIntrinsic
        setBounds(0, 0, (drawableHeight * ratio).toInt(), drawableHeight)
    }

    return this
}

fun Bitmap.ensureBitmapCanDraw(): Bitmap {
    val width = this.width
    val height = this.height
    val totalPixel = width * height
    val limitedPixel =
        DeviceInfoManager.getScreenWidthPx() * DeviceInfoManager.getScreenHeightPx() * 4
    if (totalPixel < limitedPixel) return this
    val inSampleSize = ceil((totalPixel / limitedPixel).toDouble()).toInt()
    return Bitmap.createScaledBitmap(this, width / inSampleSize, height / inSampleSize, true)
}

fun View.screenshot(): Bitmap {
    if (measuredWidth == 0 || measuredHeight == 0) {
        throw RuntimeException("调用该方法时，请确保View已经测量完毕，如果宽高为0，则抛出异常以提醒！")
    }
    return when (this) {
        is RecyclerView -> {
            this.scrollToPosition(0)
            this.measure(
                View.MeasureSpec.makeMeasureSpec(width, View.MeasureSpec.EXACTLY),
                View.MeasureSpec.makeMeasureSpec(0, View.MeasureSpec.UNSPECIFIED)
            )

            val bmp = Bitmap.createBitmap(width, measuredHeight, Bitmap.Config.ARGB_8888)
            val canvas = Canvas(bmp)

            //draw default bg, otherwise will be black
            if (background != null) {
                background.setBounds(0, 0, width, measuredHeight)
                background.draw(canvas)
            } else {
                canvas.drawColor(Color.WHITE)
            }
            this.draw(canvas)
            //恢复高度
            this.measure(
                View.MeasureSpec.makeMeasureSpec(width, View.MeasureSpec.EXACTLY),
                View.MeasureSpec.makeMeasureSpec(height, View.MeasureSpec.AT_MOST)
            )
            bmp //return
        }
        else            -> {
            val screenshot =
                Bitmap.createBitmap(measuredWidth, measuredHeight, Bitmap.Config.ARGB_4444)
            val canvas = Canvas(screenshot)
            if (background != null) {
                background.setBounds(0, 0, width, measuredHeight)
                background.draw(canvas)
            } else {
                canvas.drawColor(Color.WHITE)
            }
            draw(canvas)// 将 view 画到画布上
            screenshot //return
        }
    }
}