package com.monmonkeygroup.openapi.trade;

import com.monmonkeygroup.openapi.Config;
import com.monmonkeygroup.openapi.OpenApiException;
import com.monmonkeygroup.openapi.client.Client;
import com.monmonkeygroup.openapi.client.Profile;
import com.monmonkeygroup.openapi.protocol.ActionName;
import com.monmonkeygroup.openapi.protocol.ActionType;
import com.monmonkeygroup.openapi.protocol.Packet;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.*;
import java.util.concurrent.ConcurrentHashMap;

public class TradeContext implements AutoCloseable {
    private static final Logger log = LoggerFactory.getLogger(TradeContext.class);
    private final Config config;
    private final Client client;
    private final Set<String> subscriptions = ConcurrentHashMap.newKeySet();

    public TradeContext(Config config) {
        if (null == config.getTradeAccessToken() || config.getTradeAccessToken().isEmpty()) {
            throw new RuntimeException("don't has tradeAccessToken. please set trade access token");
        }

        Profile profile = new Profile(config.getTradeUrl(), config.getUser(), config.getTradeAccessToken());
        this.config = config;
        this.client = new Client(profile);
        this.client.setAfterConnected(this::afterReconnect);
        try {
            this.client.init();
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    private void afterReconnect(Client client) {
        try {
            resubscribe();
        } catch (Exception e) {
            log.error("trade resubscribe error: {}", e.toString());
        }
    }

    private void resubscribe() throws OpenApiException {
        for (String actionName : this.subscriptions) {
            doSubscribe(actionName, true);
        }
    }

    private void doSubscribe(String actionName, boolean isSub) throws OpenApiException {
        Packet packet = Packet.CreateSubscribe(actionName, isSub);
        packet.setActionType(ActionType.SUBSCRIBE);

        this.client.sendPacket(packet);
    }

    /**
     * Subscribe
     *
     * @param actionNameList List of {@link SubActionName}
     * @throws OpenApiException If an error occurs
     */
    public void subscribe(List<String> actionNameList) throws OpenApiException {
        for (String actionName : actionNameList) {
            doSubscribe(actionName, true);
            this.subscriptions.add(actionName);
        }
    }

    /**
     * Unubscribe
     *
     * @param actionNameList List of {@link SubActionName}
     * @throws OpenApiException If an error occurs
     */
    public void unsubscribe(List<String> actionNameList) throws OpenApiException {
        for (String actionName : actionNameList) {
            doSubscribe(actionName, false);
            this.subscriptions.remove(actionName);
        }
    }

    /**
     * Set order callback
     *
     * @param handler A order handler
     */
    public void setOnOrder(OrderHandler handler) {
        if (null == handler) return;
        this.client.addPushHandler(ActionName.TRD_ORDER_PUSH, packet -> {
            try {
                Order update = packet.deserialize(Order.class);
                handler.onOrder(update);
            } catch (Exception e) {
                log.error("order update,unmarshal error: {}", e.toString());
            }
        });
    }

    /**
     * Get a list of trading accounts
     *
     * @return list of {@link TradeAccount}
     * @throws OpenApiException If an error occurs
     */
    public List<TradeAccount> listTradeAccounts() throws OpenApiException {
        ListTradeAccountsRequest request = new ListTradeAccountsRequest();
        ListTradeAccountsReply reply = this.client.doRequest(
                ActionName.TRD_LIST_TRADE_ACCOUNTS_REQUEST, request, ListTradeAccountsReply.class);
        return reply.getTradeAccounts();
    }

    /**
     * Get a summary of the trading account,
     *
     * @param tradeAccountId The unique identifier of a trading account
     * @return {@link AccountSummaryReply}
     * @throws OpenApiException If an error occurs
     */
    public AccountSummaryReply getAccountSummary(String tradeAccountId) throws OpenApiException {
        AccountSummaryRequest request = new AccountSummaryRequest(tradeAccountId);
        return client.doRequest(
                ActionName.TRD_ACCOUNT_SUMMARY_REQUEST, request, AccountSummaryReply.class);
    }

    /**
     * Query the position list of the trading account
     *
     * @param tradeAccountId The unique identifier of a trading account
     * @return list of {@link Position}
     * @throws OpenApiException If an error occurs
     */
    public List<Position> getPosition(String tradeAccountId) throws OpenApiException {
        PositionRequest request = new PositionRequest(tradeAccountId);
        PositionReply reply = client.doRequest(ActionName.TRD_POSITION_REQUEST, request, PositionReply.class);
        return reply.getApiPositions();
    }

    /**
     * Create a new order
     *
     * @param request parameters for this request,not null
     * @return {@link Order}
     * @throws OpenApiException If an error occurs
     */
    public Order createOrder(CreateOrderRequest request) throws OpenApiException {
        CreateOrderReply reply = client.doRequest(
                ActionName.TRD_CREATE_ORDER_REQUEST, request, CreateOrderReply.class);
        return reply.getOrder();
    }

    /**
     * Amends an existing open order in the trading system.
     *
     * @param request parameters for this request,not null
     * @throws OpenApiException If an error occurs
     */
    public void amendOrder(AmendOrderRequest request) throws OpenApiException {
        client.doRequest(
                ActionName.TRD_AMEND_ORDER_REQUEST, request, AmendOrderReply.class);
    }

    /**
     * Cancel a existing open orders in the trading system.
     *
     * @param request parameters for this request,not null
     * @throws OpenApiException If an error occurs
     */
    public void cancelOrder(CancelOrderRequest request) throws OpenApiException {
        client.doRequest(
                ActionName.TRD_CANCEL_ORDER_REQUEST, request, CancelOrderRequest.class);
    }

    /**
     * Get today orders
     *
     * @param request parameters for this request,not null
     * @return List of {@link Order}
     * @throws OpenApiException If an error occurs
     */
    public List<Order> getTodayOrders(TodayOrderRequest request) throws OpenApiException {
        TodayOrderReply reply = client.doRequest(
                ActionName.TRD_TODAY_ORDERS_REQUEST, request, TodayOrderReply.class);
        return reply.getOrders();
    }

    /**
     * Obtain the maximum allowed purchase quantity
     *
     * @param request parameters for this request,not null
     * @return {@link MaxBuyQuantityReply}
     * @throws OpenApiException If an error occurs
     */
    public MaxBuyQuantityReply getMaxBuyQuantity(MaxBuyQuantityRequest request) throws OpenApiException {
        return client.doRequest(
                ActionName.TRD_MAX_BUY_QUANTITY_REQUEST, request, MaxBuyQuantityReply.class);
    }

    public Config getConfig() {
        return config;
    }


    @Override
    public void close() {
        this.client.close();
    }
}
